/*
 * radwho	Show who is logged in on the terminal servers.
 *		Can also be installed as fingerd on the UNIX
 *		machine RADIUS runs on.
 *
 * Version:	@(#)radwho  1.20  25-Jan-1997  miquels@cistron.nl
 *
 */
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <pwd.h>
#include <errno.h>
#include <sys/stat.h>
#include <sys/utsname.h>
#include <time.h>
#include <netinet/in.h>
#include "sysdep.h"

#include "radutmp.h"
#include "radiusd.h"

/*
 *	FIXME: put in header file.
 */
#define SYS_FINGER "/usr/bin/finger"
#define FINGER_DIR "/usr/local/lib/finger"

/*
 *	Header above output and format.
 */
char *hdr = 
"Login    Name              What  TTY  When      From      Location";
char *ufmt = "%-8.8s %-17.17s %-5.5s %-4.4s %-9.9s %-9.9s %-16.16s%s";
char *rfmt = "%-8.8s %-17.17s %-5.5s S%-3d %-9.9s %-9.9s %-19.19s%s";

/*
 *	End of line indicator; can be LF or CRLF.
 */
char *eol = "\n";

CLIENT *clients;


/*
 *	Read the clients file.
 */
static CLIENT *my_read_clients_file(char *file)
{
	FILE	*fp;
	char	buffer[256];
	char	hostnm[128];
	char	secret[32];
	char	shortnm[32];
	int	lineno = 0;
	CLIENT	*cl = NULL;
	CLIENT	*c;

	if ((fp = fopen(file, "r")) == NULL) {
		fprintf(stderr, "cannot open %s\n", file);
		return NULL;
	}
	while(fgets(buffer, 256, fp) != NULL) {
		lineno++;
		if (buffer[0] == '#' || buffer[0] == '\n')
			continue;
		shortnm[0] = 0;
		if (sscanf(buffer, "%s%s%s", hostnm, secret, shortnm) < 2) {
			fprintf(stderr, "%s[%d]: syntax error\n", file, lineno);
			continue;
		}
		if ((c = malloc(sizeof(CLIENT))) == NULL) {
			fprintf(stderr, "%s[%d]: out of memory\n",
				file, lineno);
			return NULL;
		}

		c->ipaddr = get_ipaddr(hostnm);
		strcpy(c->secret, secret);
		strcpy(c->shortname, shortnm);
		strcpy(c->longname, ip_hostname(c->ipaddr));

		c->next = cl;
		cl = c;
	}
	fclose(fp);

	return cl;
}


/*
 *	Print a file from FINGER_DIR. If the file is executable,
 *	execute it instead. Return 0 if succesfull.
 */
int ffile(char *arg)
{
	FILE *fp;
	char fn[1024];
	int p = 0;
	char *s;

	sprintf(fn, "%s/%.32s", FINGER_DIR, arg);
	if (access(fn, X_OK) == 0) {
		p = 1;
		sprintf(fn, "exec %s/%.32s 2>&1", FINGER_DIR, arg);
		fp = popen(fn, "r");
	} else fp = fopen(fn, "r");

	if (fp == NULL) return -1;

	while(fgets(fn, 1024, fp)) {
		if ((s = strchr(fn, '\n')) != NULL)
			*s = 0;
		fprintf(stdout, "%s\r\n", fn);
	}
	if (p)
		pclose(fp);
	else
		fclose(fp);
	fflush(stdout);
	return 0;
}


/*
 *	Execute the system finger and translate LF to CRLF.
 */
void sys_finger(char *l)
{
	FILE *fp;
	char fn[1024];
	char *p;

	if (ffile(l) == 0)
		exit(0);

	sprintf(fn, "exec %s %s", SYS_FINGER, l);
	if ((fp = popen(fn, "r")) == NULL) {
		printf("popen: %s\r\n", strerror(errno));
		exit(1);
	}

	while(fgets(fn, 1024, fp)) {
		if ((p = strchr(fn, '\n')) != NULL)
			*p = 0;
		fprintf(stdout, "%s\r\n", fn);
	}
	pclose(fp);
	exit(0);
}


/*
 *	Get fullname of a user.
 */
char *fullname(char *login)
{
	struct passwd *pwd;
	char *s;

	if ((pwd = getpwnam(login)) != NULL) {
		if ((s = strchr(pwd->pw_gecos, ',')) != NULL) *s = 0;
		return pwd->pw_gecos;
	}
	return login;
}

/*
 *	Return protocol type.
 */
char *proto(int id)
{
	if (id == 'S') return "SLIP";
	if (id == 'P') return "PPP";

	return "shell";
}

/*
 *	Return a time in the form day hh:mm
 */
char *dotime(time_t t)
{
	char *s = ctime(&t);

	strncpy(s + 4, s + 11, 5);
	s[9] = 0;

	return s;
}

#if 0 /*UNUSED*/
/*
 *	See how long a tty has been idle.
 */
char *idletime(char *line)
{
	char tty[16];
	static char tmp[8];
	time_t t;
	struct stat st;
	int hr, min, days;

	if (line[0] == '/')
		strcpy(tty, "/dev/");
	else
		tty[0] = 0;
	strcat(tty, line);

	tmp[0] = 0;
	if (stat(tty, &st) == 0) {
		time(&t);
		t -= st.st_mtime;
		if (t >= 60) {
			min = (t / 60);
			hr = min / 24;
			days = hr / 24;
			min %= 60;
			hr %= 24;
			if (days > 0)
				sprintf(tmp, "%dd", days);
			else
				sprintf(tmp, "%2d:%02d", hr, min);
		}
	}
	return tmp;
}
#endif

/*
 *	Shorten tty name.
 */
char *ttyshort(char *tty)
{
	static char tmp[16];

	if (tty[0] == '/') tty += 5;

	if (strncmp(tty, "tty", 3) == 0) {
		if (tty[3] >= '0' && tty[3] <= '9')
			sprintf(tmp, "v%s", tty + 3);
		else
			sprintf(tmp, "%s", tty + 3);
		return tmp;
	}
	if (strncmp(tty, "vc", 2) == 0) {
		sprintf(tmp, "v%s", tty + 2);
		return tmp;
	}
	if (strncmp(tty, "cu", 2) == 0) {
		return tmp + 2;
	}
	return "??";
}


/*
 *	Find name of NAS
 */
char *nasname(UINT4 ipaddr)
{
	CLIENT *cl;

	for(cl = clients; cl; cl = cl->next)
		if (cl->ipaddr == ipaddr)
			break;
	if (cl == NULL)
		return "";
	if (cl->shortname[0])
		return cl->shortname;
	return cl->longname;
}


/*
 *	Print address of client.
 */
char *hostname(UINT4 ipaddr)
{
	if (ipaddr == 0 || ipaddr == (UINT4)-1 || ipaddr == (UINT4)-2)
		return "";
	return ip_hostname(ntohl(ipaddr));
}


/*
 *	Print usage message and exit.
 */
void usage(void)
{
	fprintf(stderr, "Usage: radwho [-l] [-h] [-f]\n");
	fprintf(stderr, "       -l: show local (shell) users too\n");
	fprintf(stderr, "       -h: hide shell users from radius\n");
	fprintf(stderr, "       -f: give fingerd output\n");
	exit(1);
}


/*
 *	Main program, either pmwho or fingerd.
 */
int main(int argc, char **argv)
{
	FILE *fp;
	struct radutmp rt;
	struct utmp ut;
	int hdrdone = 0;
	char inbuf[128];
	char myname[128];
	int fingerd = 0;
	int showlocal = 0;
	int hideshell = 0;
	char *p, *q;
	int c;

	while((c = getopt(argc, argv, "flh")) != EOF) switch(c) {
		case 'f':
			fingerd++;
			break;
		case 'l':
			showlocal = 1;
			break;
		case 'h':
			hideshell = 1;
			break;
		default:
			usage();
			break;
	}

	/*
	 *	Read the "clients" file.
	 */
	sprintf(inbuf, "%s/%s", RADIUS_DIR, RADIUS_CLIENTS);
	if ((clients = my_read_clients_file(inbuf)) == NULL)
		exit(1);

	/*
	 *	See if we are "fingerd".
	 */
	if (strstr(argv[0], "fingerd")) {
		fingerd++;
		eol = "\r\n";
	}

	if (showlocal && (fp = fopen(UTMP_FILE, "r"))) {
		/*
		 *	Read first line of the input.
		 */
		fgets(inbuf, 128, stdin);
		p = inbuf;
		while(*p == ' ' || *p == '\t') p++;
		if (*p == '/' && *(p + 1)) p += 2;
		while(*p == ' ' || *p == '\t') p++;
		for(q = p; *q && *q != '\r' && *q != '\n'; q++)
			;
		*q = 0;

		/*
		 *	See if we fingered a specific user.
		 */
		ffile("header");
		if (*p) sys_finger(p);

		fputs(hdr, stdout);
		fputs(eol, stdout);
		hdrdone = 1;

		/*
		 *	Show the logged in UNIX users.
		 */
		gethostname(myname, 128);
		while(fread(&ut, sizeof(ut), 1, fp) == 1) {
#ifdef USER_PROCESS
			if (ut.ut_user[0] && ut.ut_line[0] &&
				ut.ut_type == USER_PROCESS) {
#else
			if (ut.ut_user[0] && ut.ut_line[0]) {
#endif
				printf(ufmt,
					ut.ut_name,
					fullname(ut.ut_name),
					"shell",
					ttyshort(ut.ut_line),
#ifdef __svr4__
					dotime(ut.ut_xtime),
#else
					dotime(ut.ut_time),
#endif
					ut.ut_host,
					myname, eol);
			}
		}
		fclose(fp);
	}

	/*
	 *	Show the users logged in on the terminal server(s).
	 */
	if ((fp = fopen(RADUTMP, "r")) == NULL)
		return 0;

	if (!hdrdone) {
		fputs(hdr, stdout);
		fputs(eol, stdout);
	}

	while(fread(&rt, sizeof(rt), 1, fp) == 1) {
		if (rt.type == P_LOGIN) {
			/*
			 *	We don't show shell users if we are
			 *	fingerd, as we have done that above.
			 */
			if (hideshell && !strchr("PCS", rt.proto))
				continue;

			printf(rfmt,
				rt.login,
				fullname(rt.login),
				proto(rt.proto),
				rt.nas_port,
				dotime(rt.time),
				nasname(ntohl(rt.nas_address)),
				hostname(rt.framed_address), eol);
		}
	}
	fclose(fp);

	return 0;
}

