/*
 * proxy.c	Functions for the proxy server functionality.
 *
 *		What we do here is stateless; we put the request-authenticator
 *		from the original client in the Proxy-State attribute. Then
 *		we re-use the request-authenticator as our own when we
 *		send the request to the radius server. When it comes back
 *		we can get the request-authenticator from the reply packet,
 *		so that we can use it to send a reply to the client.
 *
 *		I'm not sure that's really safe. It works though.
 *
 * Version:	@(#)proxy.c  1.00  24-Nov-1997  miquels@cistron.nl
 *
 */

#include	<sys/types.h>
#include	<sys/socket.h>
#include	<sys/time.h>
#include	<netinet/in.h>

#include	<stdio.h>
#include	<netdb.h>
#include	<time.h>
#include	<ctype.h>

#include	"radiusd.h"

static int invalid(AUTH_REQ *authreq, char *s)
{
	log(L_ERR, "proxy-reply from %s: invalid Proxy-State packet", s);
	authfree(authreq);
	return -1;
}

/*
 *	We got a packet from a remote radius server, and need to send
 *	it back to the original client (-NAS).
 */
int rad_relay(AUTH_REQ *authreq, int activefd)
{
	CLIENT		*cl;
	VALUE_PAIR 	*vp, *lp;
	UINT4		ip;
	char		*p;
	char		*name;
	char		req_auth[AUTH_VECTOR_LEN];
	char		digest[AUTH_VECTOR_LEN];
	char		buffer[128];
	int		i, n;
	int		secretlen;
	int		port;
	int		id;
	unsigned int	nip;
	int		np;

	/*
	 *	First see if we know this client, and verify the
	 *	md5 digest.
	 */
	if ((cl = client_find(authreq->ipaddr)) == NULL) {
		log(L_ERR, "proxy-reply from %s: unknown server",
			client_name(authreq->ipaddr));
		authfree(authreq);
		return -1;
	}
	name = client_name(cl->ipaddr);

	/*
	 *	Find the last Proxy-State packet, that one
	 *	must be ours. We also check that it's a string type,
	 *	and that it starts with `cistron:'
	 */
	lp = NULL;
	for (vp = authreq->request; vp; vp = vp->next) {
		if (vp->attribute == PW_PROXY_STATE)
			lp = vp;
	}
	if (lp == NULL) {
		log(L_ERR, "proxy-reply from %s: no Proxy-State");
		authfree(authreq);
		return -1;
	}

	/*
	 *	check validity and decode.
	 *	The proxy-state string contains:
	 *
	 *	ipaddr:port   :orig_id:reqauth
	 *	4bytes 2bytes  2bytes  16bytes
	 *	bytes are in hex (2 chars each).
	 *	port/ipaddr are in network byte order
	 */
	if (lp->type != PW_TYPE_STRING)
		return invalid(authreq, name);
	if (strncmp(lp->strvalue, "cistron:", 8) != 0)
		return invalid(authreq, name);
	if (strlen(lp->strvalue) < 18)
		return invalid(authreq, name);

	if (sscanf(lp->strvalue + 8, "%x:%x:%x:%s",
		&nip, &np, &id, buffer) != 4)
		return invalid(authreq, name);

	/*
	 *	Decode IP, port and request authenticator (vector).
	 */
	ip   = ntohl(nip);
	port = ntohs(np);
	p = buffer;
	for(i = 0; i < AUTH_VECTOR_LEN; i++) {
		sscanf(p, "%2x", &n);
		req_auth[i] = n;
	}

	/*
	 *	Fill in the original request authenticator.
	 *	Then append the shared secret to the received packet,
	 *	and calculate the MD5 sum. This must be the same
	 *	as the MD5 sum (authreq->vector).
	 */
	secretlen = strlen(cl->secret);
	memcpy(authreq->data + 4, req_auth, AUTH_VECTOR_LEN);
	memcpy(authreq->data + authreq->data_len, cl->secret, secretlen);
	md5_calc(digest, authreq->data, authreq->data_len + secretlen);

	if (memcmp(digest, authreq->vector, AUTH_VECTOR_LEN) != 0) {
		log(L_ERR, "proxy-reply from %s: wrong secret",
			client_name(authreq->ipaddr));
		authfree(authreq);
		return -1;
	}

	/*
	 *	Where do we need to send the reply?
	 */
	if ((cl = client_find(ip)) == NULL) {
		log(L_ERR, "proxy-reply from %s: no original client");
		authfree(authreq);
		return -1;
	}

	/*
	 *	Fill in the rest of the new AUTH_REQ and
	 *	remove the Proxy-State A/V pair.
	 */
	authreq->ipaddr = ip;
	authreq->udp_port = port;
	authreq->id = id;
	memcpy(authreq->vector, req_auth, AUTH_VECTOR_LEN);
	strcpy(authreq->secret, cl->secret);
	lp->type = -1;

	/*
	 *	Now send the reply.
	 */
	rad_send_reply(authreq->code, authreq, NULL, NULL, activefd);

	authfree(authreq);

	return 0;
}

