/*
 * cb.c	 callbacks for EAP-TLS.
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Copyright 2002  hereuare Communications <raghud@hereuare.com>
 */
#include "mytls.h"

#ifndef NO_OPENSSL

/* Output level:
 *     0 = nothing,
 *     1 = minimal, just errors,
 *     2 = minimal, all steps,
 *     3 = detail, all steps */
static unsigned int cb_ssl_verify_level = 1;
//static int int_verify_depth = 10;

/*
 * Activate the channels/connections only after Handshake is done.
 */
void cb_ssl_info(SSL *s, int where, int ret)
{
	char *str1, *str2;
	int w;

	if (where & SSL_CB_HANDSHAKE_START)
		fprintf(stdout, "Callback has been called because a new handshake is started.\n");
	if (where & SSL_CB_HANDSHAKE_DONE)
		fprintf(stdout, "Callback has been called because handshake is finished.\n");

	w = where & ~SSL_ST_MASK;
	str1 = (w & SSL_ST_CONNECT ? "SSL_connect" : (w & SSL_ST_ACCEPT ?
				"SSL_accept" : "undefined")),
	str2 = SSL_state_string_long(s);

	if (where & SSL_CB_LOOP)
		fprintf(stdout, "(%s) %s\n", str1, str2);
	else if (where & SSL_CB_EXIT) {
		if (ret == 0)
			fprintf(stdout, "(%s) failed in %s\n", str1, str2);
		else if (ret < 0)
			fprintf(stdout, "%s:error in %s\n", str1, str2);
	}
}

static const char *int_reason_no_issuer = "X509_V_ERR_UNABLE_TO_GET_ISSUER_CERT";
static const char *int_reason_not_yet = "X509_V_ERR_CERT_NOT_YET_VALID";
static const char *int_reason_before = "X509_V_ERR_ERROR_IN_CERT_NOT_BEFORE_FIELD";
static const char *int_reason_expired = "X509_V_ERR_CERT_HAS_EXPIRED";
static const char *int_reason_after = "X509_V_ERR_ERROR_IN_CERT_NOT_AFTER_FIELD";

int cb_ssl_verify(int ok, X509_STORE_CTX *ctx)
{
	char buf1[256]; /* Used for the subject name */
	char buf2[256]; /* Used for the issuer name */
	const char *reason = NULL; /* Error reason (if any) */
	X509 *err_cert;
	int err, depth;

	if(cb_ssl_verify_level == 0)
		return ok;
	err_cert = X509_STORE_CTX_get_current_cert(ctx);
	err = X509_STORE_CTX_get_error(ctx);
	depth = X509_STORE_CTX_get_error_depth(ctx);

	buf1[0] = buf2[0] = '\0';
	/* Fill buf1 */
	X509_NAME_oneline(X509_get_subject_name(err_cert), buf1, 256);
	/* Fill buf2 */
	X509_NAME_oneline(X509_get_issuer_name(ctx->current_cert), buf2, 256);
	switch (ctx->error) {
	case X509_V_ERR_UNABLE_TO_GET_ISSUER_CERT:
		reason = int_reason_no_issuer;
		break;
	case X509_V_ERR_CERT_NOT_YET_VALID:
		reason = int_reason_not_yet;
		break;
	case X509_V_ERR_ERROR_IN_CERT_NOT_BEFORE_FIELD:
		reason = int_reason_before;
		break;
	case X509_V_ERR_CERT_HAS_EXPIRED:
		reason = int_reason_expired;
		break;
	case X509_V_ERR_ERROR_IN_CERT_NOT_AFTER_FIELD:
		reason = int_reason_after;
		break;
	}

	if((cb_ssl_verify_level == 1) && ok)
		return ok;
	fprintf(stdout, "chain-depth=%d, ", depth);
	if(reason)
		fprintf(stdout, "error=%s\n", reason);
	else
		fprintf(stdout, "error=%d\n", err);
	if(cb_ssl_verify_level < 3)
		return ok;
	fprintf(stdout, "--> subject = %s\n", buf1);
	fprintf(stdout, "--> issuer  = %s\n", buf2);
	if(!ok)
		fprintf(stdout,"--> verify error:num=%d:%s\n",err,
			X509_verify_cert_error_string(err));
	fprintf(stdout, "--> verify return:%d\n",ok);
	return ok;
}

void msg_cb(int write_p, int version, int content_type, const void *buf, size_t len, SSL *ssl, void *arg)
{
	session_t *ssn = (session_t *)arg;
	const char *str_write_p, *str_version, *str_content_type = "", *str_details1 = "", *str_details2= "";
	
	str_write_p = write_p ? ">>>" : "<<<";

	ssn->type[3] = (unsigned char)write_p;
	ssn->type[2] = (unsigned char)content_type;
	ssn->type[1] = ((unsigned char*)buf)[1];
	ssn->type[0] = ((unsigned char*)buf)[0];

	switch (version)
	{
	case SSL2_VERSION:
		str_version = "SSL 2.0";
		break;
	case SSL3_VERSION:
		str_version = "SSL 3.0 ";
		break;
	case TLS1_VERSION:
		str_version = "TLS 1.0 ";
		break;
	default:
		str_version = "???";
	}

	if (version == SSL2_VERSION)
	{
		str_details1 = "???";

		if (len > 0)
		{
			switch (((unsigned char*)buf)[0])
			{
				case 0:
					str_details1 = ", ERROR:";
					str_details2 = " ???";
					if (len >= 3)
					{
						unsigned err = (((unsigned char*)buf)[1]<<8) + ((unsigned char*)buf)[2];
						
						switch (err)
						{
						case 0x0001:
							str_details2 = " NO-CIPHER-ERROR";
							break;
						case 0x0002:
							str_details2 = " NO-CERTIFICATE-ERROR";
							break;
						case 0x0004:
							str_details2 = " BAD-CERTIFICATE-ERROR";
							break;
						case 0x0006:
							str_details2 = " UNSUPPORTED-CERTIFICATE-TYPE-ERROR";
							break;
						}
					}

					break;
				case 1:
					str_details1 = ", CLIENT-HELLO";
					break;
				case 2:
					str_details1 = ", CLIENT-MASTER-KEY";
					break;
				case 3:
					str_details1 = ", CLIENT-FINISHED";
					break;
				case 4:
					str_details1 = ", SERVER-HELLO";
					break;
				case 5:
					str_details1 = ", SERVER-VERIFY";
					break;
				case 6:
					str_details1 = ", SERVER-FINISHED";
					break;
				case 7:
					str_details1 = ", REQUEST-CERTIFICATE";
					break;
				case 8:
					str_details1 = ", CLIENT-CERTIFICATE";
					break;
			}
		}
	}

	if (version == SSL3_VERSION || version == TLS1_VERSION)
	{
		switch (content_type)
		{
		case 20:
			str_content_type = "ChangeCipherSpec";
			break;
		case 21:
			str_content_type = "Alert";
			break;
		case 22:
			str_content_type = "Handshake";
			break;
		case 23:
			str_content_type = "ApplicationData";
			break;
		}

		if (content_type == 21) /* Alert */
		{
			str_details1 = ", ???";
			
			if (len == 2)
			{
				switch (((unsigned char*)buf)[0])
				{
				case 1:
					str_details1 = ", warning";
					break;
				case 2:
					str_details1 = ", fatal";
					break;
				}

				str_details2 = " ???";
				switch (((unsigned char*)buf)[1])
				{
				case 0:
					str_details2 = " close_notify";
					break;
				case 10:
					str_details2 = " unexpected_message";
					break;
				case 20:
					str_details2 = " bad_record_mac";
					break;
				case 21:
					str_details2 = " decryption_failed";
					break;
				case 22:
					str_details2 = " record_overflow";
					break;
				case 30:
					str_details2 = " decompression_failure";
					break;
				case 40:
					str_details2 = " handshake_failure";
					break;
				case 42:
					str_details2 = " bad_certificate";
					break;
				case 43:
					str_details2 = " unsupported_certificate";
					break;
				case 44:
					str_details2 = " certificate_revoked";
					break;
				case 45:
					str_details2 = " certificate_expired";
					break;
				case 46:
					str_details2 = " certificate_unknown";
					break;
				case 47:
					str_details2 = " illegal_parameter";
					break;
				case 48:
					str_details2 = " unknown_ca";
					break;
				case 49:
					str_details2 = " access_denied";
					break;
				case 50:
					str_details2 = " decode_error";
					break;
				case 51:
					str_details2 = " decrypt_error";
					break;
				case 60:
					str_details2 = " export_restriction";
					break;
				case 70:
					str_details2 = " protocol_version";
					break;
				case 71:
					str_details2 = " insufficient_security";
					break;
				case 80:
					str_details2 = " internal_error";
					break;
				case 90:
					str_details2 = " user_canceled";
					break;
				case 100:
					str_details2 = " no_renegotiation";
					break;
				}
			}
		}
		
		if (content_type == 22) /* Handshake */
		{
			str_details1 = "???";

			if (len > 0)
			{
				switch (((unsigned char*)buf)[0])
				{
				case 0:
					str_details1 = ", HelloRequest";
					break;
				case 1:
					str_details1 = ", ClientHello";
					break;
				case 2:
					str_details1 = ", ServerHello";
					break;
				case 11:
					str_details1 = ", Certificate";
					break;
				case 12:
					str_details1 = ", ServerKeyExchange";
					break;
				case 13:
					str_details1 = ", CertificateRequest";
					break;
				case 14:
					str_details1 = ", ServerHelloDone";
					break;
				case 15:
					str_details1 = ", CertificateVerify";
					break;
				case 16:
					str_details1 = ", ClientKeyExchange";
					break;
				case 20:
					str_details1 = ", Finished";
					break;
				}
			}
		}
	}

	fprintf(stdout, "%s %s%s [length %04lx]%s%s\n", str_write_p, str_version, str_content_type, (unsigned long)len, str_details1, str_details2);

	if (len > 0)
	{
		size_t num, i;
		
		fprintf(stdout, "   ");
		num = len;
		for (i = 0; i < num; i++)
		{
			if (i % 16 == 0 && i > 0)
				fprintf(stdout, "\n   ");
			fprintf(stdout, " %02x", ((unsigned char*)buf)[i]);
		}
		if (i < len)
			fprintf(stdout, " ...");
		fprintf(stdout, "\n");
	}
	fflush(stdout);
}

long bio_dump_cb(BIO *bio, int cmd, const char *argp, int argi,
	     long argl, long ret)
{
	BIO *out;

	out=(BIO *)BIO_get_callback_arg(bio);
	if (out == NULL) return(ret);

	if (cmd == (BIO_CB_READ|BIO_CB_RETURN))
	{
		BIO_printf(out,"read from %08X [%08lX] (%d bytes => %ld (0x%X))\n",
			bio,argp,argi,ret,ret);
		BIO_dump(out,argp,(int)ret);
		return(ret);
	}
	else if (cmd == (BIO_CB_WRITE|BIO_CB_RETURN))
	{
		BIO_printf(out,"write to %08X [%08lX] (%d bytes => %ld (0x%X))\n",
			bio,argp,argi,ret,ret);
		BIO_dump(out,argp,(int)ret);
	}
	return(ret);
}

#endif /* !defined(NO_OPENSSL) */
