/*
 * rlm_flow.c	
 *
 * Version:	$Id:$
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Copyright 2000  The FreeRADIUS server project
 * Copyright 2001  Mark Fullmer and The Ohio State University
 * Copyright 2002  Nikolay P. Romanyuk <mag@vtelecom.ru>
 */

static const char rcsid[] = "$Id:$";

#include "autoconf.h"
#include "libradius.h"

#include <sys/types.h>
#include <sys/socket.h>
#include <sys/time.h>
#include <sys/uio.h>

#include <netinet/in.h>
#include <arpa/inet.h>

#include <ctype.h>
#include <fcntl.h>
#include <netdb.h>
#include <stdio.h>
#include <stdlib.h>
#include <strings.h>

#include "radiusd.h"
#include "modules.h"
#include "conffile.h"

#include "flowmux/flowmux.h"
#include "rlm_flow.h"

static struct mppp *mppp = NULL;

static CONF_PARSER module_config[] = {
  { "flowmuxhost", PW_TYPE_STRING_PTR, offsetof(rlm_flow_t,flowmuxhost), NULL, NULL},
  { "flowmuxport", PW_TYPE_INTEGER,    offsetof(rlm_flow_t,flowmuxport), NULL,  "0"},
  { "num_sockets", PW_TYPE_INTEGER,    offsetof(rlm_flow_t,num_sockets), NULL, "3"},
  { "timeout",     PW_TYPE_INTEGER,    offsetof(rlm_flow_t,timeout), NULL, "5"},
  { NULL, -1, 0, NULL, NULL }		/* end the list */
};

/**************************
 * start of main routines *
 **************************/
static int rlm_flow_init(void)
{
	return 0;
}

static int rlm_flow_instantiate(CONF_SECTION * conf, void **instance)
{
	rlm_flow_t *inst;

	inst = rad_malloc(sizeof(rlm_flow_t));

	/*
	 * If the configuration parameters can't be parsed, then
	 * fail.
	 */
	if ( cf_section_parse(conf, inst, module_config) < 0 ) {
		free(inst);
		return (-1);
	}

	if ( inst->flowmuxhost == NULL ) {
		radlog(L_ERR, "rlm_flow: 'flowhost' must be set.");
		free(inst);
		return -1;
	}

	if ( inst->flowmuxport == 0 ) {
		radlog(L_ERR, "rlm_flow: 'flowport' must be set.");
		free(inst);
		return -1;
	}

	/*
	 * XXX Some sockets in 'flowmux' may be used other daemons.
	 */
	if ( inst->num_sockets > MAX_CONNS - RESERVED_CONS ) {
		radlog(L_ERR | L_CONS, "flow_instantiate: number of sockets cannot exceed %d (flowmux)", MAX_CONNS - RESERVED_CONS);
		free(inst);
		return (-1);
	}

	radlog(L_INFO, "rlm_flow: Attemting to connected to %s:%d", inst->flowmuxhost, inst->flowmuxport);

	if ( flow_poolinit(inst) < 0 ) {
		free(inst);
		return (-1);
	}

	*instance = inst;
	return (RLM_MODULE_OK);
}

static int rlm_flow_destroy(void)
{
	return (0);
}

static int rlm_flow_detach(void *instance)
{
	rlm_flow_t *inst = instance;

	flow_poolfree(inst);
	free(inst);

	return (0);
}

static int flow_poolinit(rlm_flow_t *inst)
{
	FLOWSOCK *flowsock;
	int i;

	inst->sockpool = NULL;
	inst->used = 0;
	inst->addr = get_fip(inst->flowmuxhost);

	for ( i = 0; i < inst->num_sockets; i++ ) {
		flowsock = rad_malloc(sizeof(FLOWSOCK));
		if ( flowsock == NULL ) {
			return(-1);
		}
		bzero(flowsock, sizeof(FLOWSOCK));
		flowsock->id = i;
		flowsock->state = sockunconnected;
#if HAVE_SEMAPHORE_H
		/*
		 * FIXME! Check return codes!
		 */
		flowsock->semaphore = (sem_t *) rad_malloc(sizeof(sem_t));
		sem_init(flowsock->semaphore, 0, FLOWSOCK_UNLOCKED);
#else
		flowsock->in_use = FLOWSOCK_UNLOCKED;
#endif
		connect_flowsock(flowsock, inst);

		flowsock->next = inst->sockpool;
		inst->sockpool = flowsock;
	}
	return(1);
}

static void flow_poolfree(rlm_flow_t *inst)
{
	FLOWSOCK *p;

	for ( p = inst->sockpool; p; p = p->next ) {
		close_flowsock(p);
	}
}

static void connect_flowsock(FLOWSOCK *p, rlm_flow_t *inst)
{
	struct sockaddr_in fsin;

	p->fd = socket(PF_INET, SOCK_STREAM, 0);
	if ( p->fd < 0 ) {
		radlog(L_CONS | L_ERR, "rlm_flow: Failed create flowsock %d", p->id);
		return;
	}
	fsin.sin_family = PF_INET;
	fsin.sin_port = htons(inst->flowmuxport);
	fsin.sin_addr.s_addr = inst->addr;
	if ( connect(p->fd, (struct sockaddr *)&fsin, sizeof(fsin)) < 0 ) {
		radlog(L_CONS | L_ERR, "rlm_flow: Failed connect flowsock %d", p->id);
		return;
	}
	if ( fcntl(p->fd, F_SETFL, O_NONBLOCK) < 0 ) {
		radlog(L_CONS | L_ERR, "rlm_flow: Failed fcntl(O_NONBLOCK) flowsock %d", p->id);
		return;
	}
	p->state = sockconnected;

	radlog(L_DBG, "rlm_flow: Socked %d connected success", p->id);
}

static void close_flowsock(FLOWSOCK *p)
{
	radlog(L_DBG, "rlm_flow: Closing flowsock %d", p->id);
	close(p->fd);
#if HAVE_SEMAPHORE_H
	sem_destroy(p->semaphore);
#endif
	free(p);
}

/* Got from Flow-Tools. See: http://www.splintered.net/sw/flow-tools/ */
static in_addr_t get_fip(char *host)
{
	struct hostent *h;
	struct in_addr *in;
	u_long addr = 0;
	u_int n;
	int dns = 0;
	char *s;

	for ( s = host; *s; s++ ) {
		if ( isalpha(*s) ) {
			dns = 1;
			break;
		}
	}
	if ( dns ) {
		h = gethostbyname(s);
		if ( h == NULL ) goto num;
		if ( h->h_addrtype != PF_INET ) goto num;
		if ( h->h_length != sizeof(u_int32_t) ) goto num;
		in = (struct in_addr *) *h->h_addr_list;
		return (in->s_addr);
	}
num:
	s = host;
	while (1) {
		n = 0;
		while (*s && (*s != '.') && (*s != ' ') && (*s != '\t'))
			n = n * 10 + *s++ - '0';
		addr <<=8;
		addr |= n & 0xff;
		if ((!*s) || (*s == ' ') || (*s == '\t'))
			return(htonl(addr));
		++s;
	}
}

static FLOWSOCK *get_socket(rlm_flow_t *inst)
{
	FLOWSOCK *p;

	if ( inst->used == inst->num_sockets ) {
		radlog(L_ERR, "rlm_flow: All sockets are being used! Please increase the maximum number of sockets!");
		return(NULL);
	}
	for ( p = inst->sockpool; p; p = p->next ) {
		if ( p->state == sockunconnected ) {
			radlog(L_INFO, "rlm_flow: Trying to (re)connect an unconnected handle...");
			connect_flowsock(p, inst);
		}
		if ( p->state == sockunconnected ) {
			radlog(L_DBG, "rlm_flow: Ignoring unconnected handle");
			continue;
		}
#if HAVE_SEMAPHORE_H
		if ( sem_trywait(p->semaphore) == 0 ) {
#else
		if ( p->in_use == FLOWSOCK_UNLOCKED ) {
#endif
			(inst->used)++;
#ifndef HAVE_SEMAPHORE_H
			p->in_use = FLOWSOCK_LOCKED;
#endif
			radlog(L_DBG, "rlm_flow: Reserving flow socket id: %d", p->id);
			return (p);
		}
	}
	radlog(L_CONS | L_ERR, "rlm_flow: There are no Flow sockets to use!");
	return (NULL);
}

static void release_socket(FLOWSOCK *flowsock, rlm_flow_t *inst)
{
	(inst->used)--;
#if HAVE_SEMAPHORE_H
	sem_post(flowsock->semaphore);
#else
	flowsock->in_use = FLOWSOCK_UNLOCKED;
#endif
	radlog(L_DBG, "rlm_flow: Released flow socket id: %d", flowsock->id);
}

/*
 *	Massage the request before recording it or proxying it
 */
static int rlm_flow_preacct(void *inst, REQUEST *request)
{
	VALUE_PAIR *pair;
	FLOWSOCK *flowsock = NULL;
	int status = 0;
	int res = 0;
	u_long output_octets = 0;
	struct cmd cmd;
	struct in_addr in;

	if ((pair = pairfind(request->packet->vps, PW_ACCT_STATUS_TYPE)) !=
							NULL ) {
		status = pair->lvalue;
	} else {
		radlog(L_ERR, "rlm_flow: packet has no account status type.");
		return (RLM_MODULE_NOOP);
	}

	if ((pair = pairfind(request->packet->vps, PW_FRAMED_IP_ADDRESS)) ==
							NULL ) {
		radlog(L_ERR, "rlm_flow: packet has no Framed-Ip-Adress.");
		return (RLM_MODULE_NOOP);
	}
	cmd.addr = pair->lvalue;

	if ( status == PW_STATUS_ACCOUNTING_ON ||
			status == PW_STATUS_ACCOUNTING_OFF ) {
	/* XXX Fix me!!! Clean mppp and 'flowmux' pools. */
		return (RLM_MODULE_NOOP);
	}

	/* Don't need any if MPPP */
	if ( check_mppp(request->username->strvalue, status, cmd.addr) ) {
		radlog(L_INFO, "rlm_flow: MPPP for login '%s'", request->username->strvalue);
		return(RLM_MODULE_NOOP);
	}

	switch ( status ) {
		case PW_STATUS_ALIVE:
		case PW_STATUS_START:
			flowsock = get_socket(inst);
			if ( flowsock == NULL ) {
				return(RLM_MODULE_FAIL);
			}
			cmd.cmd = FLOW_ACCT_START;
			res = write_sock(flowsock, &cmd);
			break;
		case PW_STATUS_STOP:
			flowsock = get_socket(inst);
			if ( flowsock == NULL ) {
				return(RLM_MODULE_FAIL);
			}
			cmd.cmd = FLOW_ACCT_STOP;
			res = write_sock(flowsock, &cmd);
			res = read_sock(flowsock, &output_octets, inst);
			break;
	}
	release_socket(flowsock, inst);
	if ( res ) return (RLM_MODULE_NOOP);

	if ( output_octets && status == PW_STATUS_STOP ) {
		if ((pair = pairfind(request->packet->vps, PW_ACCT_OUTPUT_OCTETS)) == NULL ) {
			radlog(L_ERR, "rlm_flow: packet has no Acct_Output_Octets.");
		} else {
			in.s_addr = cmd.addr;
			radlog(L_INFO, "rlm_flow: Replace Acct_Output_octets: Old %d - New %lu for '%s'(%s)", pair->lvalue, output_octets, request->username->strvalue, inet_ntoa(in));
#if 0
			/*
			 * XXX Fix me!!! Uncomment for real accounting.
			 */
			pair->lvalue = output_octets;
#endif
		}
	} else if ( status == PW_STATUS_STOP ) {
		in.s_addr = cmd.addr;
		radlog(L_INFO, "rlm_flow: Zero traffic or not response for '%s'(%s)", request->username->strvalue, inet_ntoa(in));
	}
	return (RLM_MODULE_OK);
}

static int write_sock(FLOWSOCK *p, struct cmd *cmd)
{
	int i;

	while (1) {
		i = write(p->fd, (char *)cmd, sizeof(struct cmd));
		if ( i < 0 || i != sizeof(struct cmd) ) {
			if ( errno == EAGAIN ) continue;
			else {
				radlog(L_CONS | L_ERR, "rlm_flow: write error %s", strerror(errno));
				close(p->fd);
				p->state = sockunconnected;
				return (-1);
			}
		}
		return (0);
	}
}

static int read_sock(FLOWSOCK *p, u_long *res, rlm_flow_t *inst)
{
	struct timeval t;
	int i;
	fd_set rfd;

	t.tv_sec = inst->timeout;
	t.tv_usec = 0;

	FD_ZERO(&rfd);
	FD_SET(p->fd, &rfd);

	if ( select(p->fd + 1, &rfd, (fd_set *)NULL, (fd_set *)NULL, &t) < 0 ) {
		radlog(L_CONS | L_ERR, "rlm_flow: poll");
		close(p->fd);
		p->state = sockunconnected;
		return (-1);
	}

	if ( FD_ISSET(p->fd, &rfd) ) {
		while (1) {
			i = read(p->fd, (char *)res, sizeof(u_long));
			if ( i < 0 || i != sizeof(u_long) ) {
				if ( errno == EAGAIN ) continue;
				else {
					radlog(L_CONS | L_ERR, "rlm_flow: read error %s", strerror(errno));
					close(p->fd);
					p->state = sockunconnected;
					return (-1);
				}
			}
			return (0);
		}
	}
	return (-1);
}

static int check_mppp(char *login, int status, in_addr_t addr)
{
	struct mppp *p;
	int len;

	len = strlen(login);
	for ( p = mppp; p; p = p->next ) {
		if ( p->addr == addr ) {
			if ( p->login && strncmp(p->login, login, len) == 0 ) {
				switch (status) {
				case PW_STATUS_START:
				case PW_STATUS_ALIVE:
					p->count++;
					return(1);
				case PW_STATUS_STOP:
					if ( p->count > 1 ) {
						p->count--;
						return(1);
					}
					p->count = 0;
					free(p->login);
					p->login = NULL;
					return(0);
				}
			} else {
				if ( p->login ) free(p->login);
				p->login = strdup(login);
				p->count = 1;
				return(0);
			}
		}
	}

	if ( status == PW_STATUS_START || status == PW_STATUS_ALIVE ) {
		p = rad_malloc(sizeof(struct mppp));
		if ( p == NULL ) return(-1);
		bzero(p, sizeof(struct mppp));

		p->login = strdup(login);
		p->addr = addr;
		p->count = 1;

		p->next = mppp;
		mppp = p;
	}
	return(0);
}

/*
 *	The module name should be the only globally exported symbol.
 *	That is, everything else should be 'static'.
 *
 *	If the module needs to temporarily modify it's instantiation
 *	data, the type should be changed to RLM_TYPE_THREAD_UNSAFE.
 *	The server will then take care of ensuring that the module
 *	is single-threaded.
 */
module_t rlm_flow = {
	"IPFLOW",
	RLM_TYPE_THREAD_SAFE,		/* type */
	rlm_flow_init,			/* initialization */
	rlm_flow_instantiate,		/* instantiation */
	{
		NULL,			/* authentication */
		NULL,			/* authorization */
		rlm_flow_preacct,	/* preaccounting */
		NULL,			/* accounting */
		NULL			/* checksimul */
	},
	rlm_flow_detach,		/* detach */
	rlm_flow_destroy,		/* destroy */
};
