#ifndef _LIBRADIUS_H
#define _LIBRADIUS_H

#include <sys/types.h>
#ifndef WIN32
#include <sys/time.h>
#include <netinet/in.h>
#include <sys/socket.h>
#include <sys/stat.h>
#include <netdb.h>
#include <unistd.h>
#else WIN32
#include <winsock.h>
#include <process.h>
#include <time.h>
#include <io.h>
#endif WIN32
#include <fcntl.h>
#include <assert.h>
#include <stdio.h>
#ifdef AIX
#include <time.h>
#endif AIX
#include "radius.h"

#define _LIBRADIUS_H_ID "$Id: libradius.h,v 1.14 1999/03/12 12:51:18 aland Exp $"

/***********************************************************************
 * Miscellaneous defines
 ***********************************************************************/

/* For applications: we're using libradius */
#define LIBRADIUS

/* RFC2138 p.11 */
#define MIN_PACKET_SIZE 20
#define MAX_PACKET_SIZE 4096

/* most clients/servers can only handle 128, but we want to be pedantic */
#define MAX_ATTR_LEN 253

/* RFC2138 recommends 16 as the minimum */
#define MAX_SECRET_SIZE 256

#ifndef FALSE
#define FALSE 0
#define TRUE (!FALSE)
#endif FALSE

/* for lrad_packet_compliant, packets with attributes not in the RFCs */
#define CONDITIONAL 2

/* encrypt or decrypt passwords */
#define LRAD_ENCRYPT 0
#define LRAD_DECRYPT (!LRAD_ENCRYPT)

/* libRADIUS specific defines for pseudo-functions */
#define lrad_attr_next(attr) (RADIUS_ATTR *) ((char *)attr + attr->length)
#define lrad_packet_lengthen(packet, attr) packet->length = htons(ntohs(packet->length) + attr->length)

/* defines for lazy typist functions */
#define lrad_select	lrad_socket_select
#define lrad_sendto	lrad_socket_sendto
#define lrad_recvfrom	lrad_socket_recvfrom

#ifdef WIN32
#define strcasecmp stricmp
#define strncasecmp strnicmp
#endif WIN32

/***********************************************************************
 * Structure definitions.
 * These are not RADIUS structures, so they're defined here.
 ***********************************************************************/

typedef struct dict_attr {
	struct dict_attr	*next;
	u_int			value;
	u_char			name[48];
	int			type;
	struct dict_value	*values;
	struct dict_vendor	*vendor;
	char			allowed[16];
} DICT_ATTR;

typedef struct dict_value {
	struct dict_value	*next;
	u_int			value;
	u_char			name[48];
} DICT_VALUE;

typedef struct dict_vendor {
	struct dict_vendor	*next;
	u_int			id;
	u_char			name[32];
	int			attrib_nmc; /* for USR style VSA's */
	DICT_ATTR		*attributes;
} DICT_VENDOR;

/*
 *  To make searching linked lists easier
 */
typedef struct dict_list {
	struct dict_list	*next;
	u_int			value;
	u_char			name[32]; /* at least this large */
} DICT_LIST;

typedef struct lrad_token_t {
	const char		*token;
	int			value;
} lrad_token_t;

/***********************************************************************
 * defines for operators used in parsing Attribute = Value strings
 ***********************************************************************/
enum {
  PW_OPERATOR_EQUAL = 0,	/* 0  = */
  PW_OPERATOR_COMMA,		/* 1  , */
  PW_OPERATOR_SET_EQUAL,	/* 2  := */
  PW_OPERATOR_ADD_EQUAL,	/* 3  += */
  PW_OPERATOR_SUB_EQUAL,	/* 4  -= */
  PW_OPERATOR_NOT_EQUAL,	/* 5  != */
  PW_OPERATOR_LESS_EQUAL,	/* 6  <= */
  PW_OPERATOR_GREATER_EQUAL,	/* 7  >= */
  PW_OPERATOR_LESS_THAN,	/* 8  < */
  PW_OPERATOR_GREATER_THAN,    	/* 9  > */
  PW_OPERATOR_OR,		/* 10  || */
  PW_OPERATOR_AND,		/* 11  && */
  PW_OPERATOR_NOT,		/* 12  ! */
  PW_OPERATOR_STRING,		/* 13  a generic string */
  PW_OPERATOR_EOS,		/* 14  end of the string */
  PW_OPERATOR_EOL,		/* 15  end of the line */
  PW_OPERATOR_ILLEGAL,		/* 16 illegal operator */
};
#define PW_OPERATOR_MAX PW_OPERATOR_ILLEGAL

typedef struct value_pair {
	struct value_pair	*next;
	DICT_ATTR		*attr;
	int			length;	/* for strings: may be opaque data */
	int			operator;
	int			tag;    /* only used in Tag-Int, Tag-String */
	u_int			lvalue;
	u_char			strvalue[32]; /* grows as needed */
} VALUE_PAIR;

/***********************************************************************
 * libRADIUS extensions to RADIUS attributes
 ***********************************************************************/
enum {
  PW_PACKET_CODE = 2000,	/* packet->code (Access-Request, etc) */
  PW_PACKET_ID,			/* ID of the packet */
  PW_PACKET_LENGTH,		/* Length of the packet */
  PW_PACKET_VECTOR,		/* authentication vector of the packet */
  PW_SERVER_ADDRESS,		/* IP address of the server */
  PW_SERVER_PORT,		/* port for the server */
  PW_SERVER_SECRET,		/* the obvious */
  PW_CLIENT_ADDRESS,		/* IP address of the client */
  PW_CLIENT_PORT,		/* port the client is coming from */
  PW_CLIENT_TIMEOUT,		/* wait time before the client gives up */
  PW_RAW_ATTRIBUTES,		/* raw RADIUS attributes for quick replies */
  PW_CACHE_TOKEN,		/* cache the current token yes/no */
  PW_CACHE_TIMEOUT,		/* timeout for the cache entry */
  PW_CACHED_CALLING_STATION_ID,	/* used as a key for the cache */
  PW_TIMESTAMP,			/* timestamp from the 'detail' file */
  PW_REQUEST_AUTHENTICATOR,	/* from 'detail' file */
  PW_PROXY_SERVER,		/* Server to proxy the request to */
  PW_PROXY_PORT,		/* port for the proxy server */
  PW_TEXT,			/* miscellaneous text */
};

/***********************************************************************
 * Global variables
 *
 * Yes, I know that this is bad practice.  However, it's SO much more
 * useful if it can be done in certain limited situations.
 *
 * You can step through the dictionary yourself, but any modifications to the
 * data structures will have undefined side effects.
 *
 * You can turn hostname lookups on and off.  Turning lookups off
 * may make your server respond more quickly.
 ***********************************************************************/
extern DICT_ATTR	*lrad_dict_attributes;
extern DICT_VENDOR	*lrad_dict_vendors;
extern int		lrad_hostname_lookups;
extern char 		**lrad_tunnel_names;
extern const char	*lrad_operators[];
extern const char	*lrad_data_types[];
/***********************************************************************
 * Function prototypes
 ***********************************************************************/

/* attribute.c */
extern DICT_LIST *lrad_find_entry_type(DICT_LIST *list, u_int type);
extern DICT_LIST *lrad_find_entry_name(DICT_LIST *list, const u_char *name);
extern RADIUS_ATTR *lrad_attr_fetch(AUTH_HDR *packet, u_char type);
extern RADIUS_ATTR *lrad_attr_fetch_next(AUTH_HDR *packet, RADIUS_ATTR * attr, u_char type);
extern u_char *lrad_attr_fetch_password(AUTH_HDR *packet, u_char type, const u_char *secret);
extern RADIUS_ATTR *lrad_attr_add(AUTH_HDR *packet, u_char type, const u_char *data, int length);
extern RADIUS_ATTR *lrad_attr_move2(AUTH_HDR *packet, RADIUS_ATTR *from);
extern RADIUS_ATTR *lrad_attr_add_int(AUTH_HDR *packet, u_char type, int data);
extern RADIUS_ATTR *lrad_attr_add_password(AUTH_HDR *packet, u_char type, const u_char *password, const u_char *secret);
extern RADIUS_ATTR *lrad_attr_add_pair(AUTH_HDR *packet, VALUE_PAIR *pair);
extern void lrad_attr_add_pair_list(AUTH_HDR *packet, VALUE_PAIR *pair);
extern VALUE_PAIR *lrad_attr_2pair(RADIUS_ATTR *rad);
extern void lrad_packet_add_pair(AUTH_HDR *packet, VALUE_PAIR *pair);

/* compliance.c */
extern int lrad_compliant_pair(VALUE_PAIR *pair);
extern int lrad_compliant_pair_list(VALUE_PAIR *pair);
extern int lrad_compliant_packet(int code, VALUE_PAIR *first);

/* dict.c */
extern int lrad_dict_merge(const char *filename);
extern int lrad_dict_init(const char *filename);
extern DICT_ATTR *lrad_dict_attr_type(u_int attribute);
extern DICT_ATTR *lrad_dict_attr_name(const u_char *name);
extern DICT_VALUE *lrad_dict_value_type(DICT_ATTR *attr, u_int lvalue);
extern DICT_VALUE *lrad_dict_value_name(DICT_ATTR *attr, const u_char *name);
extern DICT_VENDOR *lrad_dict_vendor_type(int vendor_id);
extern DICT_VENDOR *lrad_dict_vendor_name(const u_char *name);
extern DICT_ATTR *lrad_dict_vendor_attr_type(DICT_VENDOR *vendor, u_int type);
extern DICT_ATTR * lrad_dict_vendor_attr_name(DICT_VENDOR *vendor, const u_char *name);
extern DICT_ATTR *lrad_dict_any_attr_name(const u_char *name);
extern void lrad_dict_free(void);

/* error.c */
extern const char *lrad_strerror(void);

/* exec.c */
extern int lrad_exec(char *const argv[], int exec_wait, char **output);

/* misc.c */
extern void lrad_random_vector(u_char *vector);
extern u_char lrad_random_byte(void);
extern int lrad_string_dequote(u_char *buffer, u_char **start, u_char **end);
extern int lrad_string_split(char *buffer, char **array, int max_entries);
extern unsigned int lrad_get_ipaddr(char *hostname);
extern void lrad_free(void *p, int size);
extern int lrad_date_2tm(char *date, struct tm *tm);
extern void lrad_hmac(u_char *output, u_char *key, u_char *data, int length);
extern int lrad_data_pack(u_char *data, VALUE_PAIR *pair);
extern void lrad_data_unpack(VALUE_PAIR *pair, u_char *data, int length);
#ifdef WIN32
#ifndef HAVE_GETTIMEOFDAY
extern int gettimeofday(struct timeval *tv, void *);
#endif HAVE_GETTIMEOFDAY
#endif WIN32
extern int lrad_verify_signature(AUTH_HDR *packet, u_char *secret);

/* nmc.c */
extern NMC_ATTR * lrad_nmc_attr_add(AUTH_HDR *packet, u_int vendor_id, u_int attrib_nmc, u_char *data, int length);
extern NMC_ATTR *lrad_nmc_attr_add_int(AUTH_HDR *packet, u_int vendor_id, u_int attrib_nmc, int data);
extern VALUE_PAIR *lrad_nmc_attr_2pair(DICT_VENDOR *vendor, NMC_ATTR *nmc);
extern int lrad_nmc_pair_2attr(RADIUS_ATTR *input_attr, VALUE_PAIR *pair);

/* packet.c */
extern AUTH_HDR *lrad_packet_create(int code);
extern void lrad_packet_sign(AUTH_HDR *request, AUTH_HDR *reply, u_char *secret);
extern int lrad_packet_verify(AUTH_HDR *request, AUTH_HDR *reply, u_char *secret);
extern VALUE_PAIR *lrad_packet_2pair(AUTH_HDR *request);

/* pair.c */
extern void lrad_pair_list_free(VALUE_PAIR *pair);
extern VALUE_PAIR * lrad_pair_fetch(VALUE_PAIR *first, u_int attribute);
extern VALUE_PAIR *lrad_pair_vsa_fetch(VALUE_PAIR *pair, u_int vendor_id, u_int attribute);
extern void lrad_pair_delete(VALUE_PAIR **first, u_int attribute);
extern void lrad_pair_add(VALUE_PAIR **first, VALUE_PAIR *new);
extern VALUE_PAIR *lrad_pair_create(DICT_ATTR *attr, u_char *data, int length);
extern VALUE_PAIR *lrad_pair_list_copy(VALUE_PAIR *from);
extern void lrad_pair_chain(VALUE_PAIR **to, VALUE_PAIR *from);
extern void lrad_pair_move(VALUE_PAIR **to, VALUE_PAIR **from);
extern void lrad_pair_move2(VALUE_PAIR **to, VALUE_PAIR **from, u_int type);
extern int lrad_pair_2attr(RADIUS_ATTR *attr, VALUE_PAIR *pair);
extern int lrad_pair_list_2attr(u_char *buffer, VALUE_PAIR *pair);
extern AUTH_HDR *lrad_pair_2packet(VALUE_PAIR *pair);

/* parse.c */
extern int lrad_parse_ipaddr(u_char *hostname, u_int *ipaddr);
extern int lrad_parse_lvalue(DICT_ATTR *attr, u_char *string, int *lvalue);
extern int lrad_parse_string(u_char *string, u_char *data, int *length);
extern int lrad_parse_date(u_char *string, u_int *date);
extern int lrad_parse_octets(u_char *p, u_char *data, int *length);
extern VALUE_PAIR *lrad_parse_value(DICT_ATTR *attr, char *string);
extern int lrad_parse_token(char **string, char *data, int length);
extern VALUE_PAIR * lrad_parse_pair(char **string);
extern int lrad_split_pair_string(char *buffer, char ***argv, int max_entries);

/* password.c */
extern void lrad_password_crypt(u_char *out, u_char *in, int length, u_char *vector, const u_char *secret, int flag);
extern void lrad_password_chap(u_char *chap_password, int id, const u_char *password, int password_len, const u_char *challenge, int challenge_len);

/* print.c */
extern void lrad_print_ipaddr(char *output, unsigned int ipaddr);
extern void lrad_print_lvalue(char *output, DICT_ATTR *attr, u_int lvalue);
extern void lrad_print_string(char *output, u_char *string, int length);
extern void lrad_print_date(char *output, int date);
extern void lrad_print_octets(char *output, u_char *data, int length);
extern void lrad_print_tagint(char *output, DICT_ATTR *attr, int tag, u_int lvalue);
extern void lrad_print_tagstr(char *output, int tag, u_char *p, int length);
extern void lrad_print_pair_list(char *output, VALUE_PAIR *pair);
extern void lrad_print_pair(char *output, VALUE_PAIR *pair);
extern void lrad_print_packet(char *output, AUTH_HDR *packet);
extern void lrad_print_packet_raw(char *output, AUTH_HDR *packet);
extern void lrad_print_attr_value(char *output, DICT_ATTR *dict_attr, char *data, int length);
extern void lrad_print_vendor_attr(char *output, DICT_VENDOR *dict_vendor, VENDOR_ATTR *vsa);
extern void lrad_print_attr(char *output, RADIUS_ATTR *attr);

/* proxy.c */
extern int lrad_proxy_pass(AUTH_HDR *packet, u_char *proxy_secret, u_char *remote_secret);
extern int lrad_proxy_return(AUTH_HDR *packet, AUTH_HDR *reply, u_char *proxy_secret, u_char *remote_secret);

/* socket.c */
extern int lrad_socket_select(int maxfd, fd_set *readfds, int waittime);
extern int lrad_socket_sendto(AUTH_HDR *packet, struct sockaddr *saremote, int sockfd);
extern int lrad_socket_recvfrom(AUTH_HDR *packet, struct sockaddr *saremote, int sockfd, int timeout);
extern u_short lrad_socket_get_port(char *port);
extern int lrad_socket_open(struct sockaddr *salocal);
extern int lrad_socket_client_open(char *server, char *port, struct sockaddr *saremote);
extern int lrad_socket_server_open(char *port);
extern int lrad_client_send_recv(AUTH_HDR *packet, AUTH_HDR *reply, char *server, char *port, int timeout);
#ifdef WIN32
extern int lrad_initialize_winsock(int major, int minor);
#endif WIN32

/* vendor.c */
extern VENDOR_ATTR *lrad_vsa_fetch(AUTH_HDR *packet, u_int vendor_id);
extern VENDOR_ATTR *lrad_vsa_fetch_next(AUTH_HDR *packet, VENDOR_ATTR *vsa, u_int vendor_id);
extern VENDOR_ATTR *lrad_vsa_add(AUTH_HDR *packet, u_int vendor_id, u_char *data, int length);
extern VENDOR_ATTR *lrad_vsa_attr_add(AUTH_HDR *packet, u_int vendor_id, u_char type, u_char *data, int length);
extern VENDOR_ATTR * lrad_vsa_attr_add_int(AUTH_HDR *packet, u_int vendor_id, u_char type, int data);
extern RADIUS_ATTR *lrad_vsa_attr_fetch(VENDOR_ATTR *vsa, u_char vendor_type);
extern VALUE_PAIR *lrad_vsa_attr_2pair(DICT_VENDOR *vendor, VENDOR_ATTR *vsa);
extern int lrad_vsa_pair_2attr(RADIUS_ATTR *input_attr, VALUE_PAIR *pair);

/*
 *  These next defines are REALLY paranoid, but they will make
 *  your application MUCH more secure!
 */
#ifdef PARANOID_SECURITY
#define popen		INSECURE_DONT_USE_popen
#define system		INSECURE_DONT_USE_system

/*
 *  Allow strcpy(foo, "bar"), but NOT strcpy(foo, bar)
 */
#define strcpy(FOO, BAR) 	strcpy(FOO, BAR "")
#define strcat(FOO, BAR)	strcat(FOO, BAR "")
#define strcmp(FOO, BAR)	strcmp(FOO, BAR "")
#define strcoll(FOO, BAR)	strcoll(FOO, BAR "")
#define strspn(FOO, BAR)	strspn(FOO, BAR "")
#define strcspn(FOO, BAR)	strcspn(FOO, BAR "")
#define strstr(FOO, BAR)	strstr(FOO, BAR "")
#define strlen(FOO)		strlen(FOO "")
#define strdup(FOO)		strdup(FOO "")

/*
 *  You don't want to use these functions at ALL!
 */
#define gets		UNBOUNDED_SIDE_EFFECTS_gets
#define scanf		UNBOUNDED_SIDE_EFFECTS_scanf
#define fscanf		UNBOUNDED_SIDE_EFFECTS_fscanf
#define sscanf		UNBOUNDED_SIDE_EFFECTS_sscanf
#endif PARANOID_SECURITY

#endif _LIBRADIUS_H
