#ifndef RADIUS_H
#define RADIUS_H

#define AUTH_VECTOR_LEN		16
#define AUTH_PASS_LEN		16
#define AUTH_PASS_MASK          (AUTH_PASS_LEN - 1)
#define AUTH_STRING_LEN		128
#define MAXPASS                 16
#define MAX_PACKET_SIZE         4096

#define CHAP_VALUE_LENGTH		16

/* The RFC's say to use 1812, but everyone uses 1645. */
#ifdef USE_RFC_PORT
#define PW_AUTH_UDP_PORT		1812
#define PW_ACCT_UDP_PORT		1813
#else USE_RFC_PORT
#define PW_AUTH_UDP_PORT		1645
#define PW_ACCT_UDP_PORT		1646
#endif USE_RFC_PORT

/*
 *  The RADIUS RFC's are 2138 (authentication) and 2139 (accounting)
 *
 *  A good background to RADIUS can be found at:
 *
 * http://www.cisco.com/univercd/cc/td/doc/product/software/ios112/112cg_cr/2cbook/2caaa.htm
 *
 * Multiple dictionaries are:
 * http://www.actsofia.bg/cis/soft/csec/doc/append_g.htm
 * http://www.merit.edu/aaa/dictionary24.txt
 */


/*
 * Data types
 */
enum {
  PW_TYPE_STRING = 0,		/* look, Pa, it's a string! */
  PW_TYPE_INTEGER,		/* a raw 4-octet unsigned integer */
  PW_TYPE_IPADDR,		/* like an int, but an IPv4 address */
  PW_TYPE_DATE,			/* like an int, but a Unix-style date in sec */
  PW_TYPE_OCTET,		/* 1-octet unsigned integer */
  PW_TYPE_SHORT,		/* 2-octet unsigned integer */
  PW_TYPE_OCTETS,		/* like a string, but displays as hex */
  PW_TYPE_TAGINT,		/* one octet tag, 3 octet integer */
  PW_TYPE_TAGSTR,		/* one octet tag, 0-253 octet string */
  PW_TYPE_MAX
};

/*
 * RADIUS Codes from RFC2138, p.10
 */
enum {
  PW_ACCESS_REQUEST = 1,	/*  1  client requests access */
  PW_ACCESS_ACCEPT,		/*  2  server acknowledges request */
  PW_ACCESS_REJECT,		/*  3  server rejects request */
  PW_ACCOUNTING_REQUEST,	/*  4  client requests accounting */
  PW_ACCOUNTING_RESPONSE,	/*  5  server says yes to accounting */
  PW_ACCOUNTING_STATUS,		/*  6  Not in RFC2138 */
  PW_PASSWORD_REQUEST,		/*  7  Not in RFC2138 */
  PW_PASSWORD_ACK,		/*  8  Not in RFC2138 */
  PW_PASSWORD_REJECT,		/*  9  Not in RFC2138 */
  PW_ACCOUNTING_MESSAGE,	/* 10  Not in RFC2138 */
  PW_ACCESS_CHALLENGE,		/* 11  server challenges the client */
  PW_STATUS_SERVER,		/* 12  RFC2138 experimental */
  PW_STATUS_CLIENT,		/* 13  RFC2138 experimental */

  /* USR Codes */
#ifdef USR_RADIUS
  PW_RESOURCE_FREE_REQ = 21	/* 21 */
  PW_RESOURCE_FREE_RESP,	/* 22 */
  PW_RESOURCE_QUERY_REQ,	/* 23 */
  PW_RESOURCE_QUERY_RESP,	/* 24 */
  PW_ALT_RESOURCE_RECLAIM_REQ,	/* 25  USR NAS accepts 25 AND 265 */
  PW_NAS_REB_REQ,		/* 26 */
  PW_NAS_REB_RESP,		/* 27 */
#endif USR_RADIUS

  /* see draft-calhoun-diameter-04.txt */
  PW_EXTENDED_FORMAT = 254,	/* DIAMETER: Extended format */
  PW_RESERVED,			/* reserved in RFC 2138 */
  PW_REPLY_TIMEOUT,		/* libRADIUS: timeout waiting for reply */
};
#define PW_ACCESS_ACK PW_ACCESS_ACCEPT

/*
 * RADIUS Attributes, from RFC2138 p.19
 */
enum {
  PW_USER_NAME = 1,
  PW_USER_PASSWORD,
  PW_CHAP_PASSWORD,
  PW_NAS_IP_ADDRESS,
  PW_NAS_PORT,
  PW_SERVICE_TYPE,
  PW_FRAMED_PROTOCOL,
  PW_FRAMED_IP_ADDRESS,
  PW_FRAMED_IP_NETMASK,

  /* 10 */
  PW_FRAMED_ROUTING,
  PW_FILTER_ID,
  PW_FRAMED_MTU,
  PW_FRAMED_COMPRESSION,
  PW_LOGIN_IP_HOST,
  PW_LOGIN_SERVICE,
  PW_LOGIN_TCP_PORT,
  PW_OLD_PASSWORD,		/* RFC2138 says this is unassigned */
  PW_REPLY_MESSAGE,
  PW_CALLBACK_NUMBER,

  /* 20 */
  PW_CALLBACK_ID,
  PW_EXPIRATION,		/* RFC2138 says this is unassigned */
  PW_FRAMED_ROUTE,
  PW_FRAMED_IPXNET,
  PW_STATE,
  PW_CLASS,
  PW_VENDOR_SPECIFIC,
  PW_SESSION_TIMEOUT,
  PW_IDLE_TIMEOUT,
  PW_TERMINATION_ACTION ,

  /* 30 */
  PW_CALLED_STATION_ID,
  PW_CALLING_STATION_ID,
  PW_NAS_IDENTIFIER,
  PW_PROXY_STATE,
  PW_LOGIN_LAT_SERVICE,
  PW_LOGIN_LAT_NODE,
  PW_LOGIN_LAT_GROUP,
  PW_FRAMED_APPLETALK_LINK,
  PW_FRAMED_APPLETALK_NETWORK,
  PW_FRAMED_APPLETALK_ZONE,

  /* 
   * 40 - 59 are for accounting.  See RFC2139 for more information.
   */

  /* 40 */
  PW_ACCT_STATUS_TYPE,
  PW_ACCT_DELAY_TIME,
  PW_ACCT_INPUT_OCTETS,
  PW_ACCT_OUTPUT_OCTETS,
  PW_ACCT_SESSION_ID,
  PW_ACCT_AUTHENTIC,
  PW_ACCT_SESSION_TIME,
  PW_ACCT_INPUT_PACKETS,
  PW_ACCT_OUTPUT_PACKETS,
  PW_ACCT_TERMINATE_CAUSE,

  /* 50 */
  PW_ACCT_MULTI_SESSION_ID,
  PW_ACCT_LINK_COUNT,
  PW_ACCT_INPUT_GIGAWORDS,	/* radius-ext */
  PW_ACCT_OUTPUT_GIGAWORDS,	/* radius-ext */
  PW_ATTRIBUTE_54,		/* radius-ext (unused) */
  PW_EVENT_TIMESTAMP,		/* radius-ext */
  
  /* 60 */
  PW_CHAP_CHALLENGE = 60,
  PW_NAS_PORT_TYPE,
  PW_PORT_LIMIT,
  PW_LOGIN_LAT_PORT,

  /* Tunnel protocol */
  PW_TUNNEL_TYPE,
  PW_TUNNEL_MEDIUM_TYPE,
  PW_TUNNEL_CLIENT_ENDPOINT,
  PW_TUNNEL_SERVER_ENDPOINT,
  PW_ACCT_TUNNEL_CONNECTION,
  PW_TUNNEL_PASSWORD,

  /* 70 */
  PW_ARAP_PASSWORD,		/* radius-ext */
  PW_ARAP_FEATURES,		/* radius-ext */
  PW_ARAP_ZONE_ACCESS,		/* radius-ext */
  PW_ARAP_SECURITY,		/* radius-ext */
  PW_ARAP_SECURITY_DATA,	/* radius-ext */
  PW_PASSWORD_RETRY,		/* radius-ext */
  PW_PROMPT,			/* radius-ext */
  PW_CONNECT_INFO,		/* radius-ext */
  PW_CONFIGURATION_TOKEN,	/* radius-ext */
  PW_EAP_MESSAGE,		/* radius-ext */
  PW_SIGNATURE,			/* radius-ext */
  
  /* 80 */
  PW_TUNNEL_PRIVATE_GROUP_ID = 81,
  PW_TUNNEL_ASSIGNMENT_ID,
  PW_TUNNEL_PREFERENCE,
  PW_ARAP_CHALLENGE_RESPONSE,	/* radius-ext */
  PW_ACCT_INTERVAL_TIME,	/* radius-ext */

};

/*
 * Definitions for backwards compatibility.  I prefer to take the names
 * Foo-Bar from the RFC, and convert them to PW_FOO_BAR, instead of using
 * something else.
 */
#define PW_AUTHENTICATION_REQUEST	PW_ACCESS_REQUEST
#define PW_AUTHENTICATION_ACK		PW_ACCESS_ACK
#define PW_AUTHENTICATION_REJECT	PW_ACCESS_REJECT
#define PW_PASSWORD			PW_USER_PASSWORD
#define PW_NAS_PORT_ID			PW_NAS_PORT
#define PW_VENDOR			PW_VENDOR_SPECIFIC


/*      Non-protocol attributes.  Not in any RFC. */
enum {
  PW_HUNTGROUP_NAME = 221,

  PW_AUTHTYPE = 1000,		/* Authentication type (local, system, etc) */
  PW_MENU,			/* Livingston style menus */
  PW_TERMINATION_MENU,
  PW_PREFIX,			/* User-Name prefix */
  PW_SUFFIX,			/* User-Name suffix */
  PW_GROUP,			/* User is a member of this group */
  PW_CRYPT_PASSWORD,		/* raw password from 'crypt' */
  PW_CONNECT_RATE,		/* the format for this sucks */

  PW_USER_CATEGORY = 1029,	/* ??? */
  PW_GROUP_NAME,		/* deprecated: use Group */

  PW_SIMULTANEOUS_USE = 1034,
  PW_STRIP_USERNAME,		/* strip the users' name from bob.ppp to bob */
  PW_FALL_THROUGH,		/* continure processing requests */
  PW_ADD_PORT_TO_IP_ADDRESS,
  PW_EXEC_PROGRAM,
  PW_EXEC_PROGRAM_WAIT,
  PW_HINT,
  PW_SERVER_CONFIG,

  PW_NT_DOMAIN = 1100,		/* Windows NT authentication domain */
  PW_AUTHENTICATION_PROGRAM,	/* external authentication program */
  PW_LOGIN_DAY,			/* permitted login days */
  PW_LOGIN_HOUR,		/* permitted login hours */
  PW_HAS_VALID_SHELL,		/* check /etc/shells for a valid shell */
  PW_NAS_PORT_RANGE,		/* NAS port range */
};

/*
 *  Values for the above attributes
 */

/*	PW_SERVICE_TYPE values.  RFC2138 p.27 */

enum {
  PW_LOGIN_USER = 1,		/* connects to a host */
  PW_FRAMED_USER,		/* use something like PPP or SLIP*/
  PW_CALLBACK_LOGIN_USER,	/* call them back and login */
  PW_CALLBACK_FRAMED_USER,	/* call them back and start PPP or SLIP */
  PW_OUTBOUND_USER,		/* connect to an outbound device */
  PW_ADMINISTRATIVE_USER,	/* allow them to administer the NAS */
  PW_NAS_PROMPT_USER,		/* give them a non-privileged NAS prompt */
  PW_AUTHENTICATE_ONLY,		/* we only want to authenticate */
  PW_CALLBACK_NAS_PROMPT,	/* call back, and give non-priv. NAS prompt */
};

/*	PW_FRAMED_PROTOCOL values.  RFC2138 p.28 */

enum {
  PW_PPP = 1,			/* Point to Point Protocol */
  PW_SLIP,			/* Serial Line IP */
  PW_APPLETALK,			/* Appletalk Remote Access Protocol (ARAP) */
  PW_GANDALF,			/* proprietary Singlelink/Multilink (SL/ML) */
  PW_XYLOGICS,			/* proprietary IPX/SLIP */
};

/*	PW_FRAMED_IP_ADDRESS values.  RFC2138 p.29 */

enum {
  PW_ASSIGNED = 0xfffffffe,	/* Assigned from a pool kept by the NAS */
  PW_NEGOTIATED,		/* NAS allows the user to select an address */
};

/*	PW_FRAMED_ROUTING values.  RFC2138 p.30 */

enum {
  PW_NONE = 0,			/* don't do anything */
  PW_BROADCAST,			/* send routing packets */
  PW_LISTEN,			/* listen for routing packets */
  PW_BROADCAST_LISTEN,		/* send and listen */
};

/*	PW_FRAMED_COMPRESSION values.  RFC2138 p.33 */

enum {
  PW_VAN_JACOBSEN_TCP_IP = 1,	/* see RFC1144 for details */
  PW_IPX_HEADER,		/* IPX header compression */
};
/*	PW_LOGIN_SERVICE values.  RFC2138 p.35 */

enum {
  PW_TELNET = 0,		/* use basic telnet access */
  PW_RLOGIN,			/* use rlogin: NOT RECOMMENDED FOR SECURITY! */
  PW_TCP_CLEAR,			/* ? */
  PW_PORTMASTER,		/* proprietary */
  PW_LAT,			/* ? */
};

/*      PW_VENDOR (Vendor-Specific) RFC2138 p.43
 *
 * See      RFC 1700 pp.134-161
 * or       ftp://ftp.isi.edu/in-notes/iana/assignments/enterprise-numbers
 *
 * for a complete list of Vendor IDs. (about 250k worth)
 *
 * Note that these are *not* the same as the Ethernet vendor codes.
 *
 * We don't list all of the vendor codes here, just some of the more useful
 * ones.
 *
 * Documentation for Cisco TACACS+ attributes inside of a Vendor-Specific
 * RADIUS attribute can be found at:
 *
 * http://www.cisco.com/univercd/cc/td/doc/product/software/ios112/112cg_cr/2cbook/2caaa.htm#xtocid2997910
 *
 */
#define PW_VENDOR_PROTEON	1
#define PW_VENDOR_IBM		2
#define PW_VENDOR_CISCO		9
#define PW_VENDOR_XYLOGICS	15
#define PW_VENDOR_NOVELL	23
#define PW_VENDOR_XYPLEX	33
#define PW_VENDOR_SUN		42
#define PW_VENDOR_3COM		43
#define PW_VENDOR_MERIT		61
#define PW_VENDOR_APPLE		63
#define PW_VENDOR_GANDALF	64
#define PW_VENDOR_MICROCOM	102
#define PW_VENDOR_TELEBIT	117
#define PW_VENDOR_TEKTRONIX	128
#define PW_VENDOR_BANYAN	130
#define PW_VENDOR_MOTOROLA	161
#define PW_VENDOR_SHIVA		166
#define PW_VENDOR_BAY		1584
#define PW_VENDOR_CRYPTOCARD	1609

/*      PW_TERMINATION_ACTION values.  RFC2138 p.46 */
enum {
  PW_DEFAULT_TERMINATE = 0,	/* when access is terminated, do the default */
  PW_RADIUS_REQUEST,		/* maybe send new Access-Request */
};

/*	PW_NAS_PORT_TYPE values.  RFC2138 p.56 */
enum {
  PW_NAS_PORT_ASYNC = 0,	/* asynchronous */
  PW_NAS_PORT_SYNC,		/* synchronous */
  PW_NAS_PORT_ISDN,		/* ISDN Sync */
  PW_NAS_PORT_ISDN_V120,	/* ISDN Async v.120 */
  PW_NAS_PORT_ISDN_V110,	/* ISDN Async v.110 */
  PW_NAS_PORT_VIRTUAL,		/* Virtual */
};

/*	PW_ACCT_STATUS_TYPE values.  RFC2139 p.11 */

enum {
  PW_STATUS_START = 1,		/* start accounting */
  PW_STATUS_STOP,		/* stop accounting */
  PW_STATUS_INTERIM_UPDATE,    	/* draft-ietf-radius-ext-02 */
  PW_STATUS_MODEM_START,	/* ??? */
  PW_STATUS_MODEM_STOP,		/* ??? */
  PW_STATUS_CANCEL,		/* ??? */
  PW_STATUS_ACCOUNTING_ON,	/* turn accounting on */
  PW_STATUS_ACCOUNTING_OFF	/* turn accounting off */
};
#define PW_STATUS_ALIVE PW_STATUS_INTERIM_UPDATE

/*      PW_ACCT_AUTHENTIC values.  RFC2139 p.15 */
enum {
  PW_AUTH_NONE = 0,		/* not in RFC2139 */
  PW_AUTH_RADIUS,		/* authenticated by radiusd */
  PW_AUTH_LOCAL,		/* authenticated by the NAS */
  PW_AUTH_REMOTE,		/* used a remote authentication protocol (?) */
};

/*      Internal authentication types.  Not in any RFC. */
enum {
  PW_AUTHTYPE_LOCAL = 0,	/* use NAS authentication? */
  PW_AUTHTYPE_SYSTEM,		/* use system authentication functions */
  PW_AUTHTYPE_SECURID,		/* SecureID yucky stuff */
  PW_AUTHTYPE_CRYPT,		/* encrypted (copied from /etc/passwd) */
  PW_AUTHTYPE_REJECT,		/* reject the user */
  PW_AUTHTYPE_EXTERNAL,		/* external authentication program */
  
  PW_AUTHTYPE_CRYPTOCARD = 10,	/* CRYPTOCard authentication */
  PW_AUTHTYPE_PROXY_REQUEST,	/* Proxy the request to a server */

  PW_AUTHTYPE_SQL = 252,	/* from PostgreSQL patches to Cistron */
  PW_AUTHTYPE_PAM,		/* Pluggable Authentication Module */
  PW_AUTHTYPE_ACCEPT,		/* accept everyone */
  PW_AUTHTYPE_NONE,		/* don't use any authentication type (?) */
};


/*      PW_ACCT_TERMINATE_CAUSE values.  RFC2139 p.18 */
enum {
  PW_USER_REQUEST = 1,		/* user requested the session termination */
  PW_LOST_CARRIER,		/* DCD was dropped on the port */
  PW_LOST_SERVICE,		/* service can no longer be provided */
  PW_ACCT_IDLE_TIMEOUT,		/* idle timer expired */
  PW_ACCT_SESSION_TIMEOUT,	/* maximum session length timer expired */
  PW_ADMIN_RESET,		/* administrative reset */
  PW_ADMIN_REBOOT,		/* administrator is rebooting the NAS */
  PW_PORT_ERROR,		/* NAS detected an error on the port */
  PW_NAS_ERROR,			/* NAS detected some other error */
  PW_NAS_REQUEST,		/* NAS ended it for some non-error condition */
  PW_NAS_REBOOT,		/* NAS ended in order to reboot ("crash") */
  PW_PORT_UNNEEDED,		/* NAS decided the port was no longer needed */
  PW_PORT_PREEMPTED,		/* NAS had a higher priority user */
  PW_PORT_SUSPENDED,		/* suspended a virtual session */
  PW_SERVICE_UNAVAILABLE,	/* NAS unable to provide requested service */
  PW_CALLBACK,			/* NAS will call you back */
  PW_USER_ERROR,		/* Input from user is in error: goodbye */
  PW_HOST_REQUEST,		/* Login-Host terminated session normally */
};

/*	CRYPTOCard VSA definitions  */
enum {
  PW_CRYPTOCARD_AWARE = 1,	/* the client is aware of cryptocard */
  PW_CRYPTOCARD_CHALLENGE,	/* the DES challenge, as a string */
  PW_CRYPTOCARD_CHALLENGE_MESSAGE = 256	/* CRYPTOCard challenge message */
};

/* Allowed packet entries.  From the end of the relevent RFC's */
#define PW_ALLOW_ZERO		0
#define PW_ALLOW_ONE		1
#define PW_ALLOW_ZERO_PLUS	2
#define PW_ALLOW_ZERO_ONE	3

/* Windows specific definitions */
#ifdef WIN32
typedef unsigned int u_int;
typedef unsigned char u_char;
typedef unsigned short u_short;
#endif WIN32

/***********************************************************************
 * Generic Radius data structures, for packets and attributes.
 *
 * For portability, all elements of these structs MUST be u_char
 * However, the 'u_short length' bit seems to work, so we'll leave it.
 ***********************************************************************/
typedef struct pw_auth_hdr {
	u_char		code;
	u_char		id;
	u_short		length;	/* always kept in network byte order! */
	u_char		vector[AUTH_VECTOR_LEN];
	u_char		data[2];
} AUTH_HDR;

#define AUTH_HDR_LEN	(AUTH_VECTOR_LEN + sizeof(u_char)*2 + sizeof(u_short))

/* RADIUS attribute data structure */
typedef struct RADIUS_ATTR {
	u_char attribute;
	u_char length;
	u_char data[1];
} RADIUS_ATTR;
#define ATTR_HDR_LEN 2
#define MAX_ATTR_SIZE 253

/* Vendor specific attribute from RFC2138 */
typedef struct VENDOR_ATTR {
	u_char attribute;
	u_char length;
	u_char vendor_id[4];
	u_char data[1];
} VENDOR_ATTR;
#define VSA_HDR_LEN (1+1+4)

/* Vendor specific attribute USR definitions */
typedef struct NMC_ATTR {
	u_char attribute;
	u_char length;
	u_char vendor_id[4];
	u_char nmc[4];
	u_char data[1];
} NMC_ATTR;
#define NMC_HDR_LEN (1+1+4+4)

#endif RADIUS_H
