/*
 * auth.c	User authentication.
 *
 *
 * Version:	@(#)auth.c  1.84  05-May-1999  miquels@cistron.nl
 *
 */
char auth_sccsid[] =
"@(#)auth.c	1.84 Copyright 1998-1999 Cistron Internet Services B.V.";

#include	<sys/types.h>
#include	<sys/time.h>

#include	<stdio.h>
#include	<stdlib.h>
#include	<pwd.h>
#include	<time.h>
#include	<unistd.h>
#include	<signal.h>
#include	<errno.h>
#include	<sys/wait.h>

#if !defined(NOSHADOW)
#include	<shadow.h>
#endif /* !NOSHADOW */

#if defined(__GLIBC__)
#  include <crypt.h>
#endif

#ifdef OSFC2
#  include	<sys/security.h>
#  include	<prot.h>
#endif

#include	"radiusd.h"

#include	"cache.h"

#if !defined(__linux__) && !defined(__GLIBC__)
  extern char *crypt();
#endif

/*
 *	Tests to see if the users password has expired.
 *
 *	Return: Number of days before expiration if a warning is required
 *		otherwise 0 for success and -1 for failure.
 */
static int pw_expired(UINT4 exptime)
{
	struct timeval	tp;
	struct timezone	tzp;
	UINT4		exp_remain;
	int		exp_remain_int;

	gettimeofday(&tp, &tzp);
	if (tp.tv_sec > exptime)
		return -1;

	if (warning_seconds != 0) {
		if (tp.tv_sec > exptime - warning_seconds) {
			exp_remain = exptime - tp.tv_sec;
			exp_remain /= (UINT4)SECONDS_PER_DAY;
			exp_remain_int = exp_remain;
			return exp_remain_int;
		}
	}
	return 0;
}


/*
 *	Check if account has expired, and if user may login now.
 */
static int check_expiration(VALUE_PAIR *check_item, char *umsg, char **user_msg)
{
	int result;
	int retval;

	result = 0;
	while (result == 0 && check_item != (VALUE_PAIR *)NULL) {

		/*
		 *	Check expiration date if we are doing password aging.
		 */
		if (check_item->attribute == PW_EXPIRATION) {
			/*
			 *	Has this user's password expired
			 */
			retval = pw_expired(check_item->lvalue);
			if (retval < 0) {
				result = -1;
				*user_msg = "Password Has Expired\r\n";
				break;
			} else {
				if (retval > 0) {
					sprintf(umsg,
					  "Password Will Expire in %d Days\r\n",
					  retval);
					*user_msg = umsg;
				}
			}
		}
		check_item = check_item->next;
	}
	return result;
}


/*
 *	Check the users password against the standard UNIX
 *	password table.
 */
static int unix_pass(char *name, char *passwd)
{
	struct passwd	*pwd;
	char		*encpw;
	char		*encrypted_pass;
	int		ret;
#if !defined(NOSHADOW)
#if defined(M_UNIX)
	struct passwd	*spwd;
#else
	struct spwd	*spwd;
#endif
#endif /* !NOSHADOW */
#ifdef OSFC2
	struct pr_passwd *pr_pw;
#endif

	if (cache_passwd && (ret = H_unix_pass(name, passwd)) != -2)
		return ret;

#ifdef OSFC2
	if ((pr_pw = getprpwnam(name)) == NULL)
		return -1;
	encrypted_pass = pr_pw->ufld.fd_encrypt;
#else /* OSFC2 */
	/*
	 *	Get encrypted password from password file
	 */
	if ((pwd = rad_getpwnam(name)) == NULL) {
		return -1;
	}
	encrypted_pass = pwd->pw_passwd;
#endif /* OSFC2 */

#if !defined(NOSHADOW)
	/*
	 *      See if there is a shadow password.
	 */
	if ((spwd = getspnam(name)) != NULL)
#if defined(M_UNIX)
		encrypted_pass = spwd->pw_passwd;
#else
		encrypted_pass = spwd->sp_pwdp;
#endif	/* M_UNIX */
#endif	/* !NOSHADOW */
#
#ifdef DENY_SHELL
	/*
	 *	Undocumented temporary compatibility for iphil.NET
	 *	Users with a certain shell are always denied access.
	 */
	if (strcmp(pwd->pw_shell, DENY_SHELL) == 0) {
		log(L_AUTH, "unix_pass: [%s]: invalid shell", name);
		return -1;
	}
#endif

#if !defined(NOSHADOW) && !defined(M_UNIX)
	/*
	 *      Check if password has expired.
	 */
	if (spwd && spwd->sp_expire > 0 &&
	    (time(NULL) / 86400) > spwd->sp_expire) {
		log(L_AUTH, "unix_pass: [%s]: password has expired", name);
		return -1;
	}
#endif

#if defined(__FreeBSD__) || defined(bsdi) || defined(_PWF_EXPIRE)
	/*
	 *      Check if password has expired.
	 */
	if (pwd->pw_expire > 0 && time(NULL) > pwd->pw_expire) {
		log(L_AUTH, "unix_pass: [%s]: password has expired", name);
		return -1;
	}
#endif

#ifdef OSFC2
	/*
	 *	Check if account is locked.
	 */
	if (pr_pw->uflg.fg_lock!=1) {
		log(L_AUTH, "unix_pass: [%s]: account locked", name);
		return -1;
	}
#endif /* OSFC2 */

	/*
	 *	We might have a passwordless account.
	 */
	if (encrypted_pass[0] == 0) return 0;

	/*
	 *	Check encrypted password.
	 */
	encpw = crypt(passwd, encrypted_pass);
	if (strcmp(encpw, encrypted_pass))
		return -1;

#ifdef ETC_SHELL
        {
          char *shell;

          while ((shell = getusershell()) != NULL) {
            if (strcmp(shell, pwd->pw_shell) == 0) ||
		strcpy(shell, "/RADIUSD/ANY/SHELL") == 0) {
              endusershell();
              return 0;
            }
          }

          endusershell();
          return -1;
        }
#endif /* ETC_SHELL */

	return 0;
}


/*
 *	Check password.
 *
 *	Returns:	0  OK
 *			-1 Password fail
 *			-2 Rejected
 *			1  End check & return.
 */
static int rad_check_password(AUTH_REQ *authreq, int activefd, VALUE_PAIR *check_item, VALUE_PAIR *namepair,
			 char *pw_digest, char **user_msg, char *userpass , char *dbpass) {

	char		string[AUTH_STRING_LEN];
	char		name[AUTH_STRING_LEN];
	VALUE_PAIR	*auth_type_pair;
	VALUE_PAIR	*password_pair;
	VALUE_PAIR	*auth_item;
	VALUE_PAIR	*presuf_item;
	VALUE_PAIR	*tmp;
	int		auth_type = -1;
	int		i;
	int		strip_username;
	int		result;
	char		*ptr;
	/*
	 *	cjd 19980706 --
	 *	pampair contains the pair of PAM_AUTH_ATTR
	 *	pamauth is the actual string
	 */
        VALUE_PAIR      *pampair;
	char		*pamauth = NULL;

	result = 0;
	userpass[0] = 0;
	string[0] = 0;

	/*
	 *	Look for matching check items. We skip the whole lot
	 *	if the authentication type is PW_AUTHTYPE_ACCEPT or
	 *	PW_AUTHTYPE_REJECT.
	 */
	if ((auth_type_pair = pairfind(check_item, PW_AUTHTYPE)) != NULL)
		auth_type = auth_type_pair->lvalue;

	if (auth_type == PW_AUTHTYPE_ACCEPT)
		return 0;

	if (auth_type == PW_AUTHTYPE_REJECT) {
		*user_msg = NULL;
		return -2;
	}

	/*
	 *	cjd 19980706 --
	 *	Fish out the the PAM_AUTH_ATTR info for this match and
	 *	get the string for pamauth.
	 *	Pamauth is passed to pam_pass so we can have selective
	 *	pam configuration.
	 */
        if ((pampair = pairfind(check_item, PAM_AUTH_ATTR)) != NULL) {
		pamauth = pampair->strvalue;
        }

	/*
	 *	Find the password sent by the user. It SHOULD be there,
	 *	if it's not authentication fails.
	 *
	 *	FIXME: add MS-CHAP support ?
	 */
	if ((auth_item = pairfind(authreq->request, PW_CHAP_PASSWORD)) == NULL)
		auth_item = pairfind(authreq->request, PW_PASSWORD);
	if (auth_item == NULL)
		return -1;

	/*
	 *	Find the password from the users file.
	 */
	if ((password_pair = pairfind(check_item, PW_CRYPT_PASSWORD)) != NULL)
		auth_type = PW_AUTHTYPE_CRYPT;
	else
		password_pair = pairfind(check_item, PW_PASSWORD);

	/*
	 *	See if there was a Prefix or Suffix included.
         *      Note: sizeof(name) == sizeof(namepair->strvalue)
         *      so strcpy is safe.
	 */
	strip_username = 1;
	if ((presuf_item = pairfind(check_item, PW_PREFIX)) == NULL)
		presuf_item = pairfind(check_item, PW_SUFFIX);
	if (presuf_item) {
		tmp = pairfind(check_item, PW_STRIP_USERNAME);
		if (tmp != NULL && tmp->lvalue == 0)
			strip_username = 0;
		i = presufcmp(presuf_item, namepair->strvalue, name, sizeof(name));
		if (i != 0 || strip_username == 0)
			strcpy(name, namepair->strvalue);
	} else
		strcpy(name, namepair->strvalue);


	/*
	 *	For backward compatibility, we check the
	 *	password to see if it is the magic value
	 *	UNIX if auth_type was not set.
	 */
	if (auth_type < 0) {
		if (password_pair && !strcmp(password_pair->strvalue, "UNIX"))
			auth_type = PW_AUTHTYPE_SYSTEM;
		else if(password_pair && !strcmp(password_pair->strvalue,"PAM"))
			auth_type = PW_AUTHTYPE_PAM;
		else
			auth_type = PW_AUTHTYPE_LOCAL;
	}

	/*
	 *	Decrypt the password.
	 */
	if (auth_item != NULL && auth_item->attribute == PW_PASSWORD) {
		memcpy(string, auth_item->strvalue, AUTH_PASS_LEN);
		for(i = 0;i < AUTH_PASS_LEN;i++) {
			string[i] ^= pw_digest[i];
		}
		string[AUTH_PASS_LEN] = '\0';
		strcpy(userpass, string);
	}


#if 0 /* DEBUG */
	printf("auth_type=%d, string=%s, namepair=%s, password_pair=%s\n",
		auth_type, string, name,
		password_pair ? password_pair->strvalue : "");
#endif

	switch(auth_type) {
		case PW_AUTHTYPE_SYSTEM:
			DEBUG2("  auth: System");
			/*
			 *	Check the password against /etc/passwd.
			 */
			if (unix_pass(name, string) != 0)
				result = -1;
			break;
		case PW_AUTHTYPE_PAM:
#ifdef PAM
			DEBUG2("  auth: Pam");
			/*
			 *	Use the PAM database.
			 *
			 *	cjd 19980706 --
			 *	Use what we found for pamauth or set it to
			 *	the default "radius" and then jump into
			 *	pam_pass with the extra info.
			 */
			pamauth = pamauth ? pamauth : PAM_DEFAULT_TYPE;
			if (pam_pass(name, string, pamauth) != 0)
				result = -1;
#else
			log(L_ERR, "%s: PAM authentication not available",
                               name);
			result = -1;
#endif
			break;
		case PW_AUTHTYPE_CRYPT:
			DEBUG2("  auth: Crypt");
			if (password_pair == NULL) {
				result = string[0] ? -1 : 0;
				break;
			}
			if (strcmp(password_pair->strvalue,
			    crypt(string, password_pair->strvalue)) != 0)
					result = -1;
			break;
		case PW_AUTHTYPE_LOCAL:
			DEBUG2("  auth: Local");
			/*
			 *	Local password is just plain text.
	 		 */
			if (auth_item->attribute != PW_CHAP_PASSWORD) {
				/*
				 *	Plain text password.
				 */
				snprintf(dbpass, 128, " Password should be '%s'", password_pair->strvalue);
				if (password_pair == NULL ||
				    strcmp(password_pair->strvalue, string)!=0)
					result = -1;
				break;
			}

			/*
			 *	CHAP - calculate MD5 sum over CHAP-ID,
			 *	plain-text password and the Chap-Challenge.
			 *	Compare to Chap-Response (strvalue + 1).
			 *
			 *	FIXME: might not work with Ascend because
			 *	we use vp->length, and Ascend gear likes
			 *	to send an extra '\0' in the string!
			 */
			strcpy(string, "{chap-password}");
			if (password_pair == NULL) {
				result= -1;
				break;
			}
			i = 0;
			ptr = string;
			*ptr++ = *auth_item->strvalue;
			i++;
			memcpy(ptr, password_pair->strvalue,
				password_pair->length);
			ptr += password_pair->length;
			i += password_pair->length;
			/*
			 *	Use Chap-Challenge pair if present,
			 *	Request-Authenticator otherwise.
			 */
			if ((tmp = pairfind(authreq->request,
			    PW_CHAP_CHALLENGE)) != NULL) {
				memcpy(ptr, tmp->strvalue, tmp->length);
				i += tmp->length;
			} else {
				memcpy(ptr, authreq->vector, AUTH_VECTOR_LEN);
				i += AUTH_VECTOR_LEN;
			}
			md5_calc(pw_digest, string, i);

			/*
			 *	Compare them
			 */
			if (memcmp(pw_digest, auth_item->strvalue + 1,
					CHAP_VALUE_LENGTH) != 0)
				result = -1;
			else
				strcpy(userpass, password_pair->strvalue);
			break;
		default:
			result = -1;
			break;
	}

	if (result < 0)
		*user_msg = NULL;

	return result;
}

/*
 *	Initial step of authentication.
 *	Find username, calculate MD5 digest, and
 *	process the hints and huntgroups file.
 */
int rad_auth_init(AUTH_REQ *authreq, int activefd) {
	VALUE_PAIR	*namepair;
	char		pw_digest[16];

	/*
	 *	Get the username from the request
	 */
	namepair = pairfind(authreq->request, PW_USER_NAME);

	if ((namepair == (VALUE_PAIR *)NULL) || 
	   (strlen(namepair->strvalue) <= 0)) {
		log(L_ERR, "No username: [] (from nas %s)",
			nas_name2(authreq));
		pairfree(authreq->request);
		memset(authreq, 0, sizeof(AUTH_REQ));
		free(authreq);
		return -1;
	}

	strncpy(authreq->username, namepair->strvalue,
		sizeof(authreq->username));
	authreq->username[sizeof(authreq->username) - 1] = 0;

	/*
	 *	Verify the client and Calculate the MD5 Password Digest
	 */
	if (calc_digest(pw_digest, authreq) != 0) {
		/*
		 *	We dont respond when this fails
		 */
		log(L_ERR, "Authenticate: from client %s - Security Breach: %s",
			client_name(authreq->ipaddr), namepair->strvalue);
		authfree(authreq);
		return -1;
	}

	/*
	 *	Add any specific attributes for this username.
	 */
	hints_setup(authreq->request);

/* FIXME: find out what this logs */
/*	if (log_auth_detail)
		rad_accounting_orig(authreq, -1, "detail.auth");
*/

	/*
	 *	See if the user has access to this huntgroup.
	 */
	if (!huntgroup_access(authreq->request)) {
		log(L_AUTH, "No huntgroup access: [%s] (%s)",
			namepair->strvalue, nas_name2(authreq));
		rad_send_reply(PW_AUTHENTICATION_REJECT, authreq,
			authreq->request, NULL, activefd);
		authfree(authreq);
		return -1;
	}

	return 0;
}

/*
 *	Process and reply to an authentication request
 */
int rad_authenticate(AUTH_REQ *authreq, int activefd, SQLSOCK *socket) {
	VALUE_PAIR	*namepair;
	VALUE_PAIR	*check_item;
	VALUE_PAIR	*reply_item;
	VALUE_PAIR	*auth_item;
	VALUE_PAIR	*user_check;
	VALUE_PAIR	*user_reply;
	int		result, r;
	char		pw_digest[16];
	char		userpass[AUTH_STRING_LEN];
	char		umsg[AUTH_STRING_LEN];
	char		*user_msg;
	char		*ptr;
	char		*exec_program;
	int		exec_wait;
	int		seen_callback_id;
	char		string[AUTH_STRING_LEN];
	char		dbpass[128];
	int 		i;

	user_check = NULL;
	user_reply = NULL;
	user_msg = NULL;
	userpass[0] = 0;
	dbpass[0] = 0;

	/*
	 *	Get the username from the request.
	 *	All checking has been done by rad_auth_init().
	 */
	namepair = pairfind(authreq->request, PW_USER_NAME);

	/*
	 *	FIXME: we calculate the digest twice ...
	 *	once here and once in rad_auth_init()
	 */
	calc_digest(pw_digest, authreq);

	/*
	 *	If this request got proxied to another server, we need
	 *	to add an initial Auth-Type: Auth-Accept for success,
	 *	Auth-Reject for fail. We also need to add the reply
	 *	pairs from the server to the initial reply.
	 */
	if (authreq->server_code == PW_AUTHENTICATION_REJECT ||
	    authreq->server_code == PW_AUTHENTICATION_ACK) {
		if ((user_check = malloc(sizeof(VALUE_PAIR))) == NULL) {
			log(L_ERR|L_CONS, "no memory");
			exit(1);
		}
		memset(user_check, 0, sizeof(VALUE_PAIR));
		user_check->attribute = PW_AUTHTYPE;
		strcpy(user_check->name, "Auth-Type");
		user_check->type = PW_TYPE_INTEGER;
		user_check->length = 4;
	}
	if (authreq->server_code == PW_AUTHENTICATION_REJECT)
		user_check->lvalue = PW_AUTHTYPE_REJECT;
	if (authreq->server_code == PW_AUTHENTICATION_ACK)
		user_check->lvalue = PW_AUTHTYPE_ACCEPT;

	if (authreq->server_reply)
		user_reply = authreq->server_reply;


	if (authreq->server_code) {

		VALUE_PAIR	*realm_check = NULL;

		auth_item = pairfind(authreq->request, PW_PASSWORD);
		/*
		 *	Decrypt the password.
		 */
		if (auth_item != NULL) {
			memcpy(string, auth_item->strvalue, AUTH_PASS_LEN);
			for(i = 0;i < AUTH_PASS_LEN;i++)
				string[i] ^= pw_digest[i];
			string[AUTH_PASS_LEN] = '\0';
			strcpy(userpass, string);
		}
		sql_getvpdata(socket, sql->config->sql_groupcheck_table, &realm_check, authreq->realm, PW_VP_REALMDATA);
		sql_getvpdata(socket, sql->config->sql_groupreply_table, &user_reply, authreq->realm, PW_VP_REALMDATA);

		if (authreq->server_code == PW_AUTHENTICATION_ACK) {
			result = 0;

			if((result = check_expiration(realm_check, umsg, &user_msg)) < 0) {
					rad_send_reply(PW_AUTHENTICATION_REJECT, authreq, user_reply, user_msg, activefd); 
					log(L_AUTH, "Realm Expired: [%s] (%s)", authreq->realm, auth_name(authreq, 1));
					return -1;
			}

			if (result >= 0 && (check_item = pairfind(realm_check, PW_SIMULTANEOUS_USE)) != NULL) {

				SQL_ROW		row;
				char		querystr[256];
				int		num = 0;

				sprintf(querystr, "SELECT COUNT(*) FROM radacct WHERE Realm = '%s' and AcctStopTime = 0", authreq->realm);
				sql_select_query(socket, querystr);
				row = sql_fetch_row(socket);
				num = atol(row[0]);
				sql_finish_select_query(socket);
				if (num >= check_item->lvalue) {
					sprintf(umsg, "\r\nYour realm has reached it's limit of %d simultaneous logins\r\n\n", (int)check_item->lvalue);
					user_msg = umsg;
					result = -1;
					rad_send_reply(PW_AUTHENTICATION_REJECT, authreq, user_reply, user_msg, activefd);
					log(L_ERR, "Realm multiple logins: [%s] (%s) max. %d", authreq->realm, auth_name(authreq, 1), check_item->lvalue);
				}
			}

			if (result >=0 && (check_item = pairfind(realm_check, PW_TOTAL_TIME_LIMIT)) != NULL) {

				SQL_ROW		row;
				char		querystr[256];
				unsigned long	allowed;
				unsigned long	diff;
				unsigned long	used = 0;
	
				allowed = check_item->lvalue;
				sprintf(querystr, "SELECT SUM(AcctSessionTime) FROM radacct WHERE Realm = '%s'", authreq->realm);
				sql_select_query(socket, querystr);
				row = sql_fetch_row(socket);
				if (row[0] == NULL)
					used = 0;
				else
					used = atol(row[0]);
				sql_finish_select_query(socket);
				
				if (used >= allowed) {
					sprintf(umsg, "\r\nYour realm has reached it's total limit of %d seconds\r\n\n", (int)check_item->lvalue);
					user_msg = umsg;
					result = -1;
					rad_send_reply(PW_AUTHENTICATION_REJECT, authreq, user_reply, user_msg, activefd);
					log(L_ERR, "Realm usage total limit of %d seconds reached: [%s] (%s)", (int)check_item->lvalue, authreq->realm, auth_name(authreq, 1));
				}
			
				if (result >=0 && (reply_item = pairfind(user_reply, PW_SESSION_TIMEOUT)) != NULL) {
					diff = allowed - used;
					if (diff < reply_item->lvalue)
						reply_item->lvalue = diff;
				} else if (result >= 0) {
					if (!(reply_item = malloc(sizeof(VALUE_PAIR)))){
						log(L_ERR|L_CONS, "no memory");
						exit(1);
					}
					memset(reply_item, 0, sizeof(VALUE_PAIR));
					reply_item->attribute = PW_SESSION_TIMEOUT;
					strcpy(reply_item->name, "Session-Timeout");
					reply_item->type = PW_TYPE_INTEGER;
					reply_item->length = 4;
					reply_item->lvalue = allowed - used;
					pairadd(&user_reply, reply_item);
				}

			}

			if (result >=0 && (check_item = pairfind(realm_check, PW_MONTHLY_TIME_LIMIT)) != NULL) {

				SQL_ROW		row;
				char		querystr[256];
				unsigned long	allowed;
				unsigned long	diff;
				unsigned long	used = 0;
	
				allowed = check_item->lvalue;
				sprintf(querystr, "SELECT SUM(AcctSessionTime) FROM radacct WHERE Realm = '%s' and AcctStartTime >= CONCAT(YEAR(NOW()),'-',MONTH(NOW()),'-01 00:00:00') AND AcctStartTime <= ADDDATE(CONCAT(YEAR(NOW()),'-',MONTH(NOW()),'-01 00:00:00'), INTERVAL 1 MONTH)", authreq->realm);
				sql_select_query(socket, querystr);
				row = sql_fetch_row(socket);
				if (row[0] == NULL)
					used = 0;
				else
					used = atol(row[0]);
				sql_finish_select_query(socket);
				
				if (used >= allowed) {
					sprintf(umsg, "\r\nYour realm has reached it's limit of %d seconds\r\n\n", (int)check_item->lvalue);
					user_msg = umsg;
					result = -1;
					rad_send_reply(PW_AUTHENTICATION_REJECT, authreq, user_reply, user_msg, activefd);
					log(L_ERR, "Realm usage limit of %d seconds reached: [%s] (%s)", (int)check_item->lvalue, authreq->realm, auth_name(authreq, 1));
				}
			
				if (result >=0 && (reply_item = pairfind(user_reply, PW_SESSION_TIMEOUT)) != NULL) {
					diff = allowed - used;
					if (diff < reply_item->lvalue)
						reply_item->lvalue = diff;
				} else if (result >= 0) {
					if (!(reply_item = malloc(sizeof(VALUE_PAIR)))){
						log(L_ERR|L_CONS, "no memory");
						exit(1);
					}
					memset(reply_item, 0, sizeof(VALUE_PAIR));
					reply_item->attribute = PW_SESSION_TIMEOUT;
					strcpy(reply_item->name, "Session-Timeout");
					reply_item->type = PW_TYPE_INTEGER;
					reply_item->length = 4;
					reply_item->lvalue = allowed - used;
					pairadd(&user_reply, reply_item);
				}

			}
			/* Old attribute in hours format */
			if (result >=0 && (check_item = pairfind(realm_check, PW_MAX_HOURS)) != NULL) {

				SQL_ROW		row;
				char		querystr[256];
				unsigned long	allowed;
				unsigned long	diff;
				unsigned long	used = 0;
	
				allowed = check_item->lvalue * 3600;
				sprintf(querystr, "SELECT SUM(AcctSessionTime) FROM radacct WHERE Realm = '%s' and AcctStartTime >= CONCAT(YEAR(NOW()),'-',MONTH(NOW()),'-01 00:00:00') AND AcctStartTime <= ADDDATE(CONCAT(YEAR(NOW()),'-',MONTH(NOW()),'-01 00:00:00'), INTERVAL 1 MONTH)", authreq->realm);
				sql_select_query(socket, querystr);
				row = sql_fetch_row(socket);
				if (row[0] == NULL)
					used = 0;
				else
					used = atol(row[0]);
				sql_finish_select_query(socket);
				
				if (used >= allowed) {
					sprintf(umsg, "\r\nYour realm has reached it's limit of %d hours\r\n\n", (int)check_item->lvalue);
					user_msg = umsg;
					result = -1;
					rad_send_reply(PW_AUTHENTICATION_REJECT, authreq, user_reply, user_msg, activefd);
					log(L_ERR, "Realm usage limit of %d hours reached: [%s] (%s)", (int)check_item->lvalue, authreq->realm, auth_name(authreq, 1));
				}
			
				if (result >=0 && (reply_item = pairfind(user_reply, PW_SESSION_TIMEOUT)) != NULL) {
					diff = allowed - used;
					if (diff < reply_item->lvalue)
						reply_item->lvalue = diff;
				} else if (result >= 0) {
					if (!(reply_item = malloc(sizeof(VALUE_PAIR)))){
						log(L_ERR|L_CONS, "no memory");
						exit(1);
					}
					memset(reply_item, 0, sizeof(VALUE_PAIR));
					reply_item->attribute = PW_SESSION_TIMEOUT;
					strcpy(reply_item->name, "Session-Timeout");
					reply_item->type = PW_TYPE_INTEGER;
					reply_item->length = 4;
					reply_item->lvalue = allowed - used;
					pairadd(&user_reply, reply_item);
				}

			}

			if (result >= 0 &&
			   (check_item = pairfind(realm_check, PW_LOGIN_TIME)) != NULL) {
	
				/*
				 *	Authentication is OK. Now see if this
				 *	realm may login at this time of the day.
				 */
				r = timestr_match(check_item->strvalue, time(NULL));
				if (r < 0) {
					/*
					 *	User called outside allowed time interval.
					 */
					result = -1;
					user_msg =
					"You are calling outside your allowed timespan\r\n";
					rad_send_reply(PW_AUTHENTICATION_REJECT, authreq,
						user_reply, user_msg, activefd);
					log(L_ERR, "Realm Outside allowed timespan: [%s]"
						   " (from nas %s) time allowed: %s",
							authreq->realm,
							nas_name2(authreq),
							check_item->strvalue);
				} else if (r > 0) {
					/*
					 *	User is allowed, but set Session-Timeout.
					 */
					if ((reply_item = pairfind(user_reply,
					    PW_SESSION_TIMEOUT)) != NULL) {
						if (reply_item->lvalue > r)
							reply_item->lvalue = r;
					} else {
						if (!(reply_item = malloc(sizeof(VALUE_PAIR)))){
							log(L_ERR|L_CONS, "no memory");
							exit(1);
						}
						memset(reply_item, 0, sizeof(VALUE_PAIR));
						reply_item->attribute = PW_SESSION_TIMEOUT;
						strcpy(reply_item->name, "Session-Timeout");
						reply_item->type = PW_TYPE_INTEGER;
						reply_item->length = 4;
						reply_item->lvalue = r;
						pairadd(&user_reply, reply_item);
					}
				}
			}
			if (result < 0) {
				authfree(authreq);
				pairfree(realm_check);
				pairfree(user_reply);
				return 0;
			}

		} else if (authreq->server_code == PW_AUTHENTICATION_REJECT) {
			pairfree(realm_check);
			result = -1;
		}
	} else {
		/*
		 *	Get the user from the database
		 */
		if (user_find(namepair->strvalue, authreq->request,
		   &user_check, &user_reply, socket) != 0) {
			log(L_AUTH, "Invalid user: [%s] (%s)",
				namepair->strvalue, auth_name(authreq, 1));
			rad_send_reply(PW_AUTHENTICATION_REJECT, authreq,
				NULL, NULL, activefd);
			pairfree(user_check);
			pairfree(user_reply);
			authfree(authreq);
			return -1;
		}

		if (paircmp(authreq->request, user_check) != 0) {
			log(L_AUTH, "Check list does not match request list [%s] (%s)", namepair->strvalue, auth_name(authreq, 1));
			rad_send_reply(PW_AUTHENTICATION_REJECT, authreq, NULL, NULL, activefd);
			pairfree(user_check);
			pairfree(user_reply);
			authfree(authreq);
			return -1;
		}
			

		/*
		 *	Validate the user
		 */
		do {
			if ((result = check_expiration(user_check, umsg, &user_msg))<0) {
					rad_send_reply(PW_AUTHENTICATION_REJECT, authreq, user_reply, user_msg, activefd); 
					log(L_AUTH, "Account Expired: [%s] (%s)", namepair->strvalue, auth_name(authreq, 1));
					pairfree(user_check);
					pairfree(user_reply);
					authfree(authreq);
					return -1;
			}

			result = rad_check_password(authreq, activefd, user_check, namepair, pw_digest, &user_msg, userpass , dbpass);

			if (result > 0) {
				authfree(authreq);
				pairfree(user_check);
				pairfree(user_reply);
				return -1;
			}
			if (result == -2) {
				if ((reply_item = pairfind(user_reply, PW_REPLY_MESSAGE)) != NULL)
					user_msg = reply_item->strvalue;
			}
		} while(0);

	}

	if (result < 0) {
		/*
		 *	Failed to validate the user.
		 */
		rad_send_reply(PW_AUTHENTICATION_REJECT, authreq, user_reply, user_msg, activefd);
		if (log_auth)
			log(L_AUTH, "Login incorrect: [%s/%s]%s (%s)", namepair->strvalue, userpass, dbpass, auth_name(authreq, 1));
	}


	if (result >=0 && (check_item = pairfind(user_check, PW_TOTAL_TIME_LIMIT)) != NULL) {

		SQL_ROW		row;
		char		querystr[256];
		unsigned long	allowed = 0;
		unsigned long	used = 0;
		unsigned long	diff = 0;

		allowed = check_item->lvalue;
		sprintf(querystr, "SELECT SUM(AcctSessionTime) FROM radacct WHERE UserName = '%s'", namepair->strvalue);
		sql_select_query(socket, querystr);
		row = sql_fetch_row(socket);
		if (row[0] == NULL)
			used = 0;
		else
			used = atol(row[0]);
		sql_finish_select_query(socket);
		
		if (used >= allowed) {
			sprintf(umsg, "\r\nYour account has reached it total limit of %d seconds\r\n\n", (int)check_item->lvalue);
			user_msg = umsg;
			result = -1;
			rad_send_reply(PW_AUTHENTICATION_REJECT, authreq, user_reply, user_msg, activefd);
			log(L_ERR, "Total usage limit of %d seconds reached: [%s] (%s)", (int)check_item->lvalue, namepair->strvalue, auth_name(authreq, 1));
		}
			
		if (result >=0 && (reply_item = pairfind(user_reply, PW_SESSION_TIMEOUT)) != NULL) {
			diff = allowed - used;
			if (diff < reply_item->lvalue)
				reply_item->lvalue = diff;
		} else {
			if (!(reply_item = malloc(sizeof(VALUE_PAIR)))){
				log(L_ERR|L_CONS, "no memory");
				exit(1);
			}
			memset(reply_item, 0, sizeof(VALUE_PAIR));
			reply_item->attribute = PW_SESSION_TIMEOUT;
			strcpy(reply_item->name, "Session-Timeout");
			reply_item->type = PW_TYPE_INTEGER;
			reply_item->length = 4;
			reply_item->lvalue = allowed - used;
			pairadd(&user_reply, reply_item);
		}

	}

	if (result >=0 && (check_item = pairfind(user_check, PW_MONTHLY_TIME_LIMIT)) != NULL) {

		SQL_ROW		row;
		char		querystr[256];
		unsigned long	allowed = 0;
		unsigned long	used = 0;
		unsigned long	diff = 0;

		allowed = check_item->lvalue;
		sprintf(querystr, "SELECT SUM(AcctSessionTime) FROM radacct WHERE UserName = '%s' and AcctStartTime >= CONCAT(YEAR(NOW()),'-',MONTH(NOW()),'-01 00:00:00') AND AcctStartTime <= ADDDATE(CONCAT(YEAR(NOW()),'-',MONTH(NOW()),'-01 00:00:00'), INTERVAL 1 MONTH)", namepair->strvalue);
		sql_select_query(socket, querystr);
		row = sql_fetch_row(socket);
		if (row[0] == NULL)
			used = 0;
		else
			used = atol(row[0]);
		sql_finish_select_query(socket);

		if (used >= allowed) {
			sprintf(umsg, "\r\nYour account has reached it limit of %d seconds\r\n\n", (int)check_item->lvalue);
			user_msg = umsg;
			result = -1;
			rad_send_reply(PW_AUTHENTICATION_REJECT, authreq, user_reply, user_msg, activefd);
			log(L_ERR, "Usage limit of %d seconds reached: [%s] (%s)", (int)check_item->lvalue, namepair->strvalue, auth_name(authreq, 1));
		}
			
		if (result >=0 && (reply_item = pairfind(user_reply, PW_SESSION_TIMEOUT)) != NULL) {
			diff = allowed - used;
			if (diff < reply_item->lvalue)
				reply_item->lvalue = diff;
		} else {
			if (!(reply_item = malloc(sizeof(VALUE_PAIR)))){
				log(L_ERR|L_CONS, "no memory");
				exit(1);
			}
			memset(reply_item, 0, sizeof(VALUE_PAIR));
			reply_item->attribute = PW_SESSION_TIMEOUT;
			strcpy(reply_item->name, "Session-Timeout");
			reply_item->type = PW_TYPE_INTEGER;
			reply_item->length = 4;
			reply_item->lvalue = allowed - used;
			pairadd(&user_reply, reply_item);
		}

	}
	/* Old attribute in hours format */
	if (result >=0 && (check_item = pairfind(user_check, PW_MAX_HOURS)) != NULL) {

		SQL_ROW		row;
		char		querystr[256];
		unsigned long	allowed = 0;
		unsigned long	used = 0;
		unsigned long	diff = 0;

		allowed = check_item->lvalue * 3600;
		sprintf(querystr, "SELECT SUM(AcctSessionTime) FROM radacct WHERE UserName = '%s' and AcctStartTime >= CONCAT(YEAR(NOW()),'-',MONTH(NOW()),'-01 00:00:00') AND AcctStartTime <= ADDDATE(CONCAT(YEAR(NOW()),'-',MONTH(NOW()),'-01 00:00:00'), INTERVAL 1 MONTH)", namepair->strvalue);
		sql_select_query(socket, querystr);
		row = sql_fetch_row(socket);
		if (row[0] == NULL)
			used = 0;
		else
			used = atol(row[0]);
		sql_finish_select_query(socket);
		
		if (used >= allowed) {
			sprintf(umsg, "\r\nYour account has reached it limit of %d hours\r\n\n", (int)check_item->lvalue);
			user_msg = umsg;
			result = -1;
			rad_send_reply(PW_AUTHENTICATION_REJECT, authreq, user_reply, user_msg, activefd);
			log(L_ERR, "Usage limit of %d hours reached: [%s] (%s)", (int)check_item->lvalue, namepair->strvalue, auth_name(authreq, 1));
		}
			
		if (result >=0 && (reply_item = pairfind(user_reply, PW_SESSION_TIMEOUT)) != NULL) {
			diff = allowed - used;
			if (diff < reply_item->lvalue)
				reply_item->lvalue = diff;
		} else {
			if (!(reply_item = malloc(sizeof(VALUE_PAIR)))){
				log(L_ERR|L_CONS, "no memory");
				exit(1);
			}
			memset(reply_item, 0, sizeof(VALUE_PAIR));
			reply_item->attribute = PW_SESSION_TIMEOUT;
			strcpy(reply_item->name, "Session-Timeout");
			reply_item->type = PW_TYPE_INTEGER;
			reply_item->length = 4;
			reply_item->lvalue = allowed - used;
			pairadd(&user_reply, reply_item);
		}

	}


	if (result >= 0 && (check_item = pairfind(user_check, PW_ACTIVATION)) != NULL) {

		time_t		now;
		struct tm	*actdate;

		now = time(0);

		if (now < check_item->lvalue) {
			actdate = localtime(&check_item->lvalue);
			sprintf(umsg, "\r\nYour account is not yet activated\r\n\n");
			user_msg = umsg;
			result = -1;
			rad_send_reply(PW_AUTHENTICATION_REJECT, authreq, user_reply, user_msg, activefd);
			log(L_ERR, "Account not activated until %d-%d-%d: [%s] (%s)", actdate->tm_year+1900, actdate->tm_mon+1, actdate->tm_mday, namepair->strvalue, auth_name(authreq, 1));
		}
	}

	if (result >= 0 &&
	   (check_item = pairfind(user_check, PW_SIMULTANEOUS_USE)) != NULL) {
#if 0 /* DEBUG */
		VALUE_PAIR *tmp;
		char ipno[32];
#endif
		/*
		 *	User authenticated O.K. Now we have to check
		 *	for the Simultaneous-Use parameter.
		 */
		if ((r = sql_check_multi(socket, namepair->strvalue, authreq->request, check_item->lvalue)) != 0) {

			if (check_item->lvalue > 1) {
				sprintf(umsg, "\r\nYou are already logged in %d times  - access denied\r\n\n", (int)check_item->lvalue);
				user_msg = umsg;
			} else {
				user_msg = "\r\nYou are already logged in - access denied\r\n\n";
			}
			rad_send_reply(PW_AUTHENTICATION_REJECT, authreq, user_reply, user_msg, activefd);
		log(L_ERR, "Multiple logins: [%s] (%s) max. %d%s", namepair->strvalue, auth_name(authreq, 1),
				check_item->lvalue, r == 2 ? " [MPP attempt]" : "");
			result = -1;
#if 0 /* DEBUG */
			ipno[0] = 0;
			if ((tmp = pairfind(authreq->request, PW_FRAMED_IP_ADDRESS)) != NULL)
				ipaddr2str(ipno, tmp->lvalue);
			log(L_INFO, "User asked for IP address [%s]", ipno);
#endif
		}
	}

	if (result >= 0 &&
	   (check_item = pairfind(user_check, PW_LOGIN_TIME)) != NULL) {

		/*
		 *	Authentication is OK. Now see if this
		 *	user may login at this time of the day.
		 */
		r = timestr_match(check_item->strvalue, time(NULL));
		if (r < 0) {
			/*
			 *	User called outside allowed time interval.
			 */
			result = -1;
			user_msg =
			"You are calling outside your allowed timespan\r\n";
			rad_send_reply(PW_AUTHENTICATION_REJECT, authreq, user_reply, user_msg, activefd);
			log(L_ERR, "Outside allowed timespan: [%s]" " (%s) time allowed: %s", namepair->strvalue,
					auth_name(authreq, 1), check_item->strvalue);
		} else if (r > 0) {
			/*
			 *	User is allowed, but set Session-Timeout.
			 */
			if ((reply_item = pairfind(user_reply,
			    PW_SESSION_TIMEOUT)) != NULL) {
				if (reply_item->lvalue > r)
					reply_item->lvalue = r;
			} else {
				if (!(reply_item = malloc(sizeof(VALUE_PAIR)))){
					log(L_ERR|L_CONS, "no memory");
					exit(1);
				}
				memset(reply_item, 0, sizeof(VALUE_PAIR));
				reply_item->attribute = PW_SESSION_TIMEOUT;
				strcpy(reply_item->name, "Session-Timeout");
				reply_item->type = PW_TYPE_INTEGER;
				reply_item->length = 4;
				reply_item->lvalue = r;
				pairadd(&user_reply, reply_item);
			}
		}
	}

	/*
	 *	Result should be >= 0 here - if not, we return.
	 */
	if (result < 0) {
		authfree(authreq);
		pairfree(user_check);
		pairfree(user_reply);
		return 0;
	}

	/*
	 *	See if we need to execute a program.
	 *	FIXME: somehow cache this info, and only execute the
	 *	program when we receive an Accounting-START packet.
	 *	Only at that time we know dynamic IP etc.
	 *
	 *	FIXME: ICRADIUS support realm to Exec-Program (as %R ?)
	 */
	exec_program = NULL;
	exec_wait = 0;
	if ((auth_item = pairfind(user_reply, PW_EXEC_PROGRAM)) != NULL) {
		exec_wait = 0;
		exec_program = strdup(auth_item->strvalue);
		pairdelete(&user_reply, PW_EXEC_PROGRAM);
	}
	if ((auth_item = pairfind(user_reply, PW_EXEC_PROGRAM_WAIT)) != NULL) {
		exec_wait = 1;
		exec_program = strdup(auth_item->strvalue);
		pairdelete(&user_reply, PW_EXEC_PROGRAM_WAIT);
	}

	/*
	 *	Hack - allow % expansion in certain value strings.
	 *	This is nice for certain Exec-Program programs.
	 */
	seen_callback_id = 0;
	if ((auth_item = pairfind(user_reply, PW_CALLBACK_ID)) != NULL) {
		seen_callback_id = 1;
		ptr = radius_xlate(auth_item->strvalue,
			authreq->request, user_reply);
		strncpy(auth_item->strvalue, ptr, sizeof(auth_item->strvalue));
		auth_item->strvalue[sizeof(auth_item->strvalue) - 1] = 0;
		auth_item->length = strlen(auth_item->strvalue);
	}


	/*
	 *	If we want to exec a program, but wait for it,
	 *	do it first before sending the reply.
	 */
	if (exec_program && exec_wait) {
		if (radius_exec_program(exec_program,
		    authreq->request, &user_reply, exec_wait, &user_msg) != 0) {
			/*
			 *	Error. radius_exec_program() returns -1 on
			 *	fork/exec errors, or >0 if the exec'ed program
			 *	had a non-zero exit status.
			 */
			if (user_msg == NULL)
		user_msg = "\r\nAccess denied (external check failed).";
			rad_send_reply(PW_AUTHENTICATION_REJECT, authreq,
				user_reply, user_msg, activefd);
			if (log_auth) {
				log(L_AUTH,
					"Login incorrect: [%s] (%s) "
					"(external check failed)",
					namepair->strvalue,
					auth_name(authreq, 1));
			}
			authfree(authreq);
			pairfree(user_check);
			pairfree(user_reply);
			return 0;
		}
	}

	/*
	 *	Delete "normal" A/V pairs when using callback.
	 *
	 *	FIXME: This is stupid. The portmaster should accept
	 *	these settings instead of insisting on using a
	 *	dialout location.
	 *
	 *	FIXME2: Move this into the above exec thingy?
	 *	(if you knew how I use the exec_wait, you'd understand).
	 */
	if (seen_callback_id) {
		pairdelete(&user_reply, PW_FRAMED_PROTOCOL);
		pairdelete(&user_reply, PW_FRAMED_IP_ADDRESS);
		pairdelete(&user_reply, PW_FRAMED_IP_NETMASK);
		pairdelete(&user_reply, PW_FRAMED_ROUTE);
		pairdelete(&user_reply, PW_FRAMED_MTU);
		pairdelete(&user_reply, PW_FRAMED_COMPRESSION);
		pairdelete(&user_reply, PW_FILTER_ID);
		pairdelete(&user_reply, PW_PORT_LIMIT);
		pairdelete(&user_reply, PW_CALLBACK_NUMBER);
	}

	/*
	 *	Filter Port-Message value through radius_xlate
	 */
	if (user_msg == NULL) {
		if ((reply_item = pairfind(user_reply,
		    PW_REPLY_MESSAGE)) != NULL) {
			user_msg = radius_xlate(reply_item->strvalue,
				authreq->request, user_reply);
			strNcpy(reply_item->strvalue, user_msg,
				sizeof(reply_item->strvalue));
			reply_item->length = strlen(reply_item->strvalue);
			user_msg = NULL;
		}
	}

	rad_send_reply(PW_AUTHENTICATION_ACK, authreq, user_reply, user_msg, activefd);
	if (log_auth) {

		UINT4	curtime = (UINT4)time(NULL);

		log(L_AUTH, "Login OK: [%s%s%s] (%s) socket %d (%d sec)", namepair->strvalue, log_auth_pass ? "/" : "",
			log_auth_pass ? userpass : "", auth_name(authreq, 0), socket->id, curtime - authreq->timestamp);
	}
	if (exec_program && !exec_wait) {
		/*
		 *	No need to check the exit status here.
		 */
		radius_exec_program(exec_program, authreq->request, &user_reply, exec_wait, NULL);
	}

	if (exec_program) free(exec_program);
	authfree(authreq);
	pairfree(user_check);
	pairfree(user_reply);

	return 0;
}

