#! /usr/bin/perl
#
# checkrad.pl	See if a user is (still) logged in on a certain port.
#
#		This is used by the cistron-radius server to check
#		if its idea of a user logged in on a certain port/nas
#		is correct if a double login is detected.
#
# Called as:	nas_type nas_ip nas_port login session_id
#
# Returns:	0 = no duplicate, 1 = duplicate, >1 = error.
#
# Version:	@(#)checkrad.pl  1.21  04-Jan-2000  miquels@cistron.nl
#
#		livingston_snmp  1.1	Author: miquels@cistron.nl
#		portslave_finger 1.0	Author: miquels@cistron.nl
#		max40xx_finger	 1.0	Author: costa@mdi.ca
#		ascend_snmp      1.1    Author: blaz@amis.net
#		computone_finger 1.1	Author: costa@mdi.ca
#		sub tc_tccheck	 1.1	Author: alexisv@compass.com.ph
#		cyclades_telnet	 1.1	Author: accdias@sst.com.br
#		patton_snmp	 1.0	Author: accdias@sst.com.br
#		digitro_rusers	 1.1	Author: accdias@sst.com.br
#		cyclades_snmp	 1.0	Author: accdias@sst.com.br
#		usrhiper_snmp    1.0    Author: igor@ipass.net
#		multitech_snmp   1.0    Author: ehonzay@willmar.com
#		netserver_telnet 1.0	Author: mts@interplanet.es
#		versanet_snmp    1.0    Author: support@versanetcomm.com
#
#	Config: $debug is the file you want to put debug messages in
#		$snmpget is the location of your ``snmpget'' program
#		$snmpwalk is the location of your ``snmpwalk'' program
#		$rusers is the location of your ``rusers'' program
#		$naspass is the location of your NAS admin password file
#

$prefix		= "/usr/local";
$localstatedir	= "/var";
$logdir		= "${localstatedir}/log";
$sysconfdir	= "/etc";
$raddbdir	= "${sysconfdir}/raddb";

$debug		= "";
#$debug		= "$logdir/checkrad.log";

$snmpget	= "/usr/bin/snmpget";
$snmpwalk	= "/usr/bin/snmpwalk";
$rusers		= "/usr/bin/rusers";
$naspass	= "$raddbdir/naspasswd";

#
#	PM3:	$lv_offs is where the last S port is before one or two
#		ports are skipped (22 or 29, for US or Europe)
#		$lv_hole is the size of the hole (1 or 2, for US or Europe).
#
$lv_offs = 22;
$lv_hole = 1;

#
#	Try to load Net::Telnet, SNMP_Session etc.
#	Do not complain if we cannot find it.
#	Prefer a locally installed copy.
#
BEGIN {
	unshift @INC, "/usr/local/lib/site_perl";

	eval "use Net::Telnet 3.00;";
	$::HAVE_NET_TELNET = ($@ eq "");

	eval "use SNMP_Session;";
	if ($@ eq "") {
		eval "use BER;";
		$::HAVE_SNMP_SESSION = ($@ eq "");
		eval "use Socket;";
	}
};

#
#	Get password from /etc/raddb/naspasswd file.
#	Returns (login, password).
#
sub naspasswd {
	my ($terminalserver, $emptyok) = @_;
	my ($login, $password);
	my ($ts, $log, $pass);

	unless (open(NFD, $naspass)) {
		if (!$emptyok) {
			print LOG "checkrad: naspasswd file not found; " .
			"possible match for $ARGV[3]\n" if ($debug);
			print STDERR "checkrad: naspassswd file not found; " .
			"possible match for $ARGV[3]\n";
		}
		return ();
	}
	while (<NFD>) {
		chop;
		next if (m/^(#|$|[\t ]+$)/);
		($ts, $log, $pass) = split(/\s+/, $_, 3);
		if ($ts eq $terminalserver) {
			$login = $log;
			$password = $pass;
			last;
		}
	}
	close NFD;
	if ($password eq "" && !$emptyok) {
		print LOG "checkrad: password for $ARGV[1] is null; " .
			"possible match for $ARGV[3] on " . 
			"port $ARGV[2]\n" if ($debug);
		print STDERR "checkrad: password for $ARGV[1] is null; " .
			"possible match for $ARGV[3] on port $ARGV[2]\n"; 
	}
	($login, $password);
}

#
#	See if Net::Telnet is there.
#
sub check_net_telnet {
	if (!$::HAVE_NET_TELNET) {
		print LOG
		"  checkrad: Net::Telnet 3.00+ CPAN module not installed\n"
		if ($debug);
		print STDERR
		"checkrad: Net::Telnet 3.00+ CPAN module not installed\n";
		return 0;
	}
	1;
}

#
#	Do snmpget by calling snmpget.
#
sub snmpget_prog {
	my ($host, $community, $oid) = @_;
	my ($ret);
	local $_;

	$_ = `$snmpget $host $community $oid`;
	if (/^.*(\s|\")([0-9A-Za-z]{8})(\s|\"|$).*$/) {
		# Session ID format.
		$ret = $2;
	} elsif (/^.*=.*"(.*)"/) {
		# oid = "...." junk format.
		$ret = $1;
	} elsif (/^.*=\s*(\S+)/) {
		# oid = string format
		$ret = $1;
	}

	# Strip trailing junk if any.
	$ret =~ s/\s*Hex:.*$//;
	$ret;
}

#
#	Do snmpget by using SNMP_Session.
#	Coded by Jerry Workman <jerry@newwave.net>
#
sub snmpget_session {
	my ($host, $community, $OID) = @_;
	my ($ret);
	local $_;
	my (@enoid, $var,$response, $bindings, $binding, $value);
	my ($inoid, $outoid, $upoid, $oid, @retvals);

	$OID =~ s/^.iso.org.dod.internet.private.enterprises/.1.3.6.1.4.1/;

	push @enoid,  encode_oid((split /\./, $OID));
	srand();

	my $session = SNMP_Session->open($host, $community, 161);
	if (!$session->get_request_response(@enoid)) {
		$e = "No SNMP answer from $ARGV[0].";
		print LOG "$e\n" if ($debug);
		print STDERR "checkrad: $e\n";
		return "";
	}
	$response = $session->pdu_buffer;
	($bindings) = $session->decode_get_response ($response);
	$session->close ();
	while ($bindings) {
		($binding,$bindings) = decode_sequence ($bindings);
		($oid,$value) = decode_by_template ($binding, "%O%@");
		my $tempo = pretty_print($value);
		$tempo=~s/\t/ /g;
		$tempo=~s/\n/ /g;
		$tempo=~s/^\s+//;
		$tempo=~s/\s+$//;

		push @retvals, $tempo;
	}
	$retvals[0];
}

#
#	Do snmpget
#
sub snmpget {
	my $ret;

	if ($::HAVE_SNMP_SESSION) {
		$ret = snmpget_session(@_);
	} elsif (-x $snmpget) {
		$ret = snmpget_prog(@_);
	} else {
		$e = "Neither SNMP_Session module or $snmpget found!";
		print LOG "$e\n" if ($debug);
		print STDERR "checkrad: $e\n";
		$ret = "";
	}
	$ret;
}

#
#	Strip domains, prefixes and suffixes from username
#	
#	Known prefixes: (P)PP, (S)LIP e (C)SLIP
#	Known suffixes: .ppp, .slip e .cslip
#
#	Author: Antonio Dias of SST Internet <accdias@sst.com.br>
#
sub strip_username {
	my ($user) = @_;
	#
	#	Trim white spaces.
	#
	$user =~ s/^\s*(.*?)\s*$/$1/;
	#
	#	Strip out domains, prefix and suffixes
	#
	$user =~ s/\@(.)*$//;
	$user =~ s/^[PSC]//;
	$user =~ s/\.(ppp|slip|cslip)$//;
	$user;
}

#
#	See if the user is logged in using the Livingston MIB.
#	We don't check the username but the session ID.
#
$lvm	 = '.iso.org.dod.internet.private.enterprises.307';
sub livingston_snmp {

	#
	#	First find out the offset (ugly!!). Also, if the portno
	#	is greater than 29, substract 2 (S30 and S31 don't exist).
	#	You might need to change this to 23 and 1 for the USA.
	#
	$_ = snmpget($ARGV[1], $ARGV[5], "$lvm.3.2.1.1.1.2.5");
	/S([0-9]+)/;
	$xport = $1 + 0;
	$portidx = $ARGV[2] + (5 - $xport);
	$portidx -= $lv_hole if ($ARGV[2] > $lv_offs);
	chop;
	print LOG "  using $xport offset for port / SNMPno translation\n"
		if ($debug);

	#
	#	Now get the session id from the terminal server.
	#
	$sessid = snmpget($ARGV[1], $ARGV[5], "$lvm.3.2.1.1.1.5.$portidx");

	print LOG "  session id at port S$ARGV[2]: $sessid\n" if ($debug);

	($sessid eq $ARGV[4]) ? 1 : 0;
}

#
#	See if the user is logged in using the Cisco MIB
#
$csm	 = '.iso.org.dod.internet.private.enterprises.9';
sub cisco_snmp {
	$login = snmpget($ARGV[1], $ARGV[5], "$csm.2.9.2.1.18.$ARGV[2]");

	print LOG "  user at port S$ARGV[2]: $login\n" if ($debug);

	($login eq $ARGV[3]) ? 1 : 0;
}

#
#       Check a MultiTech CommPlete Server ( CC9600 & CC2400 )
#
#       Author: Eric Honzay of Bennett Office Products <ehonzay@willmar.com>
#
$msm    = '.iso.org.dod.internet.private.enterprises.995';
sub multitech_snmp {
	my $temp = $ARGV[2] + 1;

        $login = snmpget($ARGV[1], $ARGV[5], "$msm.2.31.1.1.1.$temp");
        print LOG " user at port S$ARGV[2]: $login\n" if ($debug);

        ($login eq $ARGV[3]) ? 1 : 0;
}

#
#	Check a Computone Powerrack via finger
#
#	Author: Shiloh Costa of MDI Internet Inc. <costa@mdi.ca>
#
sub computone_finger {
	$online = 0;
	open(FD, "finger \@$ARGV[1]|");
	while(<FD>) {  
	   @line = <FD>;
	   }
	 $numline = @line;
	 for ($num = 0; $num < $numline; $num++) {
	   if( $line[$num]  =~ / $ARGV[3] / ){
	     #print "$ARGV[3] is online\n";
	     $online = 1; # user is online
	   }
	 }
	close FD;
	return $online;
}

#
#	Check an Ascend Max4000 or similar model via finger
#
#	Note: Not all software revisions support finger
#	      You may also need to enable the finger option.
#
#	Author: Shiloh Costa of MDI Internet Inc. <costa@mdi.ca>
#
sub max40xx_finger {
	open(FD, "finger $ARGV[3]\@$ARGV[1]|");
	while(<FD>) {
	   $line = <FD>;
	   if( $line =~ /Session/ ){
	      return 1; # user is online
	   }else{
	      return 0; # user is offline
	   }
	}
	close FD;
}

#
#	Check an Ascend Max4000 or similar model via SNMP
#
#	Author: Blaz Zupan of Medinet <blaz@amis.net>
#
$asm   = '.iso.org.dod.internet.private.enterprises.529';
sub ascend_snmp {
	my $sess_id;
	my $l1, $l2;

	$l1 = '';
	$l2 = '';

	#
	#	If it looks like hex, only try it as hex,
	#	otherwise try it as both decimal and hex.
	#
	$sess_id = $ARGV[4];
	if ($sess_id !~ /^0/ && $sess_id !~ /[a-f]/i) {
		$l1 = snmpget($ARGV[1], $ARGV[5], "$asm.12.3.1.4.$sess_id");
	}
	$sess_id = hex $ARGV[4];
	$l2 = snmpget($ARGV[1], $ARGV[5], "$asm.12.3.1.4.$sess_id");

	print LOG "  user at port S$ARGV[2]: $l1 (dec)\n" if ($debug && $l1);
	print LOG "  user at port S$ARGV[2]: $l2 (hex)\n" if ($debug && $l2);

	(($l1 && $l1 eq $ARGV[3]) || ($l2 && $l2 eq $ARGV[3])) ? 1 : 0;
}
  

#
#	See if the user is logged in using the portslave finger.
#
sub portslave_finger {
	my ($Port_seen);

	$Port_seen = 0;

	open(FD, "finger \@$ARGV[1]|");
	while(<FD>) {
		#
		#	Check for ^Port. If we don't see it we
		#	wont get confused by non-portslave-finger
		#	output too.
		#
		if (/^Port/) {
			$Port_seen++;
			next;
		}
		next if (!$Port_seen);
		next if (/^---/);

		($port, $user) = /^.(...) (...............)/;

		$port =~ s/ .*//;
		$user =~ s/ .*//;
		$ulen = length($user);
		#
		#	HACK: strip [PSC] from the front of the username,
		#	and things like .ppp from the end.
		#
		$user =~ s/^[PSC]//;
		$user =~ s/\.(ppp|slip|cslip)$//;

		#
		#	HACK: because ut_user usually has max. 8 characters
		#	we only compare up the the length of $user if the
		#	unstripped name had 8 chars.
		#
		$argv_user = $ARGV[3];
		if ($ulen == 8) {
			$ulen = length($user);
			$argv_user = substr($ARGV[3], 0, $ulen);
		}

		if ($port == $ARGV[2]) {
			if ($user eq $argv_user) {
				print LOG "  $user matches $argv_user " .
					"on port $port" if ($debug);
				close FD;
				return 1;
			} else {
				print LOG "  $user doesn't match $argv_user " .
					"on port $port" if ($debug);
				close FD;
				return 0;
			}
		}
	}
	close FD;
	0;
}

#
#	See if the user is already logged-in at the 3Com/USR Total Control.
#	(this routine by Alexis C. Villalon <alexisv@compass.com.ph>).
#	You must have the Net::Telnet module from CPAN for this to work.
#	You must also have your /etc/raddb/naspasswd made up.
# 
sub tc_tccheck {
	#
	#	Localize all variables first.
	#
	my ($Port_seen, $ts, $terminalserver, $log, $login, $pass, $password);
	my ($telnet, $curprompt, $curline, $ok, $totlines, $ccntr);
	my (@curlines, @cltok, $user, $port, $ulen);

	return 2 unless (check_net_telnet());

	$terminalserver = $ARGV[1];
	$Port_seen = 0;
	#
	#	Get login name and password for a certain NAS from $naspass.
	#
	($login, $password) = naspasswd($terminalserver, 1);
	return 2 if ($password eq "");

	#
	#	Communicate with NAS using Net::Telnet, then issue
	#	the command "show sessions" to see who are logged in.
	#	Thanks to Chris Jackson <chrisj@tidewater.net> for the
	#	for the "-- Press Return for More --" workaround.
	#
	$telnet = new Net::Telnet (Timeout => 10,
				   Prompt => '/\>/');
	$telnet->open($terminalserver);
	$telnet->login($login, $password);
	$telnet->print("show sessions");
	while ($curprompt ne "\>") {
		($curline, $curprompt) = $telnet->waitfor
			(String => "-- Press Return for More --",
			 String => "\>",
			 Timeout => 10);
		$ok = $telnet->print("");
		push @curlines, split(/^/m, $curline);
	}
	$telnet->close;
	#
	#	Telnet closed.  We got the info.  Let's examine it.
	#
	$totlines = @curlines;
	$ccntr = 0;
	while($ccntr < $totlines) {
		#
		#	Check for ^Port.
		#
		if ($curlines[$ccntr] =~ /^Port/) {
			$Port_seen++;
			$ccntr++;
			next;
		}
		#
		#	Ignore all unnecessary lines.
		#
		if (!$Port_seen || $curlines[$ccntr] =~ /^---/ ||
			$curlines[$ccntr] =~ /^ .*$/) {
			$ccntr++;
			next;
		}
		#
		#	Parse the current line for the port# and username.
		#
		@cltok = split(/\s+/, $curlines[$ccntr]);
		$ccntr++;
		$port = $cltok[0];
		$user = $cltok[1];
		$ulen = length($user);
		#
		#	HACK: strip [PSC] from the front of the username,
		#	and things like .ppp from the end.  Strip S from
		#	the front of the port number.
		#
		$user =~ s/^[PSC]//;
		$user =~ s/\.(ppp|slip|cslip)$//;
		$port =~ s/^S//;
		#		
		#	HACK: because "show sessions" shows max. 15 characters
		#	we only compare up to the length of $user if the
		#	unstripped name had 15 chars.
		#
		$argv_user = $ARGV[3];
		if ($ulen == 15) {
			$ulen = length($user);
			$argv_user = substr($ARGV[3], 0, $ulen);
		}
		if ($port == $ARGV[2]) {
			if ($user eq $argv_user) {
				print LOG "  $user matches $argv_user " .
					"on port $port" if ($debug);
				return 1;
			} else {
				print LOG "  $user doesn't match $argv_user " .
					"on port $port" if ($debug);
				return 0;
			}
		}
	}
	0;
}

#
#	Check a Cyclades PathRAS via telnet
#
#	Version: 1.1
#
#	Author: Antonio Dias of SST Internet <accdias@sst.com.br>
#
sub cyclades_telnet {
   #
   #	Localize all variables first.
   #
   my (@users, $pathras, $port, $errmsg, $user, $login, $password);
   #
   #	Do we have Net::Telnet installed?
   #
   return 2 unless (check_net_telnet());
   #
   #	Get login name and password for NAS from $naspass file.
   #
   ($login, $password) = naspasswd($ARGV[1], 1);
   #
   #	NAS doesn't have a password defined
   #
   return 2 if ($password eq "");
   #
   #	Communicate with PathRAS using Net::Telnet, then access
   #	menu option 6.8 to see who are logged in. This assumes 
   #	you are using the default Prompt String (menu option 6.2)
   #	and is based on PathRAS firmware version 1.1.9g
   # 
   #
   $pathras = new Net::Telnet (
   	Timeout => 10, 
   	Prompt => '/(.*)\s:*$/', 
   	Errmode => 'return'
   );
   #
   #
   #
   if (!$pathras->open($ARGV[1])) {;
      print LOG "  Error: " . $pathras->errmsg . "\n" if ($debug);
      $pathras->close;
      return 2;
   }
   #
   #	Dunno why but PathRAS only issue its banner and 
   #	the login prompt after receive a "CR" 
   #
   if (!$pathras->print("")) {
      print LOG "  Error: " . $pathras->errmsg . "\n" if ($debug);
      $pathras->close;
      return 2;

   }
   #
   #	Log on PathRAS
   #
   if (!($pathras->login($login, $password))) {
      print LOG "  Error: " . $pathras->errmsg . "\n" if ($debug);
      $pathras->close;
      return 2;
   }
   #
   #  Access menu option 6 "PathRAS Management"
   #
   if (!($pathras->cmd(String => '6', 
   	Prompt => '/ ==\> $/', 
   	Timeout => 10))) {
      print LOG "  Error: " . $pathras->errmsg . "\n" if ($debug);
      $pathras->close;
      return 2;
   }
   #
   #  Access menu option 8 "Show Active Ports"
   #
   if (!(@users = $pathras->cmd(String => '8', 
   	Prompt => '/Type \<enter\>/', 
   	Timeout => 10))) {
      print LOG "  Error: " . $pathras->errmsg . "\n" if ($debug);
      $pathras->close;
      return 2;
   }
   #
   # Since we got the info we want, let's close the session
   #
   $pathras->close;
   #
   #	Telnet closed.  We got the info so let's examine it.
   #
   foreach(@users) {
      if (m/PPP Active|SLIP Active/) {
         ($port, $user) = split;
         #
         # Strip out any prefixes and suffixes 
         # from the username
         #
         $user = strip_username($user);
         #
         # Check if $user is already connected
         #
         if ($user eq $ARGV[3]) {
            print LOG "  User $ARGV[3] found on " . 
               "NAS $ARGV[1] port $port.\n" if ($debug);
            return 1;
         }
      }
   }
   print LOG "  User $ARGV[3] not found on " .
   	"NAS $ARGV[1].\n" if ($debug);
   0;
}

#
#	Check a Patton 2800 via snmp
#
#	Version: 1.0
#
#	Author: Antonio Dias of SST Internet <accdias@sst.com.br>
#
sub patton_snmp {
   my($oid);

   $oid = '.1.3.6.1.4.1.1768.5.100.1.40.' . hex $ARGV[4];
   #
   # Check if the session still active
   # 
   if (snmpget($ARGV[1], $ARGV[5], "$oid") == 0) {
      print LOG "  Session $ARGV[4] still active on NAS " . 
      	"$ARGV[1], port $ARGV[2], for user $ARGV[3].\n" if ($debug);
      return 1;
   }
   0;
}

#
#      Check a Digitro BXS via rusers
#
#      Version: 1.1
#
#      Author: Antonio Dias of SST Internet <accdias@sst.com.br>
#
sub digitro_rusers {
   my ($ret);
   local $_;

   if (-e $rusers && -x $rusers) {
      #
      # Get a list of users logged in via rusers
      #
      $_ =`$rusers $ARGV[1]`;
      $ret = ((/$ARGV[3]/) ? 1 : 0);
   } else {
      print LOG "   Error: can't execute $rusers\n" if $debug;
      $ret = 2;
   }
   $ret;
}

#
#	Check Cyclades PR3000 and PR4000 via SNMP
#
#	Version: 1.0
#
#	Author: Antonio Dias of SST Internet <accdias@sst.com.br>
#
sub cyclades_snmp {
   my ($oid, $ret);
   local $_;
   
   $oid = ".1.3.6.1.4.1.2925.3.3.6.1.1.2";

   if (-e $snmpwalk && -x $snmpwalk) { 	
      #
      # Get a list of logged in users via snmpwalk
      #
      # I'm looking for a way to do this using SNMP_Session
      # module. Any help would be great.
      #
      $_ = `$snmpwalk $ARGV[1] $ARGV[5] $oid`;
      $ret = ((/$ARGV[3]/) ? 1 : 0);
   } else {
      print LOG "   Error: can't execute $snmpwalk\n" if $debug;
      $ret = 2;
   }
   $ret;
}

#
#	3Com/USR HiPer Arc Total Control.
#	This works with HiPer Arc 4.0.30
#	(this routine by Igor Brezac <igor@ipass.net>)
# 
$usrm	 = '.iso.org.dod.internet.private.enterprises.429';
sub usrhiper_snmp {

	# Somebody please verify????
	my($oidext) = $ARGV[2] + 1256;
	my ($login);

	$login = snmpget($ARGV[1], $ARGV[5], "$usrm.4.10.1.1.18.$oidext");
	#($login) = /^.*\"([^"]+)".*$/;
	if ($login =~ /\"/) {
		$login =~ /^.*\"([^"]+)\"/;
		$login = $1;
	}

	print LOG "  user at port S$ARGV[2]: $login\n" if ($debug);

	($login eq $ARGV[3]) ? 1 : 0;
}


#
#	Check USR Netserver with Telnet - based on tc_tccheck.
#	By "Marti" <mts@interplanet.es>
#
sub usrnet_telnet {
	#
	#	Localize all variables first.
	#
	my ($ts, $terminalserver, $login, $password);
	my ($telnet, $curprompt, $curline, $ok);
	my (@curlines, $user, $port);

	return 2 unless (check_net_telnet());

	$terminalserver = $ARGV[1];
	$Port_seen = 0;
	#
	#	Get login name and password for a certain NAS from $naspass.
	#
	($login, $password) = naspasswd($terminalserver, 1);
	return 2 if ($password eq "");

	#
	#	Communicate with Netserver using Net::Telnet, then access
	#	list connectionsto see who are logged in. 
	# 
	$telnet = new Net::Telnet (Timeout => 10,
				   Prompt => '/\>/');
	$telnet->open($terminalserver);

	#
	#	Log on Netserver
	#
	$telnet->login($login, $password);

	#
	#     Launch list connections command

	$telnet->print("list connections");

	while ($curprompt ne "\>") {
		($curline, $curprompt) = $telnet->waitfor
			( String => "\>",
			 Timeout => 10);
		$ok = $telnet->print("");
		push @curlines, split(/^/m, $curline);
	}

	$telnet->close;
	#
	#	Telnet closed.  We got the info.  Let's examine it.
	#
	foreach(@curlines) {
		if ( /mod\:/ ) {
			($port, $user, $dummy) = split;
			#
			# Strip out any prefixes and suffixes 
			# from the username
			#
			# uncomment this if you use the standard
			# prefixes
			#$user =~ s/^[PSC]//;
			#$user =~ s/\.(ppp|slip|cslip)$//;
			#
			# Check to see if $user is already connected
			#
			if ($user eq $ARGV[3]) {
				print LOG "  $user matches $ARGV[3] " .
					"on port $port" if ($debug);
				return 1;
			};
		};
	};
	print LOG 
	"  $ARGV[3] not found on Netserver logged users list " if ($debug);
	0;
}

#
#	Versanet's Perl Script Support:
#
#	___ versanet_snmp 1.0 by support@versanetcomm.com ___ July 1999
#	Versanet Enterprise MIB Base: 1.3.6.1.4.1.2180
#   
#	VN2001/2002 use slot/port number to locate modems. To use snmp get we
#	have to translate the original port number into a slot/port pair.
#
$vsm     = '.iso.org.dod.internet.private.enterprises.2180';
sub versanet_snmp {
        
	print LOG "argv[2] = $ARGV[2] " if ($debug);
	$port = $ARGV[2]%8;
	$port = 8 if ($port eq 0);	  
	print LOG "port = $port " if ($debug);
	$slot = (($ARGV[2]-$port)/8)+1;
	print LOG "slot = $slot" if ($debug);
	$loginname = snmpget($ARGV[1], $ARGV[5], "$vsm.27.1.1.3.$slot.$port");

	  print LOG "  user at slot $slot port $port: $loginname\n" if ($debug);	  ($loginname eq $ARGV[3]) ? 1 : 0;	
}


if ($debug) {
	open(LOG, ">>$debug");
	$now = localtime;
	print LOG "$now checkrad @ARGV\n";
}

if ($#ARGV < 4) {
	print LOG "Usage: checkrad nas_type nas_ip " .
			"nas_port login session_id [community]\n" if ($debug);
	print STDERR "Usage: checkrad nas_type nas_ip " .
			"nas_port login session_id [community]\n";
	close LOG if ($debug);
	exit(2);
}

if ($ARGV[0] eq 'livingston') {
	$ret = &livingston_snmp;
} elsif ($ARGV[0] eq 'cisco') {
	$ret = &cisco_snmp;
} elsif ($ARGV[0] eq 'multitech') {
        $ret = &multitech_snmp;
} elsif ($ARGV[0] eq 'computone') {
	$ret = &computone_finger;
} elsif ($ARGV[0] eq 'max40xx') {
	$ret = &max40xx_finger;
} elsif ($ARGV[0] eq 'ascend') {
	$ret = &ascend_snmp;
} elsif ($ARGV[0] eq 'portslave') {
	$ret = &portslave_finger;
} elsif ($ARGV[0] eq 'tc') {
	$ret = &tc_tccheck;
} elsif ($ARGV[0] eq 'pathras') {
	$ret = &cyclades_telnet;
} elsif ($ARGV[0] eq 'pr3000') {
	$ret = &cyclades_snmp;
} elsif ($ARGV[0] eq 'pr4000') {
	$ret = &cyclades_snmp;
} elsif ($ARGV[0] eq 'patton') {
	$ret = &patton_snmp;
} elsif ($ARGV[0] eq 'digitro') {
	$ret = &digitro_rusers;
} elsif ($ARGV[0] eq 'usrhiper') {
	$ret = &usrhiper_snmp;
} elsif ($ARGV[0] eq 'netserver') {
	$ret = &usrnet_telnet;
} elsif ($ARGV[0] eq 'versanet'){
        $ret = &versanet_snmp;
} elsif ($ARGV[0] eq 'other') {
	$ret = 1;
} else {
	print LOG "  checkrad: unknown NAS type $ARGV[0]\n" if ($debug);
	print STDERR "checkrad: unknown NAS type $ARGV[0]\n";
	$ret = 2;
}

if ($debug) {
	$mn = "login ok";
	$mn = "double detected" if ($ret == 1);
	$mn = "error detected" if ($ret == 2);
	print LOG "  Returning $ret ($mn)\n";
	close LOG;
}

exit($ret);
