/*
 * mysql.c	ICRADIUS SQL routines
 *
 */

#include	<sys/types.h>
#include	<sys/socket.h>
#include	<sys/time.h>
#include	<sys/file.h>
#include	<sys/stat.h>
#include	<netinet/in.h>

#include	<stdio.h>
#include	<stdlib.h>
#include	<netdb.h>
#include	<pwd.h>
#include	<time.h>
#include	<ctype.h>
#include	<unistd.h>
#include	<signal.h>
#include	<errno.h>
#include	<sys/wait.h>

#include	"conf.h"

#include	"radiusd.h"

/*************************************************************************
 *
 *	Function: mysql_start
 *
 *	Purpose: Reads SQL Config File 
 *
 *************************************************************************/

int sql_init(int reload) {
	FILE    *sqlfd;
        char    dummystr[64];
        char    namestr[64];
        int     line_no = 0;
        char    buffer[256];
        char    sqlfile[256];

	if (reload)
		free(sql->config);
	if ((sql->config = malloc(sizeof(SQL_CONFIG))) == NULL) {
		log(L_ERR|L_CONS, "no memory");
		exit(1);
	}
	strcpy(sql->config->sql_server,"localhost");
	strcpy(sql->config->sql_login,"");
	strcpy(sql->config->sql_password,"");
	strcpy(sql->config->sql_acct_server,"localhost");
	strcpy(sql->config->sql_acct_login,"");
	strcpy(sql->config->sql_acct_password,"");
	strcpy(sql->config->sql_db,"radius");
	strcpy(sql->config->sql_acct_db,"radius");
	strcpy(sql->config->sql_authcheck_table,"radcheck");
	strcpy(sql->config->sql_authreply_table,"radreply");
	strcpy(sql->config->sql_groupcheck_table,"radgroupcheck");
	strcpy(sql->config->sql_groupreply_table,"radgroupreply");
	strcpy(sql->config->sql_usergroup_table,"usergroup");
	strcpy(sql->config->sql_realmgroup_table,"realmgroup");
	strcpy(sql->config->sql_acct_table,"radacct");
	sql->config->sqltrace = 0;
	sql->config->sensitiveusername = 1;
	sql->config->deletestalesessions = 0;

	strcpy(sql->config->sql_hints_table,"hints");
	strcpy(sql->config->sql_nas_table,"nas");
	strcpy(sql->config->sql_realm_table, "realms");
	strcpy(sql->config->sql_dict_table,"dictionary");
	if (reload)
		sql->config->max_sql_oldsocks = sql->config->max_sql_socks;
	else
		sql->config->max_sql_oldsocks = MAX_SQL_SOCKS;
	sql->config->max_sql_socks = MAX_SQL_SOCKS;

        sprintf(sqlfile, "%s/%s", radius_dir, MYSQLCONFIG);
        if((sqlfd = fopen(sqlfile, "r")) == (FILE *)NULL) {
                log(L_ERR,"could not read MySQL configuration file %s",sqlfile);
                return(-1);
        }

        line_no = 0;
        while(fgets(buffer, sizeof(buffer), sqlfd) != (char *)NULL) {
                line_no++;

                /* Skip empty space */
                if(*buffer == '#' || *buffer == '\0' || *buffer == '\n') {
                        continue;
                }

                if(strncasecmp(buffer, "server", 6) == 0) {
                        /* Read the SERVER line */
                        if(sscanf(buffer, "%s%s", dummystr, namestr) != 2) {
                               log(L_ERR,"invalid attribute on line %d of configuration file %s", line_no,sqlfile);
                       } else {
                         strcpy(sql->config->sql_server,namestr);
                         strcpy(sql->config->sql_acct_server,namestr);
                       }
               }
                if(strncasecmp(buffer, "login", 5) == 0) {
                        /* Read the LOGIN line */
                        if(sscanf(buffer, "%s%s", dummystr, namestr) != 2) {
                               log(L_ERR,"invalid attribute on line %d of configuration file %s", line_no,sqlfile);
                       } else {
                         strcpy(sql->config->sql_login,namestr);
                         strcpy(sql->config->sql_acct_login,namestr);
                       }
               }
                if(strncasecmp(buffer, "password", 8) == 0) {
                        /* Read the PASSWORD line */
                        if(sscanf(buffer, "%s%s", dummystr, namestr) != 2) {
                               log(L_ERR,"invalid attribute on line %d of configuration file %s", line_no,sqlfile);
                       } else {
                         strcpy(sql->config->sql_password,namestr);
                         strcpy(sql->config->sql_acct_password,namestr);
                       }
               }
                if(strncasecmp(buffer, "acct_server", 11) == 0) {
                        /* Read the SERVER line */
                        if(sscanf(buffer, "%s%s", dummystr, namestr) != 2) {
                               log(L_ERR,"invalid attribute on line %d of configuration file %s", line_no,sqlfile);
                       } else {
                         strcpy(sql->config->sql_acct_server,namestr);
                       }
               }
                if(strncasecmp(buffer, "acct_login", 10) == 0) {
                        /* Read the LOGIN line */
                        if(sscanf(buffer, "%s%s", dummystr, namestr) != 2) {
                               log(L_ERR,"invalid attribute on line %d of configuration file %s", line_no,sqlfile);
                       } else {
                         strcpy(sql->config->sql_acct_login,namestr);
                       }
               }
                if(strncasecmp(buffer, "acct_password", 13) == 0) {
                        /* Read the PASSWORD line */
                        if(sscanf(buffer, "%s%s", dummystr, namestr) != 2) {
                               log(L_ERR,"invalid attribute on line %d of configuration file %s", line_no,sqlfile);
                       } else {
                         strcpy(sql->config->sql_acct_password,namestr);
                       }
               }
                if(strncasecmp(buffer, "radius_db", 9) == 0) {
                        /* Read the RADIUS_DB line */
                        if(sscanf(buffer, "%s%s", dummystr, namestr) != 2) {
                               log(L_ERR,"invalid attribute on line %d of configuration file %s", line_no,sqlfile);
                       } else {
                         strcpy(sql->config->sql_db,namestr);
                       }
               }
                if(strncasecmp(buffer, "radius_acct_db", 14) == 0) {
                        /* Read the RADIUS_DB line */
                        if(sscanf(buffer, "%s%s", dummystr, namestr) != 2) {
                               log(L_ERR,"invalid attribute on line %d of configuration file %s", line_no,sqlfile);
                       } else {
                         strcpy(sql->config->sql_acct_db,namestr);
                       }
               }
                if(strncasecmp(buffer, "authcheck_table", 15) == 0) {
                        /* Read the AUTHCHECK_TABLE line */
                        if(sscanf(buffer, "%s%s", dummystr, namestr) != 2) {
                               log(L_ERR,"invalid attribute on line %d of configuration file %s", line_no,sqlfile);
                       } else {
                         strncpy(sql->config->sql_authcheck_table,namestr, MAX_TABLE_LEN);
                       }
               }
                if(strncasecmp(buffer, "authreply_table", 15) == 0) {
                        /* Read the AUTHREPLY_TABLE line */
                        if(sscanf(buffer, "%s%s", dummystr, namestr) != 2) {
                               log(L_ERR,"invalid attribute on line %d of configuration file %s", line_no,sqlfile);
                       } else {
                         strncpy(sql->config->sql_authreply_table,namestr, MAX_TABLE_LEN);
                       }
               }
                if(strncasecmp(buffer, "groupcheck_table", 16) == 0) {
                        /* Read the GROUPCHECK_TABLE line */
                        if(sscanf(buffer, "%s%s", dummystr, namestr) != 2) {
                               log(L_ERR,"invalid attribute on line %d of configuration file %s", line_no,sqlfile);
                       } else {
                         strncpy(sql->config->sql_groupcheck_table,namestr, MAX_TABLE_LEN);
                       }
               }
                if(strncasecmp(buffer, "groupreply_table", 16) == 0) {
                        /* Read the GROUP_REPLY_TABLE line */
                        if(sscanf(buffer, "%s%s", dummystr, namestr) != 2) {
                               log(L_ERR,"invalid attribute on line %d of configuration file %s", line_no,sqlfile);
                       } else {
                         strncpy(sql->config->sql_groupreply_table,namestr, MAX_TABLE_LEN);
                       }
               }
                if(strncasecmp(buffer, "usergroup_table", 15) == 0) {
                        /* Read the USERGROUP_TABLE line */
                        if(sscanf(buffer, "%s%s", dummystr, namestr) != 2) {
                               log(L_ERR,"invalid attribute on line %d of configuration file %s", line_no,sqlfile);
                       } else {
                         strncpy(sql->config->sql_usergroup_table,namestr, MAX_TABLE_LEN);
                       }
               }
                if(strncasecmp(buffer, "realmgroup_table", 16) == 0) {
                        /* Read the REALMGROUP_TABLE line */
                        if(sscanf(buffer, "%s%s", dummystr, namestr) != 2) {
                               log(L_ERR,"invalid attribute on line %d of configuration file %s", line_no,sqlfile);
                       } else {
                         strncpy(sql->config->sql_realmgroup_table,namestr, MAX_TABLE_LEN);
                       }
               }
                if(strncasecmp(buffer, "acct_table", 10) == 0) {
                        /* Read the ACCT_TABLE line */
                        if(sscanf(buffer, "%s%s", dummystr, namestr) != 2) {
                               log(L_ERR,"invalid attribute on line %d of configuration file %s", line_no,sqlfile);
                       } else {
                         strncpy(sql->config->sql_acct_table,namestr, MAX_TABLE_LEN);
                       }
               }
                if(strncasecmp(buffer, "hints_table", 11) == 0) {
                        /* Read the HINTS_TABLE line */
                        if(sscanf(buffer, "%s%s", dummystr, namestr) != 2) {
                               log(L_ERR,"invalid attribute on line %d of configuration file %s", line_no,sqlfile);
                       } else {
                         strncpy(sql->config->sql_hints_table,namestr, MAX_TABLE_LEN);
                       }
               }
                if(strncasecmp(buffer, "nas_table", 9) == 0) {
                        /* Read the NAS_TABLE line */
                        if(sscanf(buffer, "%s%s", dummystr, namestr) != 2) {
                               log(L_ERR,"invalid attribute on line %d of configuration file %s", line_no,sqlfile);
                       } else {
                         strncpy(sql->config->sql_nas_table,namestr, MAX_TABLE_LEN);
                       }
               }
                if(strncasecmp(buffer, "realm_table", 9) == 0) {
                       /* Read the REALM_TABLE line */
                       if(sscanf(buffer, "%s%s", dummystr, namestr) != 2) {
                               log(L_ERR,"invalid attribute on line %d of configuration file %s", line_no,sqlfile);
                      } else {
                         strncpy(sql->config->sql_realm_table,namestr, MAX_TABLE_LEN);
                      }
               }
                if(strncasecmp(buffer, "dict_table", 9) == 0) {
                        /* Read the DICT_TABLE line */
                        if(sscanf(buffer, "%s%s", dummystr, namestr) != 2) {
                               log(L_ERR,"invalid attribute on line %d of configuration file %s", line_no,sqlfile);
                       } else {
                         strncpy(sql->config->sql_dict_table,namestr, MAX_TABLE_LEN);
                       }
               }
                if(strncasecmp(buffer, "sqltrace", 8) == 0) {
                        /* Read the SQLTRACE line */
                        if(sscanf(buffer, "%s%s", dummystr, namestr) != 2) {
                               log(L_ERR,"invalid attribute on line %d of configuration file %s", line_no,sqlfile);
                      } else {
                         if(strncasecmp(namestr, "on", 2) == 0) {
                           sql->config->sqltrace = 1;
                         } else {
                           sql->config->sqltrace = 0;
                         }
                       }
               }
               if(strncasecmp(buffer, "sensitiveusername", 17) == 0) {
                        /* Read the SENSITIVEUSERNAME line */
                        if(sscanf(buffer, "%s%s", dummystr, namestr) != 2) {
                               log(L_ERR,"invalid attribute on line %d of configuration file %s", line_no,sqlfile);
                       } else {
                         if(strncasecmp(namestr, "on", 2) == 0) {
                           sql->config->sensitiveusername = 1;
                        } else {
                           sql->config->sensitiveusername = 0;
                        }
                       }
               }
               if(strncasecmp(buffer, "deletestalesessions", 19) == 0) {
                        /* Read the DELETESTALESESSIONS line */
                        if(sscanf(buffer, "%s%s", dummystr, namestr) != 2) {
                               log(L_ERR,"invalid attribute on line %d of configuration file %s", line_no,sqlfile);
                       } else {
                         if(strncasecmp(namestr, "on", 2) == 0) {
                           sql->config->deletestalesessions = 1;
                        } else {
                           sql->config->deletestalesessions = 0;
                        }
                       }
               }
               if(strncasecmp(buffer, "max_sql_sockets", 15) == 0) {
                        /* Read the MAX_SQL_SOCKETS line */
                        if(sscanf(buffer, "%s%s", dummystr, namestr) != 2) {
                               log(L_ERR,"invalid attribute on line %d of configuration file %s", line_no,sqlfile);
                       } else {
                           sql->config->max_sql_socks = atoi(namestr);
                       }
               }

       }
       fclose(sqlfd);
       
       log(L_INFO,"SQL: Attempting to connect to %s@%s:%s", sql->config->sql_login, sql->config->sql_server, sql->config->sql_db);

	if (sql_init_socket(reload))
		return 1;
	else
		return 0;
}


/*************************************************************************
 *
 *	Function: sql_init_socket
 *
 *	Purpose: Connect to the sql server
 *
 *************************************************************************/
int sql_init_socket(int reload) {

	int i;

	/* Clear up old connections if reload */
	if (reload) {
	        for (i = 0; i < sql->config->max_sql_oldsocks; i++) {
			if (!sql_close_socket(sql->socks[i]))
				log(L_CONS|L_ERR, "Could not release socket %d", i);
		}
		free(sql->socks);
	}
	

        /* Initalize our connection pool */
	sql->socks = malloc(sizeof(SQLSOCK *) * sql->config->max_sql_socks);
	if(!sql->socks) {
		log(L_ERR,"Unable to allocate memory for sockets structure");
		return -1;
	}
	bzero(sql->socks,sizeof(SQLSOCK *) * sql->config->max_sql_socks);
        for (i = 0; i < sql->config->max_sql_socks; i++) {
	 	if ((sql->socks[i] = sql_create_socket()) == NULL) {
	   		log(L_CONS|L_ERR, "SQL: Failed to connect socket %d", i);
		} else {
	   		sql->socks[i]->id = i;
           		sql->socks[i]->in_use = 0;
	   		DEBUG2("SQL: Connected socket %d", i);
		}
	}

	return 1;
}


/*************************************************************************
 *
 *	Function: sql_close_socket
 *
 *	Purpose: Close and free a sql socket
 *
 *************************************************************************/
int sql_close_socket(SQLSOCK *socket) {
	DEBUG2("SQL: Closing socket %d", socket->id);
	sql_close(socket);
	free(socket);

	return 1;
}
 

/*************************************************************************
 *
 *	Function: sql_get_socket
 *
 *	Purpose: Return a SQL socket from the connection pool           
 *
 *************************************************************************/
SQLSOCK *sql_get_socket(void) {

	int	i = 0;
	
	DEBUG2("SQL: Attempting to reserve socket");
	while (1) {
		if (i == sql->config->max_sql_socks)
			i = 0;
		if (sql->socks[i]->in_use == 0) {
			sql->socks[i]->in_use = 1;
			gettimeofday(&(sql->socks[i]->tv), NULL);
			DEBUG2("SQL: Reserved socket %d", i);
			return sql->socks[i];
		}
		i++;
		sleep(1);
	}
}

/*************************************************************************
 *
 *	Function: sql_release_socket
 *
 *	Purpose: Frees a SQL socket back to the connection pool           
 *
 *************************************************************************/
int sql_release_socket(SQLSOCK *socket) {

	struct timeval	tv;
	double		start, end;
	char		buff[24];

	gettimeofday(&tv, NULL);
	sprintf(buff, "%ld.%2ld", tv.tv_sec, tv.tv_usec);
	end = strtod(buff, NULL);
	sprintf(buff, "%ld %2.0ld", socket->tv.tv_sec, socket->tv.tv_usec);
	start = strtod(buff, NULL);
	DEBUG2("SQL: Socket %d used for %.2f seconds", socket->id, end - start);

	socket->tv.tv_sec = tv.tv_sec;
	socket->tv.tv_usec = tv.tv_usec;
	sql->socks[socket->id]->in_use = 0;
	DEBUG2("SQL: Released socket %d", socket->id);
	return 1;
}

 
/*************************************************************************
 *
 *	Function: sql_save_acct
 *
 *	Purpose: Write data from the sqlrecord structure to the database
 *
 *************************************************************************/

int sql_save_acct(SQLSOCK *socket, SQLREC *sqlrecord) {

	char		querystr[2048];
	char		fixeduser[AUTH_STRING_LEN*2+1];
	FILE		*mysqlfile;
	int		num = 0;
	int		ret = 1;
#ifdef NT_DOMAIN_HACK
	char		*ptr;
	char		newname[AUTH_STRING_LEN];
#endif
	

	if ((mysqlfile = fopen(QUERYLOG, "a")) == (FILE *)NULL) {
		log(L_ERR, "Acct: Couldn't open file %s", QUERYLOG);
	} else { 
#if defined(F_LOCK) && !defined(BSD)
		(void)lockf((int)mysqlfile, (int)F_LOCK, (off_t)SQL_LOCK_LEN);
#else
		(void)flock(mysqlfile, SQL_LOCK_EX);
#endif
	}

#ifdef NT_DOMAIN_HACK
	/*
	 *      Windows NT machines often authenticate themselves as
	 *      NT_DOMAIN\username. Try to be smart about this.
	 *
	 *      FIXME: should we handle this as a REALM ?
	 */
	if ((ptr = strchr(sqlrecord->UserName, '\\')) != NULL) {
		strncpy(newname, ptr + 1, sizeof(newname));
		newname[sizeof(newname) - 1] = 0;
		strcpy(sqlrecord->UserName, newname);
	}
#endif /* NT_DOMAIN_HACK */

	/* keep garbled usernames from messing up our database queries */
	sql_escape_string(socket, fixeduser, sqlrecord->UserName, strlen(sqlrecord->UserName));
	strNcpy(sqlrecord->UserName, fixeduser, SQLBIGREC);

	if (sqlrecord->AcctStatusTypeId == PW_STATUS_ACCOUNTING_ON || sqlrecord->AcctStatusTypeId == PW_STATUS_ACCOUNTING_OFF) {
		log(L_INFO, "NAS %s rebooted at %s", sqlrecord->NASIPAddress, sqlrecord->AcctTimeStamp);
  
		/* The Terminal server informed us that it was rebooted
		 * STOP all records from this NAS */

		sprintf(querystr, "UPDATE %s SET AcctStopTime='%s', AcctSessionTime=unix_timestamp('%s') - unix_timestamp(AcctStartTime), AcctTerminateCause='%s', AcctStopDelay = %ld WHERE AcctSessionTime=0 AND AcctStopTime=0 AND NASIPAddress= '%s' AND AcctStartTime <= '%s'", sql->config->sql_acct_table, sqlrecord->AcctTimeStamp, sqlrecord->AcctTimeStamp, sqlrecord->AcctTerminateCause, sqlrecord->AcctDelayTime, sqlrecord->NASIPAddress, sqlrecord->AcctTimeStamp);

		if (!sql_query(socket, querystr)) {
			log(L_ERR, "Acct: Couldn't update SQL accounting after NAS reboot - %s", sql_error(socket));
			ret = 0;
		}
		sql_finish_query(socket);

		if (mysqlfile) {
			fputs(querystr, mysqlfile);
			fputs(";\n", mysqlfile);
			fclose(mysqlfile);
		}
		return ret;
	} 

	if (sqlrecord->AcctStatusTypeId == PW_STATUS_ALIVE) {
		sprintf(querystr, "UPDATE %s SET FramedIPAddress = '%s' WHERE AcctSessionId = '%s' AND UserName = '%s' AND NASIPAddress= '%s'", sql->config->sql_acct_table, sqlrecord->FramedIPAddress, sqlrecord->AcctSessionId, sqlrecord->UserName, sqlrecord->NASIPAddress);
		if (!sql_query(socket, querystr)) {
			log(L_ERR, "Acct: Couldn't update SQL accounting for ALIVE packet - %s", sql_error(socket));
			ret = 0;
		}
		sql_finish_query(socket);

		if (mysqlfile) {
			fputs(querystr, mysqlfile);
			fputs(";\n", mysqlfile);
			fclose(mysqlfile);
		}
		return ret;
	}

	/* Got start record */
	if(sqlrecord->AcctStatusTypeId == PW_STATUS_START) {
             
		/* Set start time on record with only a stop record */
		snprintf(querystr, 2048, "UPDATE %s SET AcctStartTime = '%s', AcctStartDelay = %ld WHERE AcctSessionId = '%s' AND UserName = '%s' AND NASIPAddress = '%s' AND NASPortId = '%ld'", 
		sql->config->sql_acct_table,
		sqlrecord->AcctTimeStamp,
		sqlrecord->AcctDelayTime,
		sqlrecord->AcctSessionId,
		sqlrecord->UserName,
		sqlrecord->NASIPAddress,
		sqlrecord->NASPortId
		);

		if (!sql_query(socket, querystr)) {
			log(L_ERR, "Acct: Couldn't update SQL accounting START record - %s", sql_error(socket));
			ret = 0;
		}
		sql_finish_query(socket);

		num = sql_affected_rows(socket);
		if (num == 0 && ret > 0) {

			/* Insert new record with blank stop time until we receive the stop record */
			snprintf(querystr, 2048, "INSERT INTO %s VALUES (0, '%s', '%s', '%s', '%s', %ld, '%s', '%s', 0, 0, '%s', '%s', 0, 0, '%s', '%s', '', '%s', '%s', '%s', %ld, 0)",
			sql->config->sql_acct_table,
			sqlrecord->AcctSessionId,
			sqlrecord->UserName,
			sqlrecord->Realm,
			sqlrecord->NASIPAddress,
			sqlrecord->NASPortId,
			sqlrecord->NASPortType,
			sqlrecord->AcctTimeStamp,
			sqlrecord->AcctAuthentic,
			sqlrecord->ConnectInfo,
			sqlrecord->CalledStationId,
			sqlrecord->CallingStationId,
			sqlrecord->ServiceType,
			sqlrecord->FramedProtocol,
			sqlrecord->FramedIPAddress,
			sqlrecord->AcctDelayTime
			);                  
	
			if (!sql_query(socket, querystr)) {
				log(L_ERR, "Acct: Couldn't insert SQL accounting START record - %s", sql_error(socket));
				ret = 0;
			}
			sql_finish_query(socket);
		}

           /* Got stop record */
	} else if (sqlrecord->AcctStatusTypeId == PW_STATUS_STOP) {

		char	authstr[(AUTH_STRING_LEN * 2) + 50];

		if (sql->config->sensitiveusername) 
			sprintf(authstr, "UserName = '%s' AND STRCMP(UserName, '%s') = 0", sqlrecord->UserName, sqlrecord->UserName);
		else
			sprintf(authstr, "UserName = '%s'", sqlrecord->UserName);

		sprintf(querystr, "SELECT RadAcctId FROM %s WHERE AcctSessionId='%s' AND NASIPAddress='%s' AND %s", sql->config->sql_acct_table, sqlrecord->AcctSessionId, sqlrecord->NASIPAddress, authstr);
		sql_select_query(socket, querystr);
		num = sql_num_rows(socket);
		sql_finish_select_query(socket);

		if (num > 0) {

			/* Set stop time on matching record with start time */
			snprintf(querystr, 2048, "UPDATE %s SET AcctStopTime = '%s', AcctSessionTime = '%lu', AcctInputOctets = '%lu', AcctOutputOctets = '%lu', AcctTerminateCause = '%s', AcctStopDelay = %ld WHERE AcctSessionId = '%s' AND UserName = '%s' AND NASIPAddress = '%s' AND NASPortId = '%ld'", 
			sql->config->sql_acct_table,
			sqlrecord->AcctTimeStamp,
			sqlrecord->AcctSessionTime,
			sqlrecord->AcctInputOctets,
			sqlrecord->AcctOutputOctets,
			sqlrecord->AcctTerminateCause,
			sqlrecord->AcctDelayTime,
			sqlrecord->AcctSessionId,
			sqlrecord->UserName,
			sqlrecord->NASIPAddress,
			sqlrecord->NASPortId
			);

			if (!sql_query(socket, querystr)) {
				log(L_ERR, "Acct: Couldn't update SQL accounting STOP record - %s", sql_error(socket));
				ret = 0;
			}
			sql_finish_query(socket);

		} else if (num == 0) {

#ifdef CISCO_ACCOUNTING_HACK
			/* If stop but zero session length AND no previous session found, drop it as in invalid packet
			 * This is to fix CISCO's aaa from filling our table with bogus crap */
			if (sqlrecord->AcctSessionTime <= 0) {
				log(L_ERR, "Acct: Invalid STOP record. [%s] STOP record but zero session length? (nas %s)", sqlrecord->UserName, sqlrecord->NASIPAddress);
				if(mysqlfile) {
					fclose(mysqlfile);
				}
				/* Return 1 so that we send a ACK to the NAS to keep it from resending over and over */
				return 1;
			}
#endif /* CISCO_ACCOUNTING_HACK */
            
			/* Insert record with no start time until matching start record comes */
			snprintf(querystr, 2048, "INSERT INTO %s VALUES (0, '%s', '%s', '%s', '%s', %ld, '%s', 0, '%s', '%lu', '%s', '%s', '%lu', '%lu', '%s', '%s', '%s', '%s', '%s', '%s', 0, %ld)",
			sql->config->sql_acct_table,
			sqlrecord->AcctSessionId,
			sqlrecord->UserName,
			sqlrecord->Realm,
			sqlrecord->NASIPAddress,
			sqlrecord->NASPortId,
			sqlrecord->NASPortType,
			sqlrecord->AcctTimeStamp,
			sqlrecord->AcctSessionTime,
			sqlrecord->AcctAuthentic,
			sqlrecord->ConnectInfo,
			sqlrecord->AcctInputOctets,
			sqlrecord->AcctOutputOctets,
			sqlrecord->CalledStationId,
			sqlrecord->CallingStationId,
			sqlrecord->AcctTerminateCause,
			sqlrecord->ServiceType,
			sqlrecord->FramedProtocol,
			sqlrecord->FramedIPAddress,
			sqlrecord->AcctDelayTime
			);                  

			if (!sql_query(socket, querystr)) {
				log(L_ERR, "Acct: Couldn't insert SQL accounting STOP record - %s", sql_error(socket));
				ret = 0;
			}
			sql_finish_query(socket);
		}

	} else {
		log(L_ERR, "Acct: Unknown Accounting status type %d", sqlrecord->AcctStatusTypeId);
		return 0;
	}

	if (mysqlfile) {
		fputs(querystr, mysqlfile);
		fputs(";\n", mysqlfile);
		fflush(mysqlfile);
		fclose(mysqlfile);
	}

	return ret;
}


/*************************************************************************
 *
 *	Function: sql_userparse
 *
 *	Purpose: Read entries from the database and fill VALUE_PAIR structures
 *
 *************************************************************************/
int sql_userparse(VALUE_PAIR **first_pair, SQL_ROW row, int mode) {

	int		x, ufound, gfound, hintsfound;
	char		*s;
	DICT_ATTR	*attr = NULL;
	DICT_VALUE	*dval;
	VALUE_PAIR	*pair, *pair2, *check;
	struct tm	*tm;
	time_t		timeval;
#if defined( BINARY_FILTERS )
	char		valstr[256];
#endif	/* BINARY_FILTERS */


	if((attr = dict_attrfind(row[2])) == (DICT_ATTR *)NULL) {
		log(L_ERR|L_CONS, "unknown attribute %s for user %s", row[2], row[1]);
		return(-1);
	}                              

	/* If attribute is already there, skip it because we checked usercheck first 
	   and we want user settings to over ride group settings */
	check = *first_pair;
	ufound = 0;
	gfound = 0;
	hintsfound = 0;
	while (check) {
		if (attr->value == check->attribute) {
			if (check->source == PW_VP_HINTSDATA || mode == PW_VP_HINTSDATA)
				hintsfound = 1;
			if (check->source == PW_VP_USERDATA || mode == PW_VP_USERDATA)
				ufound = 1;
			if (check->source == PW_VP_GROUPDATA || mode == PW_VP_GROUPDATA)
				gfound = 1;
		}
		
		/* HINTS items ALWAYS replace user or group items */
		if (hintsfound && (ufound || gfound)) {
		   DEBUG2("attribute %s for user %s already in hints", row[2], row[1]);
			return 1;
                }

		if (ufound && gfound) {
		   DEBUG2("attribute %s for user %s already ufound & gfound ", row[2], row[1]);
			return 1;
		}

		check = check->next;
	}

	if((pair = (VALUE_PAIR *)malloc(sizeof(VALUE_PAIR))) == (VALUE_PAIR *)NULL) {
		log(L_CONS|L_ERR, "mysql_userparse: no memory");
		return -2;
	}
	strcpy(pair->name, attr->name);
	pair->attribute = attr->value;
	pair->type = attr->type;
	pair->operator = PW_OPERATOR_EQUAL;
	pair->source = mode;
	switch(pair->type) {

#if defined( BINARY_FILTERS )
		case PW_TYPE_FILTER_BINARY:
			/*
			* special case to convert filter to binary
			*/
			strcpy(valstr, row[3]);
			if ( filterBinary( pair, valstr ) == -1 ) {
				free(pair);
				return(-1);
			}
			break;
#endif	/* BINARY_FILTERS */

		case PW_TYPE_STRING:
			strcpy(pair->strvalue, row[3]);
			pair->length = strlen(pair->strvalue);
			break;

		case PW_TYPE_INTEGER:
                       /*
                        *      For PW_NAS_PORT_ID, allow a
                        *      port range instead of just a port.
                        */
                        if (attr->value == PW_NAS_PORT_ID) {
                              for(s = row[3]; *s; s++)
                                   if (!isdigit(*s)) break;
                                   if (*s) {
                                       pair->type = PW_TYPE_STRING;
                                       strcpy(pair->strvalue, row[3]);
                                       pair->length = strlen(pair->strvalue);
                                       break;
                                   }
                        }
                        if (isdigit(*row[3])) {
                                   pair->lvalue = strtol(row[3], NULL, 0);
                                   pair->length = 4;
                        }
                        else if((dval = dict_valfind(row[3])) == (DICT_VALUE *)NULL) {
                                   free(pair);
                                   log(L_ERR|L_CONS, "unknown value %s", row[3]);
                                   return(-1);
                        }
                        else {
                                   pair->lvalue = dval->value;
                                   pair->length = 4;
                        }
                        break;

		case PW_TYPE_IPADDR:
			if (pair->attribute != PW_FRAMED_IP_ADDRESS) {
                                   pair->lvalue = get_ipaddr(row[3]);
                                   break;
                        }

                       /*
                        *      We allow a "+" at the end to
                        *      indicate that we should add the
                        *      portno. to the IP address.
                        */
                        x = 0;
                        if (row[3][0]) {
                               for(s = row[3]; s[1]; s++) ;
                                    if (*s == '+') {
                                        *s = 0;
                                        x = 1;
                                    }
                        }
                        pair->lvalue = get_ipaddr(row[3]);
                        pair->length = 4;

                       /*
                        *      Add an extra (hidden) attribute.
                        */
                        if((pair2 = malloc(sizeof(VALUE_PAIR))) == NULL) {
                               log(L_CONS|L_ERR, "no memory");
                               return -2;
                        }
                        strcpy(pair2->name, "Add-Port-To-IP-Address");
                        pair2->attribute = PW_ADD_PORT_TO_IP_ADDRESS;
                        pair2->type = PW_TYPE_INTEGER;
                        pair2->lvalue = x;
                        pair2->length = 4;
                        pairadd(first_pair, pair2);
                        break;

		case PW_TYPE_DATE:
                        timeval = time(0);
                        tm = localtime(&timeval);
                        user_gettime(row[3], tm);
#ifdef TIMELOCAL
                        pair->lvalue = (UINT4)timelocal(tm);
#else
                        pair->lvalue = (UINT4)mktime(tm);
#endif
                        pair->length = 4;
                        break;

		default:
                        log(L_ERR|L_CONS, "unknown attr. type %d", pair->type);
                        free(pair);
                        return(-1);
	}
	pairadd(first_pair, pair);

	return 1;
}


/*************************************************************************
 *
 *	Function: sql_getvpdata
 *
 *	Purpose: Get any group check or reply pairs
 *
 *************************************************************************/
int sql_getvpdata(SQLSOCK *socket, char *table, VALUE_PAIR **vp, char *user, int mode) {

	char		*querystr;
	char		*authstr = NULL;
	char		*username;
	SQL_ROW		row;
	int		rows;
	int		length;

	if (strlen(user) > AUTH_STRING_LEN)
		length = AUTH_STRING_LEN;
	else
		length = strlen(user);

	if ((username = malloc(length* 2 + 1)) == NULL) {
		log(L_CONS|L_ERR, "out of memory");
		return -1;
	}

	sql_escape_string(socket, username, user, length);

	if (mode == PW_VP_USERDATA) {
		if ((authstr = malloc((strlen(username) * 2) + 50)) == NULL) {
			free(username);
			log(L_CONS|L_ERR, "out of memory");
			return -1;
		}
		if (sql->config->sensitiveusername) 
			sprintf(authstr, "UserName = '%s' AND STRCMP(Username, '%s') = 0", username, username);
		else
			sprintf(authstr, "UserName = '%s'", username);
		if ((querystr = malloc(strlen(authstr) + strlen(table) + 70)) == NULL) {
			free(authstr);
			free(username);
			log(L_CONS|L_ERR, "out of memory");
			return -1;
		}
		sprintf(querystr, "SELECT id, UserName, Attribute, Value FROM %s WHERE %s ORDER BY id", table, authstr);

	} else if (mode == PW_VP_GROUPDATA) {
		if ((authstr = malloc((strlen(username) * 2) + strlen(sql->config->sql_usergroup_table) + 55)) == NULL) {
			free(username);
			log(L_CONS|L_ERR, "out of memory");
			return -1;
		}
		if (sql->config->sensitiveusername) 
			sprintf(authstr, "%s.UserName = '%s' AND STRCMP(%s.Username, '%s') = 0", sql->config->sql_usergroup_table, username, sql->config->sql_usergroup_table, username);
		else
			sprintf(authstr, "%s.UserName = '%s'",sql->config->sql_usergroup_table, username);
		if ((querystr = malloc(strlen(authstr) + (strlen(table) * 7) + (strlen(sql->config->sql_usergroup_table) * 2) + 125)) == NULL) {
			free(authstr);
			free(username);
			log(L_CONS|L_ERR, "out of memory");
			return -1;
		}
		sprintf(querystr, "SELECT %s.id, %s.GroupName, %s.Attribute, %s.Value FROM %s, %s WHERE %s AND %s.GroupName = %s.GroupName ORDER BY %s.id", table, table, table, table, table, sql->config->sql_usergroup_table, authstr, sql->config->sql_usergroup_table, table, table);

	} else if (mode == PW_VP_REALMDATA) {
		if ((querystr = malloc(strlen(username) + (strlen(table) * 7) + (strlen(sql->config->sql_realmgroup_table) * 3) + 125)) == NULL) {
			free(username);
			log(L_CONS|L_ERR, "out of memory");
			return -1;
		}
		sprintf(querystr, "SELECT %s.id, %s.GroupName, %s.Attribute, %s.Value FROM %s, %s WHERE %s.RealmName = '%s' AND %s.GroupName = %s.GroupName ORDER BY %s.id", table, table, table, table, table, sql->config->sql_realmgroup_table, sql->config->sql_realmgroup_table, username, sql->config->sql_realmgroup_table, table, table);
	} else if (mode == PW_VP_HINTSDATA) {
		if ((querystr = malloc(strlen(username) + strlen(table) + 125)) == NULL) {
			free(authstr);
			free(username);
			log(L_CONS|L_ERR, "out of memory");
			return -1;
		}
		sprintf(querystr, "SELECT id, GroupName, Attribute, Value FROM %s WHERE GroupName = '%s' ORDER BY id", table, username);
		
	}

	sql_select_query(socket, querystr);
	rows = sql_num_rows(socket);
	while ((row = sql_fetch_row(socket))) {

		if (!sql_userparse(vp, row, mode)) {
	 		log(L_ERR|L_CONS, "sql_userparse: [%s] Error adding attribute [%s] with value [%s] to pair list", row[1], row[2], row[3]);
			sql_finish_select_query(socket);
			free(querystr);
			if (authstr) free(authstr);
			free(username);
			return -1;
		}
	}
	sql_finish_select_query(socket);
	free(querystr);
	if (authstr) free(authstr);
	free(username);

	return rows;
}


static int got_alrm;
static void alrm_handler()
{
	got_alrm = 1;
}

/*************************************************************************
 *
 *	Function: sql_check_ts
 *
 *	Purpose: Checks the terminal server for a spacific login entry
 *
 *************************************************************************/
static int sql_check_ts(SQL_ROW row) {

	int     pid, st, e;
	int     n;
	NAS     *nas;
	char    session_id[12];
	char    *s;
	void    (*handler)(int);

	/*
	 *      Find NAS type.
	 */
	if ((nas = nas_find(ipstr2long(row[3]))) == NULL) {
                log(L_ERR, "Accounting: unknown NAS [%s]", row[3]);
                return -1;
        }

	if (!nas->usesnmp)
		return 1;

        /*
         *      Fork.
         */
        handler = signal(SIGCHLD, SIG_DFL);
        if ((pid = fork()) < 0) {
                log(L_ERR, "Accounting: fork: %s", strerror(errno));
                signal(SIGCHLD, handler);
                return -1;
        }

        if (pid > 0) {
                /*
                 *      Parent - Wait for checkrad to terminate.
                 *      We timeout in 10 seconds.
                 */
                got_alrm = 0;
                signal(SIGALRM, alrm_handler);
                alarm(10);
                while((e = waitpid(pid, &st, 0)) != pid)
                        if (e < 0 && (errno != EINTR || got_alrm))
                                break;
                alarm(0);
                signal(SIGCHLD, handler);
                if (got_alrm) {
                        kill(pid, SIGTERM);
                        sleep(1);
                        kill(pid, SIGKILL);
                        log(L_ERR, "Check-TS: timeout waiting for checkrad");
                        return 2;
                }
                if (e < 0) {
                        log(L_ERR, "Check-TS: unknown error in waitpid()");
                        return 2;
                }
                return WEXITSTATUS(st);
        }

        /*
         *      Child - exec checklogin with the right parameters.
         */
        for (n = 32; n >= 3; n--)
                close(n);

        sprintf(session_id, "%.8s", row[1]);

        s = CHECKRAD2;
        execl(CHECKRAD2, "checkrad", nas->nastype, row[3], row[4],
                row[2], session_id, nas->community, NULL);
        if (errno == ENOENT) {
                s = CHECKRAD1;
                execl(CHECKRAD1, "checklogin", nas->nastype, row[3], row[4],
                        row[2], session_id, nas->community, NULL);
        }
        log(L_ERR, "Check-TS: exec %s: %s", s, strerror(errno));

        /*
         *      Exit - 2 means "some error occured".
         */
        exit(2); 

}


/*************************************************************************
 *
 *	Function: sql_check_multi
 *
 *	Purpose: Check radius accounting for duplicate logins
 *
 *************************************************************************/
int sql_check_multi(SQLSOCK *socket, char *name, VALUE_PAIR *request, int maxsimul) {

	char		querystr[256];
	char		authstr[256];
	VALUE_PAIR	*fra;
	SQL_ROW		row;
	int		count = 0;
	UINT4		ipno = 0;
	int		mpp = 1;

	if (sql->config->sensitiveusername)
		sprintf(authstr, "UserName = '%s' AND STRCMP(UserName, '%s') = 0", name, name);
	else
		sprintf(authstr, "UserName = '%s'", name);
	sprintf(querystr, "SELECT COUNT(*) FROM %s WHERE AcctStopTime = 0 AND %s", sql->config->sql_acct_table, authstr);
	sql_select_query(socket, querystr);
	row = sql_fetch_row(socket);
	count = strtol(row[0], NULL, 0);
	sql_finish_select_query(socket);

	if (count < maxsimul)
		return 0;

	/*
	*      Setup some stuff, like for MPP detection.
	*/
	if ((fra = pairfind(request, PW_FRAMED_IP_ADDRESS)) != NULL)
		ipno = htonl(fra->lvalue);

	count = 0;
	sprintf(querystr, "SELECT RadAcctId, AcctSessionId, UserName, NASIPAddress, NASPortId, FramedIPAddress FROM %s WHERE AcctStopTime = 0 AND %s", sql->config->sql_acct_table, authstr);
	sql_select_query(socket, querystr);
	while ((row = sql_fetch_row(socket))) {
		int check = sql_check_ts(row);
		if (check == 1) {
			count++;

			if (ipno && atoi(row[5]) == ipno)
				mpp = 2;   

		} else if (check == 2)
			log(L_ERR,"Problem with checkrad [%s] (from nas %s)", name, row[3]);
		else {
			/*
			 *	False record - zap it
			 */

			if (sql->config->deletestalesessions) {
				SQLSOCK *sock;
				pid_t	chldpid;

				if ((chldpid = fork()) == 0) {
					sock = sql_create_socket();
					sprintf(querystr, "DELETE FROM %s WHERE RadAcctId = '%s'", sql->config->sql_acct_table, row[0]);
					if (!sql_query(sock, querystr)) {
						log(L_ERR, "Could not delete stale session '%s'", row[0]);
					} else {
						sql_finish_query(sock);
					}
					sql_close_socket(sock);
					exit(1);
				} else {
	   				log(L_ERR,"Deleteing stale session [%s] (child pid %d) (from nas %s/%s)", row[2], chldpid, row[3], row[4]);
				}
			}
		}
	}
	sql_finish_select_query(socket);

	return (count < maxsimul) ? 0 : mpp; 

}

/*
 *      Read the hints table
 */
PAIR_LIST  *sql_read_hints(SQLSOCK *socket) {
	char		querystr[256];
	PAIR_LIST	*c, *last = NULL, *pl = NULL;
	VALUE_PAIR	*check_tmp = NULL;
	VALUE_PAIR	*reply_tmp = NULL;
	SQL_ROW		row;
	SQLSOCK		*hintsock;

	hintsock = sql_get_socket();


	sprintf(querystr, "SELECT id, HintName, GroupName FROM %s", sql->config->sql_hints_table);
	sql_select_query(socket, querystr);
	while ((row = sql_fetch_row(socket))) {
		if ((c = malloc(sizeof(PAIR_LIST))) == NULL) {
			log(L_CONS|L_ERR, "out of memory");
			exit(1);
		}

		c->next = NULL;

		/* Why the HELL was strdup(row[1]) not working?! */
		if ((c->name = malloc(strlen(row[1]) + 1)) == NULL) {
			log(L_CONS|L_ERR, "out of memory");
			exit(1);
		}
		strcpy(c->name, row[1]);

		c->lineno = atoi(row[0]);

		sql_getvpdata(hintsock, sql->config->sql_groupcheck_table, &check_tmp, row[2], PW_VP_HINTSDATA);
		sql_getvpdata(hintsock, sql->config->sql_groupreply_table, &reply_tmp, row[2], PW_VP_HINTSDATA);

		c->check = check_tmp;
		c->reply = reply_tmp;
		check_tmp = NULL;
		reply_tmp = NULL;

#if 0
		debug_pair_list(c);
#endif

		if (last)
			last->next = c;
		else
			pl = c;
		last = c;
       	}
	sql_finish_select_query(socket);

	sql_release_socket(hintsock);

        return pl;
} 


/*
 *      Read the nas table
 */
int sql_read_naslist(SQLSOCK *socket) {
	char		querystr[256];
        NAS     	*n;
	CLIENT		*c;
	SQL_ROW		row;

        clients_free(clients);
        clients = NULL;

        nas_free(naslist);
        naslist = NULL;

	sprintf(querystr, "SELECT id, nasname, shortname, ipaddr, type, ports, secret, community, snmp FROM %s", sql->config->sql_nas_table);
	sql_select_query(socket, querystr);
	while ((row = sql_fetch_row(socket))) {
		if ((n = malloc(sizeof(NAS))) == NULL) {
			log(L_CONS|L_ERR, "out of memory");
			return -1;
		}
		if ((c = malloc(sizeof(CLIENT))) == NULL) {
			log(L_CONS|L_ERR, "out of memory");
			return -1;
		}

		c->secret[0] = 0;
		c->longname[0] = 0;
		c->shortname[0] = 0;

		n->longname[0] = 0;
		n->shortname[0] = 0;
		n->community[0] = 0;
		n->usesnmp = 0;

               	n->ipaddr = ipstr2long(row[3]);
               	c->ipaddr = ipstr2long(row[3]);

		if (row[1] != NULL) {
               		strNcpy(n->longname, row[1], sizeof(n->longname));
               		strNcpy(c->longname, row[1], sizeof(c->longname));
		}
		if (row[2] != NULL) {
               		strNcpy(n->shortname, row[2], sizeof(n->shortname));
               		strNcpy(c->shortname, row[2], sizeof(c->shortname));
		}
		if (row[4] != NULL)
                	strNcpy(n->nastype, row[4], sizeof(n->nastype));
		if (row[6] != NULL)
                	strNcpy(c->secret, row[6], sizeof(c->secret));
		if (row[7] != NULL)
			strncpy(n->community, row[7], MAX_COMMUNITY_LEN);
		if (row[8] != NULL)	
			n->usesnmp = strcmp(row[8], "ON") == 0 ||
				     strcmp(row[8], "on") == 0;

       	        c->next = clients;
               	clients = c;

       	        n->next = naslist;
               	naslist = n;
       	}
	sql_finish_select_query(socket);

        return 1;
} 


/*
 *	Initialize the dictionary.  Select all ATTRIBUTES into
 *	 the dictionary_attributes list.  Select all VALUES into
 *	 the dictionary_values list.
 */
int sql_dict_init(SQLSOCK *socket) {

	char		vendorstr[64];
	SQL_ROW		row;
	int		line_no;
	DICT_ATTR	*attr;
	DICT_VALUE	*dval;
	DICT_VENDOR	*v;
	char		querystr[256];
	int		value;
	int		type;
	int		vendor;
	int		is_attrib;
#ifdef ATTRIB_NMC
	int		vendor_usr_seen = 0;
	int		is_nmc = 0;
#endif


	dict_free();

	sprintf(querystr, "SELECT * FROM %s ORDER BY id", sql->config->sql_dict_table);
	sql_select_query(socket, querystr);
	while ((row = sql_fetch_row(socket))) {
		if (row[0] == NULL)
			line_no = 0;
		else
			line_no = strtol(row[0], NULL, 0);

		strcpy(vendorstr, row[5]);

		is_attrib = 0;
		if (strncmp(row[1], "ATTRIBUTE", 9) == 0)
			is_attrib = 1;
#ifdef ATTRIB_NMC
		is_nmc = 0;
		if (strncmp(row[1], "ATTRIB_NMC", 10) == 0)
			is_attrib = is_nmc = 1;
#endif
		if (is_attrib) {
			/* Read the ATTRIBUTE line */
			vendor = 0;

#ifdef ATTRIB_NMC
			/*
			 *	Convert ATTRIB_NMC into our format.
			 *	We might need to add USR to the list of
			 *	vendors first.
			 */
			if (is_nmc && vendorstr == 0) {
				if (!vendor_usr_seen) {
					if (addvendor("USR", VENDORPEC_USR) < 0)
						return -1;
					vendor_usr_seen = 1;
				}
				strcpy(vendorstr, "USR");
			}
#endif

			/*
			 * Validate all entries
			 */
			if(strlen(row[2]) > 31) {
				log(L_ERR|L_CONS,
		"dict_init: Invalid name length on line %d of dictionary",
					line_no);
				sql_finish_select_query(socket);
				return(-1);
			}

			if(!isdigit(*row[3])) {
				log(L_ERR|L_CONS,
			"dict_init: Invalid value on line %d of dictionary",
					line_no);
				sql_finish_select_query(socket);
				return(-1);
			}
			if (row[3][0] != '0')
				value = atoi(row[3]);
			else
				sscanf(row[3], "%i", &value);

			if(strcmp(row[4], "string") == 0) {
				type = PW_TYPE_STRING;
			}
			else if(strcmp(row[4], "integer") == 0) {
				type = PW_TYPE_INTEGER;
			}
			else if(strcmp(row[4], "ipaddr") == 0) {
				type = PW_TYPE_IPADDR;
			}
			else if(strcmp(row[4], "date") == 0) {
				type = PW_TYPE_DATE;
			}
#if defined( BINARY_FILTERS )
			else if(strcmp(row[4], "abinary") == 0) {
				type = PW_TYPE_FILTER_BINARY;
			}
#endif	/* BINARY_FILTERS */
			else {
				log(L_ERR|L_CONS,
			"dict_init: Invalid type on line %d of dictionary",
					line_no);
				sql_finish_select_query(socket);
				return(-1);
			}

			for (v = dictionary_vendors; v; v = v->next) {
				if (strcmp(vendorstr, v->vendorname) == 0)
					vendor = v->vendorcode;
			}
			if (vendorstr[0] && !vendor) {
				log(L_ERR|L_CONS,
			"dict_init: unknown vendor %s on line %d of dictionary",
				vendorstr, line_no);
				sql_finish_select_query(socket);
				return -1;
			}

			/* Create a new attribute for the list */
			if((attr = (DICT_ATTR *)malloc(sizeof(DICT_ATTR))) ==
					(DICT_ATTR *)NULL) {
				log(L_ERR|L_CONS, "dict_init: out of memory");
				sql_finish_select_query(socket);
				return(-1);
			}
			strcpy(attr->name, row[2]);
			attr->value = value;
			attr->type = type;
			if (vendor)
				attr->value |= (vendor << 16);

			/*
			 *	Add to the front of the list, so that
			 *	values at the end of the file override
			 *	those in the begin.
			 */
			attr->next = dictionary_attributes;
			dictionary_attributes = attr;

		}
		else if (strncmp(row[1], "VALUE", 5) == 0) {


			/*
			 * Validate all entries
			 */
			if(strlen(row[2]) > 31) {
				log(L_ERR|L_CONS, "dict_init: Invalid attribute length on line %d of dictionary", line_no);
				sql_finish_select_query(socket);
				return(-1);
			}

			if(strlen(row[3]) > 31) {
				log(L_ERR|L_CONS, "dict_init: Invalid name length on line %d of dictionary", line_no);
				sql_finish_select_query(socket);
				return(-1);
			}

			if(!isdigit(*row[4])) {
				log(L_ERR|L_CONS, "dict_init: Invalid value on line %d of dictionary", line_no);
				sql_finish_select_query(socket);
				return(-1);
			}
			value = strtol(row[4], NULL, 0);

			/* Create a new VALUE entry for the list */
			if((dval = (DICT_VALUE *)malloc(sizeof(DICT_VALUE))) == (DICT_VALUE *)NULL) {
				log(L_ERR|L_CONS, "dict_init: out of memory");
				sql_finish_select_query(socket);
				return(-1);
			}
			strcpy(dval->attrname, row[2]);
			strcpy(dval->name, row[3]);
			dval->value = value;

			/* Insert at front. */
			dval->next = dictionary_values;
			dictionary_values = dval;
		}
		else if(strncmp(row[1], "VENDOR", 6) == 0) {


			/*
			 * Validate all entries
			 */
			if(strlen(row[2]) > 31) {
				log(L_ERR|L_CONS, "dict_init: Invalid attribute length on line %d of dictionary", line_no);
				sql_finish_select_query(socket);
				return(-1);
			}

			if(!isdigit(*row[3])) {
				log(L_ERR|L_CONS, "dict_init: Invalid value on line %d of dictionary", line_no);
				sql_finish_select_query(socket);
				return(-1);
			}
			value = strtol(row[3], NULL, 0);

			/* Create a new VENDOR entry for the list */
			if (addvendor(row[2], value) < 0) {
				sql_finish_select_query(socket);
				return -1;
			}
#ifdef ATTRIB_NMC
			if (value == VENDORPEC_USR)
				vendor_usr_seen = 1;
#endif
		}
	}
	sql_finish_select_query(socket);
	return(1);
}

 /*
  *    Read the realms table.
  *
  */
int sql_read_realms(SQLSOCK *socket) {

   char		querystr[2048];
   SQL_ROW	row;
   char opts1[32];
   char opts2[32];
   REALM   *c;
   
   realm_free(realms);
   realms = NULL;
   
   /* Maybe realm.nas should refer exact to either nas.nasname 
      or nas.shortname. Either way, GROUP BY makes the realmnames 
      unique - CS */
   sprintf(querystr, "SELECT realmname, nasname, ipaddr, authport, options FROM %s, %s WHERE nas = nasname OR nas = shortname or nas = ipaddr GROUP BY realmname", sql->config->sql_realm_table, sql->config->sql_nas_table);
   
   sql_select_query(socket, querystr);
   
   while ((row = sql_fetch_row(socket))) {
	if ((c = malloc(sizeof(REALM))) == NULL) {
		log(L_CONS|L_ERR, "out of memory");
		return -1;
	}

	opts1[0] = opts2[0] = 0;
	c->realm[0] = 0;
	c->server[0] = 0;
	c->striprealm = 1;
	c->dohints = 0;

	if (c->realm != NULL)
		strNcpy(c->realm, row[0], sizeof(c->realm));
	if (c->server != NULL)
		strNcpy(c->server, row[1], sizeof(c->server));
	if (strcmp(row[2], "") == 0)
		c->ipaddr = get_ipaddr(row[1]);
	else
		c->ipaddr = ipstr2long(row[2]);
	if (strcmp(row[3],"0") == 0)
		c->auth_port = auth_port;
	else
		c->auth_port = strtol(row[3], NULL, 0);
	c->acct_port = c->auth_port + 1;
	if (row[4] != NULL) {
		sscanf(row[4], "%31s%31s", opts1, opts2);

		c->striprealm = strcmp(opts1, "nostrip") != 0 &&
				strcmp(opts2, "nostrip") != 0;
		c->dohints    = strcmp(opts1, "hints") == 0 ||
				strcmp(opts2, "hints") == 0;
 	}

	c->next = realms;
	realms = c;
    }
    sql_finish_select_query(socket);
    return 1;
}

