/*
 * FASTRING - support for zero copy ring buffers using the mirrored page 
 * 	      technique
 *
 * Theory of operation is as follows. The requirement for functions that 
 * access the ring buffer to support wrapping can be completely avoided by
 * following the buffer memory with a writable image of itself. Standard
 * memory access functions can operate directly on messages of the size of the
 * original buffer, without wrapping or copying, regardless of the value of the
 * read- and write pointers.
 *
 * The offered primitives by this module are: functions to get the amount of
 * free space for writing in the ring and to get the number of bytes available
 * for reading; direct access to the absolute read- and write pointers; and
 * functions to update the read and write pointers after getting and putting.
 *
 * Because reading data and updating the read pointer are done separately, the
 * distinction between peeking at the data and getting the data is removed; you
 * just decide whether or not to call the 'get' macro.
 *
 * The 'put' macro also emulates the mirrored memory in case there is no
 * operating system support for mapping the same memory twice in the program's
 * address space.
 *
 * It is required that you don't write more data than there's room; no bounds
 * checking is done by the 'put' function that updates the pointer as the 
 * damage would already be done anyway. The same goes for reading; you need
 * to call the maxget macro to know how many bytes you may access at the
 * read pointer. 
 *
 * Author:
 * Emile van Bergen, emile@e-advies.nl
 *
 * License:
 * Permission to redistribute an original or modified version of this program
 * in source, intermediate or object code form is hereby granted exclusively
 * under the terms of the GNU General Public License, version 2. Please see the
 * file COPYING for details, or refer to http://www.gnu.org/copyleft/gpl.html.
 *
 * History: 
 * 2003/05/08 - EvB - Created
 */


/* 
 * INCLUDES & DEFINES
 */


#include <evblib/fastring/pagemirr.h>

#define ring_capacity(R)  ((R)->mp.l)
#define ring_maxget(R)	  ((R)->maxget)
#define ring_maxput(R)	  ((R)->mp.l - (R)->maxget)
#define ring_empty(R)	  (((R)->maxget = 0), ((R)->r = (R)->w = (R)->mp.p))

#define ring_get(R,len) \
	do { \
	    (R)->r = (R)->mp.p + (((R)->r - (R)->mp.p + (len)) % (R)->mp.l); \
	    (R)->maxget -= (len); \
	} while(0)

#define ring_put(R,len) \
	do { \
	    mirr_update(&(R)->mp, (R)->w - (R)->mp.p, (len)); \
	    (R)->w = (R)->mp.p + (((R)->w - (R)->mp.p + (len)) % (R)->mp.l); \
	    (R)->maxget += (len); \
	} while(0)


/* 
 * TYPES
 */


typedef struct ring {
    struct mirrpage mp;
    char *r, *w;
    size_t maxget;
} RING_T;


/*
 * PROTOTYPES
 */


RING_T *ring_new(size_t size);
void ring_del(RING_T *r);

