/*
 * PAGEMIRROR - Allocates an area of memory that's followed immediately by an
 *	        image of itself. This way, normal string functions can operate
 *	        on ring buffers, without having to explicitly support wrapping.
 *
 * Author:
 * Emile van Bergen, emile@evbergen.xs4all.nl
 *
 * Permission to redistribute an original or modified version of this program
 * in source, intermediate or object code form is hereby granted exclusively
 * under the terms of the GNU General Public License, version 2. Please see the
 * file COPYING for details, or refer to http://www.gnu.org/copyleft/gpl.html.
 *
 * History:
 * 2003/05/03 - EvB - Created
 */

char pagemirror_id[] = "PAGEMIRROR - Copyright (C) 2003 Emile van Bergen.";


/*
 * INCLUDES & DEFINES
 */


#include <unistd.h>			    /* for open, close */
#include <fcntl.h>			    /* for O_RDWR et al. */
#include <sys/types.h>			    /* for mode_t, size_t */
#include <string.h>			    /* for strerror */
#include <errno.h>			    /* for errno */

#include <evblib/fastring/pagemirr.h>

#include <evblib/misc/pagesize.h>
#include <evblib/misc/misc.h>		    /* for get_random_data */
#include <evblib/misc/debug.h>		    /* for msg */


#if defined (HAVE_POSIXSHM) || defined (HAVE_DEVZEROSHM)
#include <sys/mman.h>			    /* for shm_open, mmap */
#elif defined (HAVE_SYSVSHM)
#include <sys/ipc.h>
#include <sys/shm.h>
#else
#include <stdlib.h>	/* MALLOC */			    /* for malloc */
#endif	/* HAVE_ANY */


/*
 * FUNCTIONS
 */


#ifdef HAVE_POSIXSHM

int mirr_posixshmopen(size_t size)
{
    static char fname[64];	/* 64 * 4 = 256 random bits should be enough */
    char *p;
    int n, fd;

    fname[0] = '/'; fname[sizeof(fname) - 1] = 0;
    do {
	p = fname + 1; n = sizeof(fname) - 1;
	get_random_data(p, n); for(; n > 0; n--, p++) *p = 'E' + (*p & 0xf);

	fd = shm_open(fname, O_RDWR|O_CREAT|O_EXCL, 0600);
	if (fd != -1) {
	    shm_unlink(fname);
	    if (ftruncate(n, size) != -1) return fd;
	    msg(F_MISC, L_ERR, "mirr_new: ERROR: Cannot set size of posix shm segment to %lu: %s!\n", (unsigned long)size, (strerror(errno))); 
	    close(fd);
	    return -1;
	}
    } 
    while(errno == EEXIST);
    msg(F_MISC, L_ERR, "mirr_new: ERROR: Cannot shm_open: %s!\n", strerror(errno)); 
    return -1;
}

#endif


struct mirrpage *mirr_new(size_t size, struct mirrpage *ret)
{
#if defined (HAVE_POSIXSHM) || defined (HAVE_DEVZEROSHM) || defined (HAVE_SYSVSHM)
    static size_t mirr_pagesize = 0;
    void *p;
    int fd;

    if (!mirr_pagesize) mirr_pagesize = GETPAGESIZE;
    ret->l = ((size + mirr_pagesize - 1) / mirr_pagesize) * mirr_pagesize;
    /* msg(F_MISC, L_ERR, "mirr_new(%d) called, using %d\n", size, ret->l); */
#endif

#if defined (HAVE_POSIXSHM)
    fd = mirr_posixshmopen(ret->l * 2);
    if (fd == -1) return 0;

#elif defined (HAVE_DEVZEROSHM)
    fd = open("/dev/zero", O_RDWR);
    if (fd == -1) { msg(F_MISC, L_ERR, "mirr_new: ERROR: Cannot open /dev/zero: %s!\n", strerror(errno)); return 0; }

#endif	
  
#if defined (HAVE_POSIXSHM) || defined (HAVE_DEVZEROSHM)

    /* Find address at which the double-sized segment fits, in order
       to know where the wanted size fits twice. */

    p = mmap(0, ret->l * 2, PROT_READ|PROT_WRITE, MAP_SHARED, fd, 0);
    if (p == MAP_FAILED) { msg(F_MISC, L_ERR, "mirr_new: ERROR: Cannot mmap: %s!\n", strerror(errno)); close(fd); return 0; }
    munmap(p, ret->l * 2);

    /* Now map the real size, twice. There's no syscall in between that
       can have altered our process' VM map, so this should work. We're
       not using threads and don't use malloc in signal handlers. */

    ret->p = mmap(p, ret->l, PROT_READ|PROT_WRITE, MAP_SHARED|MAP_FIXED, fd, 0);
    if (ret->p == p) { msg(F_MISC, L_ERR, "mirr_new: ERROR: Cannot mmap: %s!\n", strerror(errno)); close(fd); return 0; }
    p = mmap(ret->p + ret->l, ret->l, PROT_READ|PROT_WRITE, MAP_SHARED|MAP_FIXED, fd, 0);
    if (p != ret->p + ret->l) { msg(F_MISC, L_ERR, "mirr_new: ERROR: Cannot mmap: %s!\n", strerror(errno)); close(fd); return 0; }

    close(fd);

#elif defined (HAVE_SYSVSHM)

    /* Obtain address where we can map the requested size twice */

    fd = shmget(IPC_PRIVATE, ret->l * 2, 0600);
    if (fd == -1) { msg(F_MISC, L_ERR, "mirr_new: ERROR: Cannot get sysv shared memory: %s\n", strerror(errno)); return 0; }
    p = shmat(fd, 0, 0);
    if (p == 0 || p == (void *)-1) { msg(F_MISC, L_ERR, "mirr_new: ERROR: Cannot attach sysv shm segment: %s\n", strerror(errno)); shmctl(fd, IPC_RMID, 0); return 0; }
    shmctl(fd, IPC_RMID, 0); 
    shmdt(p);

    /* Do it again, for real this time */

    fd = shmget(IPC_PRIVATE, ret->l, 0600);
    if (fd == -1) { msg(F_MISC, L_ERR, "mirr_new: ERROR: Cannot get sysv shared memory: %s\n", strerror(errno)); return 0; }

    ret->p = shmat(fd, p, 0);
    if (ret->p != p) { msg(F_MISC, L_ERR, "mirr_new: ERROR: Cannot attach sysv shm segment: %s\n", strerror(errno)); if (ret->p) shmdt(ret->p); shmctl(fd, IPC_RMID, 0); return 0; }

    p = shmat(fd, ret->p + ret->l, 0);
    if (p != ret->p + ret->l) { 
	msg(F_MISC, L_ERR, "mirr_new: ERROR: Cannot reattach sysv shm segment at %p (%p + %x): %s\n", ret->p + ret->l, ret->p, ret->l, strerror(errno)); 
	if (p) shmdt(p);
	shmdt(ret->p); 
	shmctl(fd, IPC_RMID, 0); 
	return 0; 
    }
    shmctl(fd, IPC_RMID, 0); 

#else				/* No shared memory at all */
    ret->l = size;
    ret->p = malloc(size << 1);
    if (!ret->p) ret = 0;

#endif

    return ret;
}


#if defined (HAVE_POSIXSHM) || defined (HAVE_DEVZEROSHM)

void mirr_del(struct mirrpage *mp)
{
    munmap(mp->p, mp->l);
    munmap(mp->p + mp->l, mp->l);
}

#elif defined (HAVE_SYSVSHM)

void mirr_del(struct mirrpage *mp)
{
    shmdt(mp->p);
    shmdt(mp->p + mp->l);
}

#endif


/*
 * vim:ts=8:softtabstop=4:shiftwidth=4:noexpandtab:nowrap
 */

