/* Miscellaneous support functions */


#include <sys/time.h>
#include <time.h>

#include <malloc.h>
#include <unistd.h>
#include <stdlib.h>
#include <fcntl.h>
#include <string.h>
#include <errno.h>

#include <evblib/sysdefs/sysdefs-kern.h>

#include <evblib/misc/misc.h>
#include <evblib/hash/sfh.h>
#include <evblib/str/str.h>


void *safe_malloc(size_t size)
{
    static char msg[] = "FATAL: Out of memory!\n";
    void *p;

    if ((p = malloc(size))) return p;
    write(2, msg, sizeof(msg));
    _exit(255);
}


/* A read that never returns -1. EINTR is handled, the rest ends the program. 
   Not suitable for nonblocking I/O, unless you always select() first. */

int safe_read(int fd, void *p, size_t l)
{
    static char msg[] = "FATAL: I/O error: ";
    char *s;
    int ret;

    do {
	ret = read(fd, p, l);
	if (ret != -1) return ret;
    } while (errno == EINTR);
    s = strerror(errno); write(2, msg, sizeof(msg)); 
    write(2, s, strlen(s)); write(2, "\n", 1);
    _exit(1);
}


void hex(char *buf, void *src, ssize_t len)
{
    static char *hextbl = "0123456789abcdef";
    unsigned char *i = (unsigned char *)src;

    for( ; len > 0; len--) {
	*buf++ = hextbl[(*i >> 4) & 0xf];
	*buf++ = hextbl[*i++ & 0xf];
    }
}


#define DBG_HEXBUFLEN	(4 + 2 + 16 * 3 + 1 + 16 + 1 + 1)

void hexdumpfd(int fd, void *buf, ssize_t len, ssize_t dispofs)
{
	static char hexbuf[DBG_HEXBUFLEN];
	unsigned char *o, *s = (unsigned char *)buf;
	ssize_t ofs, i;	

	for(ofs = 0; ofs < len; ofs += 16, s += 16) {
		o = hexbuf;
		memset(o, ' ', DBG_HEXBUFLEN - 1);
		hex16(o, dispofs + ofs); o+= 6;
		for(i = 0; i < MIN(len - ofs, 16); i++) {
		    hex8(o, s[i]); o += 3;
		}
		o = hexbuf + 4 + 2 + 16 * 3 + 1;
		for(i = 0; i < MIN(len - ofs, 16); i++) 
			*o++ = s[i] > 31 && s[i] < 127 ? s[i] : '.';
		hexbuf[4 + 2 + 16 * 3 + 1 + 16] = '\n';
		write(fd, hexbuf, DBG_HEXBUFLEN - 1);
	}
}


/* Get weak random data - weak and requires floating point support */

void get_weak_random_data(char *p, ssize_t len)
{
    static int inited = 0;
    struct timeval tv;

    if (!inited) {
	if (gettimeofday(&tv, 0) < 0) {
	    write(2, pstr("FATAL: gettimeofday failed!\n"));
	    _exit(1);
	}
	/* Use Paul Hsieh's hash for mixing seconds and nanoseconds */
	srand(sfh(&tv, sizeof(tv)));	
	inited++;
    }
    while(len-- > 0) *p++ = (char)(256.0 * rand() / (RAND_MAX + 1.0));
}


/* Get reasonably good random data */

void get_random_data(char *p, ssize_t len)
{
#ifdef RANDOM_DEV
    static int fd = -1;
    ssize_t n;

    if (fd < 0) fd = open(RANDOM_DEV, O_RDONLY);
    if (fd >= 0) {
	n = read(fd, p, len);
	if (n == len) return;
	close(fd);
	fd = -1;
    }
    write(2, pstr("ERROR: No data from " RANDOM_DEV ", using weak random data!\n"));
#else
    write(2, pstr("WARNING: No random number device - using weak random data!\n"));
#endif
    get_weak_random_data(p, len);
    return;
}


/*
 * vim:softtabstop=4:sw=4
 */

