/*
 * STR - Support for explicit length strings. These have the advantage of
 *       allowing to pass around substrings without copying, to use the length
 *       as a quick hash, and to contain the \0 character.
 *
 * Author:
 * Emile van Bergen, emile@evbergen.xs4all.nl
 *
 * Permission to redistribute an original or modified version of this program
 * in source, intermediate or object code form is hereby granted exclusively
 * under the terms of the GNU General Public License, version 2. Please see the
 * file COPYING for details, or refer to http://www.gnu.org/copyleft/gpl.html.
 *
 * History:
 * 2004/01/13 - EvB - Created
 * 2004/05/11 - EvB - Split off from misc
 * 2005/03/23 - EvB - Copied from OpenRADIUS
 * 2005/05/23 - EvB - Added FREESTR macro and str_dup, str_safedup functions
 * 2005/12/09 - EvB - Renamed type to STR_T; new style
 */


#ifndef _STR_H
#define _STR_H 1


/*
 * INCLUDES & DEFINES
 */


#include <evblib/sysdefs/systypes.h>	    /* for ssize_t */
#include <malloc.h>			    /* for malloc, free */


#define FREESTR(str) do	{					\
    if ((str).p) { free((str).p); (str).p = 0; (str).l = 0; }	\
    } while (0)


/* Return a STR pointing to an object having a size known at compile time */

#define str(o)		    ((STR_T){ (char *)&(o), sizeof((o)) })

/* Return a STR_T pointing to a C string constant. Used to avoid the brainless
 * and wasteful counting of a known number of bytes at run time that would
 * otherwise be needed to generate char *, ssize_t strings. */

#define cstr(s)		    ((STR_T){ (s), sizeof(s) - 1 })

/* generate p, l function parameters from a C string constant */

#define pstr(s)		    (s), (sizeof(s) - 1)


/*
 * TYPES
 */


/* Semantics:
 * non-zero l with NULL p is illegal; the functions here don't check p if l>0
 * non-NULL p with zero l is legal; p will be freed by FREESTR
 * NULL p is legal if l is zero */

typedef struct str {
    char *p;
    ssize_t l;
} STR_T;


/*
 * PROTOTYPES
 */


void *memrchr(void *s, int c, size_t n);
size_t memspn(void *data, size_t len, char *set, int setlen);
size_t memcspn(void *data, size_t len, char *set, int setlen);

ssize_t cpstr(char *d, ssize_t dl, char *s, ssize_t sl);   /* copy string */
ssize_t cptstr(char *d, ssize_t dl, char *s, ssize_t sl);  /* copy, terminate */

void str_tolower(char *p, ssize_t l);
void str_toupper(char *p, ssize_t l);

STR_T str_getword(STR_T *s, char *delim, int delimcnt);

STR_T str_safedup(char *p, ssize_t l);		/* uses safe_malloc */
STR_T str_dup(char *p, ssize_t l);		/* may return NULL p, l */

/* Returns true if p1,l1 equals p2,l2. Zero l1 or l2 allows p1 and p2 to be 0 */

int str_eq(char *p1, ssize_t l1, char *p2, ssize_t l2);

/* Returns true if (the first part of) p1,l1 equals p2,l2. Zero l2 returns 
 * always true and allows p1 and p2 to be 0. */

int str_feq(char *p1, ssize_t l1, char *p2, ssize_t l2);

/* Compares two strings and returns 0 if equal, <0 if p1,l1 < p2,l2 and >0
 * if p1,l1 > p2,l2. */

int str_cmp(char *p1, ssize_t l1, char *p2, ssize_t l2);

#endif


/* 
 * vim:softtabstop=4:sw=4 
 */

