/*
 * EAPTLS - EAPServer TLS module. This can be used to implement EAP-TLS by 
 *          verifying the client certificate, or as the foundation for
 *	    EAP-TTLS and EAP-PEAP by exporting the TLS payload (or
 *          InnerApplication records) as RADIUS A/V pairs or EAP packets,
 *          respectively.
 *
 * Author:
 * Emile van Bergen, emile@evbergen.xs4all.nl
 *
 * Permission to redistribute an original or modified version of this program
 * in source, intermediate or object code form is hereby granted exclusively
 * under the terms of the GNU General Public License, version 2. Please see the
 * file COPYING for details, or refer to http://www.gnu.org/copyleft/gpl.html.
 *
 * History:
 */

char eaptls_id[] = "EAPTLS - Copyright (C) 2005 Emile van Bergen.";


/*
 * INCLUDES & DEFINES
 */


#include <netinet/in.h>
#include <string.h>
#include <unistd.h>
#include <stdlib.h>
#include <stdio.h>

#include <evblib/sysdefs/byteorder.h>
#include <evblib/str/str.h>
#include <evblib/fastring/fastring.h>

#include <gnutls/gnutls.h>

#include <metadata.h>		/* For META_ORD */
#include <constants.h>		/* Server constants */


/* Calculates how far ahead a is from b on a ring of size mod */

#define MODAHEAD(a, b, mod)	(((mod) + (a) - (b)) % (mod))


#define EAPTLS_VERSION		"v0.1"

#define MAX_CONVERSATIONS	128

#define RET_NOMEM		0x40
#define RET_NOCONV		0x41


/*
 * TYPES
 */


typedef struct eaptls_conversation {
    gnutls_session_t session;
    time_t exp;
    int inuse;
} eaptls_conv_t;


/* 
 * GLOBALS
 */


META_ORD rxspace, txspace;
int debug;


/*
 * FUNCTIONS
 */


void usage()
{
    fprintf(stderr, 
"Usage: eaptls [-d] [-r received attribute space number] [-s sent attrspc]\n"
"       eaptls -v\n"
"       eaptls -h\n");
    _exit(1);
}


void parseoptions(int argc, char **argv)
{
    int c;
    META_ORD *spc;

    /* Handle options */
    while((c = getopt(argc, argv, "dhvr:s:")) != -1) {
        spc = &txspace;
        switch(c) {
          case 'r': spc = &rxspace;
          case 's': *spc = atoi(optarg);
		    if (*spc) {
			fprintf(stderr, "radldap: Invalid space number '%s'!\n",
				optarg);
			usage();
		    }
		    break;
          case 'd': debug++; break;
          case 'v': fprintf(stderr, "\nEAPTLS module " EAPTLS_VERSION ". "
               "Copyright (C) 2005 Emile van Bergen / E-Advies.\n\n"
"Permission to redistribute an original or modified version of this program\n"
"in source, intermediate or object code form is hereby granted exclusively\n"
"under the terms of the GNU General Public License, version 2. Please see the\n"
"file COPYING for details, or refer to http://www.gnu.org/copyleft/gpl.html.\n"
"\n");
          case 'h':
          case '?':
            usage();
        }
    }
}


static inline void addint(uint32_t **p, uint32_t val)
{
    uint32_t *o = *p;

    *o++ = net32(C_DS_INTERNAL);
    *o++ = 0;
    *o++ = net32(C_DI_INT);
    *o++ = net32(4);
    *o++ = net32(val);
    *p = o;
}


/*
 * MAIN
 */


int main(int argc, char **argv)
{
    static gnutls_certificate_credentials_t x509_cred;
    static gnutls_dh_params_t dh_params;
    static uint32_t msgbuf[(C_MAX_MSGSIZE >> 2) + 1];
    static eaptls_conv_t convring[MAX_CONVERSATIONS];
    uint32_t spc, vnd, atr, len, *i, *e, *o, tls_flags, tls_length;
    uint32_t convr, convw, convnr;
    gnutls_session_t convsess;
    STR tls_data;

    /* Initialise static data */

    memset(convring, 0, sizeof(convring));

    /* Parse options, setup TLS */

    parseoptions(argc, argv);
    gnutls_global_init();
    gnutls_certificate_allocate_credentials(&x509_cred);
    
    /* Get CAs, CRL, our own keypair */

    gnutls_certificate_set_x509_trust_file(x509_cred, 
	"modules/eaptls/ca.pem", GNUTLS_X509_FMT_PEM);
    gnutls_certificate_set_x509_crl_file(x509_cred, 
	"modules/eaptls/crl.pem", GNUTLS_X509_FMT_PEM);
    gnutls_certificate_set_x509_key_file(x509_cred, 
	"modules/eaptls/cert.pem",
	"modules/eaptls/key.pem", GNUTLS_X509_FMT_PEM);

    /* Get DH parameters - we should read those inside the loop, from a 
     * file that is created upon installation and periodically re-created
     * by cron and reread here each time it's found to be newer than last
     * time it was read. That ensures security even if the module stays
     * running forever, and does not create any delays here. For testing
     * this will do though. */

    gnutls_dh_params_init(&dh_params);
    gnutls_dh_params_generate2(dh_params, 1024);
    gnutls_certificate_set_dh_params(x509_cred, dh_params);

    /* Request loop */

    fprintf(stderr, "eaptls: Ready for requests.\n");
    for(;;) {

        /*
         * Get the request from OpenRADIUS
         */

        /* Read header */
        if (read(0, msgbuf, 8) != 8) { perror("eaptls: read"); break; }
        if (net32(*msgbuf) != 0xbeefdead) {
            fprintf(stderr, "eaptls: Invalid magic 0x%08x!\n", net32(*msgbuf)); 
            break;
        }
        len = net32(msgbuf[1]);
        if (len < 8 || len > sizeof(msgbuf) - 4) {
            fprintf(stderr, "eaptls: Invalid length %d!\n", len); 
            break;
        }

        /* Read rest of message */
        if (read(0, msgbuf + 2, len - 8) != len - 8) {
            perror("eaptls: read"); 
            break;
        }

        /*
         * Loop through the attributes. The attribute spaces for
         * the Request- and Response-[T]TLS/PEAP-* attributes 
         * depend on the EAP type and are specified on the command
         * line. Note: we parse responses, we send requests. Attribute
         * numbers 0, 1 and 2 correspond to the Flags, TLS Message
         * Length and TLS Data fields of the EAP TLS packets.
         */

        convnr = -1;
        e = msgbuf + (len >> 2);
        for(i = msgbuf + 2; i < e; i += ((len + 3) >> 2) + 4) {

            /* Get space/vendor/attribute/length tuple */

            spc = net32(i[0]); vnd = net32(i[1]);
            atr = net32(i[2]); len = net32(i[3]);
            if (debug) {
                fprintf(stderr, "eaptls: got space %d, vendor "
                        "%d, attribute %d, len %d\n", spc, vnd, atr, len);
            }

            /* Pick TLS attributes from EAP type-specific space */

            if (spc == rxspace && (vnd == 0 || vnd == META_ORD_ERR)) {
                switch (atr) {
                    case 0: tls_flags = net32(i[4]); break;
                    case 1: tls_length = net32(i[4]); break;
                    case 2: tls_data.p = (char *)&i[4]; 
			    tls_data.l = len;
                }
                continue;
            }

            /* Pick TLS conversation pointer into timer ring */

            if (spc == C_DS_INTERNAL && atr == C_DI_TID) {
                convnr = net32(i[4]);
                continue;
            }
        }

        /* Initialise reply buffer */

        o = msgbuf; *o++ = net32(0xdeadbeef); o++;
        e = msgbuf + sizeof(msgbuf) / sizeof(msgbuf[0]);
        len = 8; 

        /* See if the request's conversation number is valid */

        if (convnr > MAX_CONVERSATIONS || !convring[convnr].inuse) {

            /* If we don't have room for a new conversation, fail 
               now */

            if (!MODAHEAD(convr, convw, MAX_CONVERSATIONS)) {
                addint(&o, RET_NOCONV);
                goto reply;
            }

            /* Create a new EAP-TLS conversation. A conversation is
             * a series of requests and responses. It does not map
             * 1:1 to a TLS session, because a new conversation may
             * resume an existing TLS session. */

            convnr = convw++;
            convring[convnr].inuse = 1;
            gnutls_init(&convring[convnr].session, GNUTLS_SERVER);
            convsess = convring[convnr].session;
            gnutls_set_default_priority(convsess);
        }

        /*
         * Send a reply with size 'len'
         */
reply:
        msgbuf[1] = net32(len);
        if (write(1, msgbuf, len) != len) { perror("eaptls: write"); 
                            break; }
    }

    gnutls_certificate_free_credentials(x509_cred);
    gnutls_global_deinit();

    return 1;
}


/*
 * vim:softtabstop=4:sw=4
 */

