/*
 * JOBS - Handles job creation from requests, job juggling on interfaces
 *	  and channels, and job ending after converting them to responses.
 *
 * Author:
 * Emile van Bergen, emile@evbergen.xs4all.nl
 *
 * Permission to redistribute an original or modified version of this program
 * in source, intermediate or object code form is hereby granted exclusively
 * under the terms of the GNU General Public License, version 2. Please see the
 * file COPYING for details, or refer to http://www.gnu.org/copyleft/gpl.html.
 *
 * History:
 * 2001/08/25 - EvB - Created
 * 2001/10/29 - EvB - Moved chan_put/getjob here as job_to/fromchan etc.
 */

char jobs_id[] = "JOBS - Copyright (C) 2001 Emile van Bergen.";


/*
 * INCLUDES & DEFINES
 */


#include <string.h>
#include <errno.h>
#include <stdlib.h>

#include <language.h>		/* Also includes metaops.h */
#include <config.h>		/* For conf_new() etc */
#include <md5.h>
#include <debug.h>

#include <jobs.h>


/*
 * FUNCTIONS
 */


static META_AV *meta_addnewav(META_AV **head, META_AV **tail, 
			      META_AV *rel, int before, 
			      META_ITEM *i, char *p, META_ORD l, int flags)
{
	META_AV *av;

	av = (META_AV *)malloc(sizeof(META_AV)); if (!av) return 0;
	memset(av, 0, sizeof(META_AV));
	av->i = i;
	av->p = p;
	av->l = l;
	av->flags = flags;

	meta_addav(head, tail, rel, before, av);
	return av;
}


JOB *job_new(SOCK *s, time_t t)
{
	JOB *j;
	META_AV *reqh, *reqt, *av;
	int sl;

	/* Allocate new job, including packet buffer */
	j = (JOB *)malloc(sizeof(JOB)); if (!j) return 0;
	memset(j, 0, sizeof(JOB));
	j->pkt = (char *)malloc(C_MAX_PKTSIZE);
	if (!j->pkt) { free(j); return 0; }

	/* Receive packet, setting also the reply fd and sockaddr_in */
	sl = sizeof(j->replyadr);
	j->pktlen = recvfrom(s->fd, j->pkt, C_MAX_PKTSIZE, 0, 
			     (struct sockaddr *)&(j->replyadr), &sl);
	if (j->pktlen == -1) { msg(F_RECV, L_NOTICE, "job_new: WARNING: recvfro"
						     "m() said: %s!\n",
				   strerror(errno)); job_del(j); return 0; }
	j->replyfd = s->fd;

	if (msg_thresh[F_RECV] >= L_NOTICE) {
		msg(F_RECV, L_NOTICE, "job_new: Received request:\n"); 
		hexdump(j->pkt, j->pktlen);
	}

	/* Decode it. Note that the pointers in the AV items may refer to
	   the data in the packet buffer, so don't free that packet buffer. */
	reqh = meta_decode(s->c->m, s->c->ds_rad_pkt, 0, j->pkt, j->pktlen, 
			   &reqt);
	if (!reqh) { msg(F_RECV, L_NOTICE, "job_new: Could not decode packet"
					   "!\n"); job_del(j); return 0; }

	/* Add extra information about this request */
	av = meta_addnewav(&reqh, &reqt, 0, 1, s->c->di_timestamp, 0, t, 0);
	av = meta_addnewav(&reqh, &reqt, av, 0, s->c->di_ip_source, 0, 
		getord((char *)&j->replyadr.sin_addr,
		       sizeof(j->replyadr.sin_addr)), 0);
	av = meta_addnewav(&reqh, &reqt, av, 0, s->c->di_ip_dest, 0, s->ip, 0);
	av = meta_addnewav(&reqh, &reqt, av, 0, s->c->di_udp_source, 0, 
		getord((char *)&j->replyadr.sin_port,
		       sizeof(j->replyadr.sin_port)), 0);
	av = meta_addnewav(&reqh, &reqt, av, 0, s->c->di_udp_dest, 0, 
		s->port, 0);
	if (!av) { msg(F_RECV, L_ERR, "job_new: ERROR: Could not allocate "
					"new AV item(s)!\n");
		   meta_freeavlist(reqh); job_del(j); return 0; }

	/* Create the VM */
	j->vm = vm_new(s->c->expr, s->c->exprlen, C_MAX_STKDEPTH,reqh,reqt,0,0);

	/* Show things */
	if (msg_thresh[F_RECV] >= L_DEBUG) {
		meta_printavlist(s->c->m, j->vm->head[VM_REQ], 0);
	}

	return j;
}


void job_del(JOB *j)
{
	if (j) {
		if (j->pkt) free(j->pkt);
		if (j->msg) free(j->msg);
		if (j->vm) {
			if (j->vm->head[VM_REQ]) 
				meta_freeavlist(j->vm->head[VM_REQ]);
			if (j->vm->head[VM_REP]) 
				meta_freeavlist(j->vm->head[VM_REP]);
			vm_del(j->vm);
		}
		free(j);
	}
}


/* Runs the expression until interface trap or done. If done, signs,
   responds, deletes the job and returns 0. If error, deletes the job and
   returns 0. If interface trap, returns the interface. */

IFACE *job_run(CONF *c, JOB *j)
{
	META_AV *av, *avtree;
	md5_state_t mds;
	IFACE *ret;
	int n;

	/* Run the expression on this job until something happens */
	n = vm_run(j->vm);
	switch(n) {

	  case VM_HALTED:	/* Done */ 

		msg(F_SEND, L_NOTICE, "job_run: Done with expression, "
				      "response:\n");
		if (msg_thresh[F_SEND] >= L_DEBUG)
			meta_printavlist(c->m, j->vm->head[VM_REP], 0);

		/* Build encapsulation tree and encode packet */
		meta_buildtree(j->vm->head[VM_REP], &avtree, c->ds_rad_pkt);
		if (!avtree) { msg(F_SEND, L_ERR, "job_run: ERROR: Could not "
					     "build response tree!\n"); break; }
		j->pktlen = meta_encode(c->ds_rad_pkt, j->pkt,
					C_MAX_PKTSIZE, avtree, 0);
		meta_freeavlist(avtree);
		if (j->pktlen == -1) { msg(F_SEND, L_ERR, "job_run: ERROR: "
				       "Could not encode response!\n"); break; }

		/* Get the shared secret from the response list */
		for(av = j->vm->head[VM_REP];
		    av && (!av->i || av->i->nr != C_DI_SECRET);
		    av = av->next);
		if (!av || !av->p || !av->l || av->l > C_MAX_SECRSIZE) {
			msg(F_SEND, L_ERR, "job_run: ERROR: No (valid) secret"
					   " to sign response!\n"); break;
		}

		/* Sign the packet using the secret described by 'av' */
		md5_init(&mds);
		md5_append(&mds, (unsigned char *)j->pkt, j->pktlen);
		md5_append(&mds, (unsigned char *)av->p, av->l);
		md5_finish(&mds, (unsigned char *)j->pkt + c->di_authenticator->val_ofs);

		/* Send the reply */
		if (msg_thresh[F_SEND] >= L_NOTICE) hexdump(j->pkt, j->pktlen);
		n = sendto(j->replyfd, j->pkt, j->pktlen, 0,
			   (struct sockaddr *)&j->replyadr,sizeof(j->replyadr));
		if (n == -1) { msg(F_SEND, L_ERR, "job_run: ERROR: Could not "
					     "reply: %s!\n", strerror(errno)); }
		break;

	  case VM_ABORTED:		/* There was an 'abort' opcode */

		msg(F_SEND, L_NOTICE, "job_run: Aborted by code - dropping request.\n");
		break;

	  case VM_IFACETRAP:		/* There was an interface call */

		ret = vm_getiface(j->vm);
		msg(F_LANG, L_NOTICE, "job_run: Interface call: %s\n", ret->name);
	  	return ret;

	  default:			/* The VM choked on the code */

		msg(F_LANG, L_ERR, "job_run: Invalid operation, error code %d - dropping request!\n", n);
		break;
	}
	
	job_del(j);
	return 0;
}


void job_toiface(IFACE *i, JOB *j, time_t t)
{
	CHAN *ch;

	/* First see if this interface has got an idle channel, starting at
	   the round-robin next one. If we don't, put the job on the shared
	   send queue. */
	for(ch = i->rrch; ; ) {

		/* If we found a channel with idle tx, make it take the job */
		if (ch->proc->state == PRS_IDLE || 
		    ch->proc->state == PRS_RECEIVING) {
			job_tochan(ch, j, t);
			i->rrch = ch->next; if (!i->rrch) i->rrch = i->chans;
			return;
		}

		/* Go to next one, wrapping around */
		ch = ch->next; if (!ch) ch = i->chans;

		/* If we're back where we started, nothing was idle */
		if (ch == i->rrch) break;
	}

	if (i->sendqlen >= C_MAX_SENDQLEN) {
		msg(F_PROC, L_NOTICE, "job_toiface: Shared queue for interface "
				      "%s overflowing (%d) - dropping job!\n",
				      i->sendqlen);
		job_del(j);
		return;
	}

	/* Put job on this interface's shared send queue */
	j->next = 0;				/* my next = none */
	j->prev = i->sendqh;			/* my prev = current head */
	if (i->sendqh) i->sendqh->next = j;	/* current head's next = me */
	else i->sendqt = j;			/* if none, then I'm tail too */
	i->sendqh = j;				/* I am always the new head */
	i->sendqlen++;

	msg(F_PROC, L_NOTICE, "job_toiface: Job put on shared queue for "
			      "interface '%s' (%d)\n", i->name, i->sendqlen);
}


void job_tochan(CHAN *ch, JOB *j, time_t t)
{
	static char reqmsg[C_MAX_MSGSIZE];
	ssize_t msglen;
	PROC *p;

	/* Put the channel's process tx'er in motion; apparently it was idle */
	p = ch->proc;
	p->state |= PRS_SENDING;
	p->timer = t + p->xfertimeout;

	/* Create the message from the request list. */
	msglen = meta_avtomsg(ch->iface->c->m, j->vm->head[VM_REQ],
			      reqmsg, C_MAX_MSGSIZE, 
			      ch->iface->flags, ch->iface->sendacl);

	/* Put the message in the channel's proc's ring. I'd agree we copy 
	   things around a lot - avtomsg could be done straight to the ring.
	   The only times meta_avtomsg is used right now is here and in
	   printav(), which would also benefit from that... */
	ring_put(p->w, reqmsg, msglen);

	msg(F_PROC, L_DEBUG, "- set sending state on %s's pid %d, %d now in "
			     "ring\n", 
	    ch->iface->name, p->pid, ring_maxget(p->w));

	/* Put job at head of channel's recv queue */
	j->next = 0;				/* my next = none */
	j->prev = ch->recvqh;			/* my prev = current head */
	if (ch->recvqh) ch->recvqh->next = j;	/* current head's next = me */
	else ch->recvqt = j;			/* if none, then I'm tail too */
	ch->recvqh = j;				/* I am always the new head */
}


JOB *job_fromchan(CHAN *ch)
{
	JOB *j;

	/* Take job from tail of channel's recv queue */
	j = ch->recvqt;
	if (!j) return j;			/* no job: shouldn't happen! */	

	ch->recvqt = ch->recvqt->next;		/* advance tail */
	if (ch->recvqt) ch->recvqt->prev = 0;	/* new tail's prev = none */
	else ch->recvqh = 0;			/* if no tail, no head either */

	return j;
}

