/*
 * INCLUDES & DEFINES
 */


#include <sys/types.h> 
#include <sys/socket.h>
#include <netinet/in.h>	/* Standard combo for sockaddr_in, socket, bind etc. */
#include <arpa/inet.h>	/* For inet_addr */
#include <netdb.h>	/* For gethostbyname */

#include <sys/time.h>	/* For struct timeval */
#include <sys/select.h>	/* For select */
#include <unistd.h> 	/* For getopt */
#include <stdlib.h>	/* For strtoul, getopt for some, malloc */ 
#include <string.h>
#include <stdio.h>

#include <metadict.h>
#include <metaops.h>	/* For META_AV etc. */

#include <debug.h>


#ifndef RADDB
#define RADDB		"../../raddb"
#endif

#define DEF_AUTHPORT	1812
#define DEF_ACCTPORT	1813

#define FLOOD_REQS	10000


/*
 * GLOBALS
 */


/* Options */	

struct sockaddr_in sin_src, sin_dst;
char *raddb, *secret;
int debug, doacct, flood;



/*
 * FUNCTIONS
 */


void usage(char *msg)
{
	if (msg) fprintf(stderr, "\nERROR: %s\n\n", msg);
	fprintf(stderr, 
		"Usage: radclient [-r raddb] [-d] [-i source] [-p port] [-a] server secret\n");

	_exit(1);
}


void parseoptions(int argc, char **argv)
{
	struct hostent *he;
	int c;
	char *s;

	/* Defaults */
	memset(&sin_src, 0, sizeof(sin_src)); 
	memset(&sin_dst, 0, sizeof(sin_dst));
	sin_src.sin_family = AF_INET; sin_dst.sin_family = AF_INET;
	raddb = RADDB; secret = 0;
	debug = 0; doacct = 0; flood = 0;

	while((c = getopt(argc, argv, "r:i:p:dDafh")) != -1) {

		switch(c) {
		  case 'r': raddb = optarg; break;
		  case 'i':
			sin_src.sin_addr.s_addr = inet_addr(optarg);
			if (sin_src.sin_addr.s_addr == 0xffffffff) {
				usage("Invalid address!");
			}
			break;
		  case 'p':
			sin_dst.sin_port = htons(strtoul(optarg, &s, 10));
			if (!s || *s) usage("Invalid port!");
			break;
		  case 'a': doacct = 1; break;
		  case 'f': flood = 1; break;
		  case 'D': debug++;
		  case 'd': debug++; break;
		  case 'h':
		  case '?':
			usage(0);
		}
	}

	if (optind != argc - 2) usage("Wrong number of arguments!");
	
	/* Get server address */
	sin_dst.sin_addr.s_addr = inet_addr(argv[optind]);
	if (sin_dst.sin_addr.s_addr == 0xffffffff) {

		he = gethostbyname(argv[optind]);
		if (!he || he->h_addrtype != AF_INET || he->h_length != 4) { 
			perror("ERROR: Could not resolve hostname"); 
			usage(0); 
		}
		sin_dst.sin_addr.s_addr = *(U_INT32_T *)(he->h_addr);
	}

	/* Set port if not already defined */
	if (!sin_dst.sin_port) {
		sin_dst.sin_port = htons(doacct ? DEF_ACCTPORT : DEF_AUTHPORT);
	}

	/* Get secret */
	secret = argv[optind + 1];
	if (!secret || !*secret) usage("No secret!");
}


/*
 * MAIN
 */


int main(int argc, char **argv)
{
	static char buf[C_MAX_MSGSIZE];
	static char pkt[C_MAX_PKTSIZE];
	static char resppkt[C_MAX_PKTSIZE];
	META *m;
	META_SPC *spc;
	META_AV *avhead, *avtail, *avtree;
	struct timeval tv, tv_s, tv_e;
	fd_set fds; 
	int len, fd, try, n;

	/*
	 * Initialise
	 */

	parseoptions(argc, argv);
	msg_init(-1, 0);	/* no syslog facility, not quiet on stderr */
	if (debug) {
		msg_setthresh(F_SEND, L_NOTICE);
		msg_setthresh(F_RECV, L_NOTICE);
	}
	if (debug > 1) {
		msg_setthresh(F_SEND, L_DEBUG);
		msg_setthresh(F_RECV, L_DEBUG);
		msg_setthresh(F_TEXT, L_NOTICE);
		msg_setthresh(F_MISC, L_NOTICE);
	}
	if (debug > 2) {
		msg_setthresh(F_TEXT, L_DEBUG);
		msg_setthresh(F_MISC, L_DEBUG);
	}

	m = meta_newfromdict(raddb);
	if (!m) { msg(F_MISC, L_ERR,"main: ERROR: Couldn't open dictionary!\n");
		  return 1; }

	spc = meta_getspcbynr(m, C_DS_RAD_PKT);
	if (!spc) { msg(F_MISC, L_ERR, "main: ERROR: RAD-PKT not defined!\n");
		    return 1; }

	/*
	 * Get ASCII pairs from stdin into META_AV list 
	 */

	printf("Enter A/V pairs, one per line, terminated by ^D.\n");
	avhead = avtail = 0;
	while(fgets(buf, C_MAX_MSGSIZE - 1, stdin)) {

		len = strlen(buf); if (buf[len - 1] == '\n') len--;
		meta_ascmsgtoav(m, buf, len, &avhead, &avtail, 0, 0);
	}

	if (debug > 2) { write(2, "Full A/V list:\n", 15);
		     meta_printavlist(m, avhead, 0); }

	/*
	 * Build encapsulation tree and encode it
	 */

	meta_buildtree(avhead, &avtree, spc);
	if (debug > 1) { write(2, "A/V tree:\n", 10);
		         meta_printavlist(m, avtree, 0); }

	len = meta_encode(spc, pkt, C_MAX_PKTSIZE, avtree, 0);
	meta_freeavlist(avhead); avhead = avtail = 0;
	meta_freeavlist(avtree); avtree = 0;
	if (len == -1) {
		msg(F_SEND, L_ERR, "ERROR: Could not encode data!\n");
		return 2;
	}
	if (debug || !flood) { 
		fprintf(stderr, "Encoded request (%d bytes):\n", len); 
		fflush(stderr); hexdump(pkt, len); write(2, "\n", 1); 
	}

	/*
	 * Do the transaction
	 */

	fd = socket(AF_INET, SOCK_DGRAM, IPPROTO_UDP);
	if (!fd) { perror("ERROR: Could not create socket"); return 3; }

	if (sin_src.sin_addr.s_addr && 
	    bind(fd, (struct sockaddr *)&sin_src, sizeof(sin_src)) == -1) {

		perror("ERROR: Could not bind socket"); return 4;
	}

	gettimeofday(&tv_s, 0);
	for(try = 0; try < (flood ? FLOOD_REQS : 5); try++) {

		/*
		 * Send packet
		 */

		if (debug || !flood) {
			fprintf(stderr, "%sending %d bytes to %s:%d\n",
				try ? "re-s" : "S", len, 
				inet_ntoa(sin_dst.sin_addr), 
				htons(sin_dst.sin_port));
		}

		if (sendto(fd, pkt, len, 0, (struct sockaddr *)&sin_dst, 
			   sizeof(sin_dst)) == -1) {
			perror("ERROR: Could not send"); return 5;
		}

		/*
		 * Wait for answer, loop if timeout
		 */

		FD_ZERO(&fds); FD_SET(fd, &fds);
		tv.tv_sec = 3; tv.tv_usec = 0;
		n = select(fd + 1, &fds, 0, 0, &tv);
		if (!n) continue;
		if (n == -1) { perror("ERROR: Could not select"); return 6; }

		/* 
		 * Receive packet
		 */

		n = sizeof(sin_src);
		n = recvfrom(fd, resppkt, C_MAX_PKTSIZE, 0,
			     (struct sockaddr *)&sin_src, &n); 
		if (n == -1) { 
			perror("WARNING: Could not receive response"); 
			continue; 
		}

		fprintf(stderr, "Received %d bytes from %s:%d\n",
			n, inet_ntoa(sin_src.sin_addr), 
			htons(sin_src.sin_port)); fflush(stderr);
		if (debug > 1) hexdump(resppkt, n);

		/*
		 * Decode, show A/V pairs, quit
		 */

		avhead = meta_decode(m, spc, 0, resppkt, n, &avtail);
		if (avhead) { 
			if (debug || !flood) meta_printavlist(m, avhead, 0); 
		}
		else fprintf(stderr, "Could not decode response!\n");

		if (!flood) {
			close(fd);
			return 0;
		}
	}

	if (flood) {
		gettimeofday(&tv_e, 0);
		tv_e.tv_sec -= tv_s.tv_sec;
		tv_e.tv_usec -= tv_s.tv_usec;
		if (tv_e.tv_usec < 0) tv_e.tv_usec += 1000000L, tv_e.tv_sec--;
		fprintf(stderr, "%ld serial transactions took %ld.%06ld seconds.\n", (long)FLOOD_REQS, (long)tv_e.tv_sec, (long)tv_e.tv_usec);
	}
	else fprintf(stderr, "No response.\n");
	close(fd);

	return 7;
}

