/*
 * METAOPS - Provides operations to be performed on data blocks
 * 	     according to metadata definitions.
 *
 * Author:
 * Emile van Bergen, emile@evbergen.xs4all.nl
 *
 * Permission to redistribute an original or modified version of this program
 * in source, intermediate or object code form is hereby granted exclusively
 * under the terms of the GNU General Public License, version 2. Please see the
 * file COPYING for details, or refer to http://www.gnu.org/copyleft/gpl.html.
 *
 * History:
 * 2001/03/30 - EvB - Created
 * 2001/04/17 - EvB - Changed decode not to add items in reverse order
 * 2001/04/24 - EvB - Added buildtree / encode
 * 2001/05/08 - EvB - Added integer value and temporary flag to META_AV
 * 2001/05/10 - EvB - Merged string length and integer value (always fixed
 *		      from dictionary)
 * 2001/05/16 - EvB - Added printav / printavlist debugging functions here
 * 2001/06/04 - EvB - Made decode create a double linked list instead of single
 * 2001/06/25 - EvB - Added meta_avtomsg
 * 2001/09/11 - EvB - Added meta_addav here; used by job_new, chan_handle_read
 *		      and vm_run (we should, at least).
 * 2001/09/14 - EvB - Added ACL support to meta_avtomsg here after some doubt.
 */

char metaops_id[] = "METAOPS - Copyright (C) 2001 Emile van Bergen.";


/*
 * INCLUDES & DEFINES
 */


#include <unistd.h>	/* For the write() in printav */
#include <malloc.h>
#include <string.h>

#include <misc.h>	/* For hex() */
#include <metaops.h>
#include <debug.h>


/* Maximum stack depth for build tree */

#define BT_STKDEPTH	6


/*
 * FUNCTIONS
 */


/* Return the attribute or vendor field */

#define getatrnr(s, d)	(getord((char *)(d) + (s)->atr_ofs, (s)->atr_size))
#define getvndnr(s, d)	(getord((char *)(d) + (s)->vnd_ofs, (s)->vnd_size))

/* Return the length field + the adjustment if the size of the length field 
   is non-zero, otherwise return the adjustment itself, as an absolute skip 
   length */

#define getskiplen(i, d)	\
	((i)->len_adj + getord((char *)(d) + (i)->len_ofs, (i)->len_size))

/* Return the given skiplength + the item's value size adjustment if that's
   zero or negative, otherwise return the item's value size itself as the 
   absolute size. */

#define getvallen(i, sl)	\
	((i)->val_size + ((i)->val_size <= 0 ? (sl) : (META_ORD)0))


/* Decodes a data block, forming a flat list of AV items. */

META_AV *meta_decode(META *m, META_SPC *s, META_ORD defvnd, 
		     char *blk, META_ORD blksize, META_AV **rettail)
{
	META_ITEM *i;
	META_ORD skiplen, vallen;
	META_ORD atrnr, vndnr;
	META_AV *av, *avlist, *tail, *newtail;
	char *p;

	/* Check arguments */
	if (!m || !s || !blk) {
		msg(F_MISC, L_ERR, "meta_decode: BUG: invalid argument!\n");
		return 0;
	}

	tail = avlist = 0;

 	/* Check if this is an A/V or fixed field space */	
	if (s->atr_size == 0) {

		/* Fixed fields - loop through the items. */
		for(i = s->items; 
		    i && i->len_ofs + i->len_size <= blksize && 
		    	 i->val_ofs + (vallen = getvallen(i, getskiplen(i, blk))) <= blksize; 
		    i = i->next) {

			/* Add the data, unless we shouldn't decode it */
			if (!i->nodec) {

				av = (META_AV *)malloc(sizeof(META_AV));
				if (!av) { meta_freeavlist(avlist); return 0; }

				memset(av, 0, sizeof(META_AV));
				av->i = i;
				if (MT_ISORD(i->val_type))
					av->l = getord(blk+i->val_ofs, vallen);
				else av->l = vallen, av->p = blk+i->val_ofs;

				if (!avlist) avlist = av;
				if (tail) tail->next = av;
				av->prev = tail;
				tail = av;
			}

			/* Always recurse into a subspace, if any */
			if (i->subspace) {

				av = meta_decode(m, i->subspace, defvnd,
						 blk + i->val_ofs, vallen,
						 &newtail);
				if (!av) { meta_freeavlist(avlist); return 0; }

				if (!avlist) avlist = av;
				if (tail) tail->next = av;
				av->prev = tail;
				tail = newtail;
			}
		}

		/* If we terminated prematurely, return -1 to signal an error */
		if (i) { meta_freeavlist(avlist); return 0; }

		/* Return the head of the created list */
		if (rettail) *rettail = tail;
		return avlist;
	}

	/* Prevent a nonsense warning - the compiler doesn't see that
	   the p += skiplen statement will only be executed if skiplen
	   actally got assigned to. Oh well. */
	skiplen = 0;

	/* Attribute space. You've just got to love C's logical operators... */
	for(p = blk;
	    p < blk + blksize &&
	    	 p + s->atr_ofs + s->atr_size <= blk + blksize &&
	    	 p + s->vnd_ofs + s->vnd_size <= blk + blksize &&
		 ((i = meta_getitembynr(m, s, (atrnr = getatrnr(s, p)),
		 	(vndnr = (s->vnd_size ? getvndnr(s, p) : defvnd)))) ||
		  (i = meta_getitembynr(m, s, atrnr, META_ORD_ERR)) ||
		  (i = meta_getitembynr(m, s, META_ORD_ERR, META_ORD_ERR))) &&
	    	 p + i->val_ofs + ((skiplen = getskiplen(i, p)), (vallen = getvallen(i, skiplen))) <= blk + blksize;
	    p += skiplen) {

		/* Add the data, unless we shouldn't decode it */
		if (!i->nodec) {

			av = (META_AV *)malloc(sizeof(META_AV));
			if (!av) { meta_freeavlist(avlist); return 0; }

			memset(av, 0, sizeof(META_AV));
			av->i = i;
			if (MT_ISORD(i->val_type))
				av->l = getord(p + i->val_ofs, vallen);
			else av->l = vallen, av->p = p + i->val_ofs;

			if (!avlist) avlist = av;
			if (tail) tail->next = av;
			av->prev = tail;
			tail = av;
		}

		/* Recurse into a subspace, if any */
		if (i->subspace) {
			av = meta_decode(m, i->subspace, vndnr,
					 p + i->val_ofs, vallen, &newtail);
			if (!av) { meta_freeavlist(avlist); return 0; }

			if (!avlist) avlist = av;
			if (tail) tail->next = av;
			av->prev = tail;
			tail = newtail;
		}
	}

	/* If the loop terminated prematurely, return 0 to signal the error */
	if (p != blk + blksize) { meta_freeavlist(avlist); return 0; }

	/* Return the (tail and) head of the created list */
	if (rettail) *rettail = tail;
	return avlist;
}


/* Encodes a AV tree created by buildtree, according to applicable metadata */

META_ORD meta_encode(META_SPC *s, char *blk, META_ORD blksize, META_AV *avlist,
		     META_ORD *retvnd)
{
	META_ITEM *i;
	META_AV *av;
	META_ORD vndnr, pos, skiplen, retlen;

	if (!blk || blksize <= 0 || !avlist) {
		msg(F_MISC, L_ERR, "meta_encode: BUG: invalid argument!\n");
		return -1;
	}

	/* Set the vendor number used when the space has a vendor field but
	   the item's vendor number is 'any', to the one inherited from the
	   caller; in most cases this same function. */
	vndnr = retvnd ? *retvnd : 0; 

	for(av = avlist, pos = retlen = 0; 
	    av && (i = av->i) && i->spc == s;
	    av = av->next) {

	    	/* Skip deleted items. Don't skip items with 'noenc' specified,
		   they're only skipped in buildtree so they can still be used
		   as encapsulating items. */
		if (av->flags & AV_DEL) continue;

		/* Check if the vendor and attribute fields fit */
		if (pos + s->atr_ofs + s->atr_size > blksize ||
		    pos + s->vnd_ofs + s->vnd_size > blksize) {
			msg(F_SEND, L_ERR, "meta_encode: ERROR: attribute "
					   "field (at %d, size %d) and/or "
					   "vendor field (at %d, size %d), "
					   "do not fit in remaining output "
					   "block, size %d!\n",
			    pos + s->atr_ofs, s->atr_size, pos + s->vnd_ofs, 
			    s->vnd_size, blksize);
			return -1;
		}

		/* Determine skiplen and put value */
		if (av->sub) {

			/* The value is a list of items again, so recurse */
			skiplen = meta_encode(i->subspace, 
					 blk+pos+i->val_ofs, 
					 blksize - (pos + i->val_ofs),
					 av->sub, &vndnr) - i->val_size;
		}
		else {
			/* The value is a real item */
			if (MT_ISORD(i->val_type)) {

				/* An integer (fixed size in val_size) */
				skiplen = i->val_ofs + i->val_size;
				putord(blk+pos+i->val_ofs, i->val_size, av->l);
			}
			else {
				/* An octet string */
				if (i->val_size <= 0) {

					/* Variable size */
					skiplen = av->l - i->val_size;
					memcpy(blk+pos+i->val_ofs, av->p,av->l);
				}
				else {
					/* Fixed size */
					skiplen = i->val_ofs + i->val_size;
					if (av->l >= i->val_size) {

						/* Limited by val_size */
						memcpy(blk+pos+i->val_ofs,av->p,
						       i->val_size);
					} 
					else {
						/* Limited by av->l, so pad */
						memcpy(blk+pos+i->val_ofs,av->p,
						       av->l);
						memset(blk+pos+i->val_ofs+av->l,
						       0, i->val_size - av->l);
					}
				}
			}
		}

		/* Put vendor field in. If the item's vendor number is 'any', 
		   use the vendor number that may be set by the just encoded 
		   sub list. */
		if (i->vnd != META_ORD_ERR) vndnr = i->vnd;
		putord(blk + pos + s->vnd_ofs, s->vnd_size, vndnr);

		/* Put the attribute field in */
		putord(blk + pos + s->atr_ofs, s->atr_size, i->nr);
				
		/* Put length field in */
		putord(blk + pos + i->len_ofs, i->len_size,
		       skiplen - i->len_adj);

		if (s->atr_size > 0) {
			/* If A/V space, add skiplen to pos and retlen */
			pos += skiplen;
			retlen += skiplen;
		}
		else {
			/* Otherwise, set retlen to the highest skiplen found */
			if (skiplen > retlen) retlen = skiplen;
		}
	}

	/* Return the vendor number of the last encoded item in this list
	   if that's requested. */
	if (retvnd) *retvnd = vndnr;

	return retlen;
}


/* Build a tree from a list, inserting encapsulating attributes where needed.
   It's become a bit of an ugly routine, after adding the double-linked AV
   list and the AV_DEL flag & noenc handling. */

void meta_buildtree(META_AV **srclst, META_AV **dstlst, META_SPC *groundspc)
{
	META_AV **dststk[BT_STKDEPTH];
	META_SPC *spcstk[BT_STKDEPTH];
	META_ITEM *itmstk[BT_STKDEPTH];

	META_ITEM *curitem, *i;
	META_AV *curav, *newav;
	META_SPC *curspc;
	int itmsp, dstsp;

	if (!srclst || !dstlst || !groundspc) return;
	itmsp = dstsp = 0; 
	curspc = groundspc;
	
skipdel_0:
	/* Take the first AV from the source list. We _move_ items here. */
	curav = *srclst;
	if (!curav) { *dstlst = 0; return; }
	*srclst = curav->next;
	curav->next = 0;
	curav->prev = 0;
	if (curav->flags & AV_DEL || (curav->i && curav->i->noenc))
		goto skipdel_0;

	for(;;) {

		/* See if the current item fits in current space */
		if (curav->i->spc == curspc) {

			/* Yes, so add the item to the destination list */
			*dstlst = curav;
			dstlst = &(curav->next);

skipdel_1:		/* Take the next item from the source list and cont. */
			curav = *srclst;
			if (curav) {

				/* There actually is a next item */
				*srclst = curav->next;
				curav->next = 0;
				curav->prev = 0;
				if (curav->flags & AV_DEL || 
				    (curav->i && curav->i->noenc))
					goto skipdel_1;

				/* If this is a single attr. space, leave
				   the space now. */
				while(curspc->single && dstsp) {
					curspc = spcstk[--dstsp];
					dstlst = dststk[dstsp];
				}
				continue;
			}
			
			/* Nothing left on the source list, so we're done. */
			return;
		}

		/* No, so try to reach the current space from this item's space
		   by backtracking encapsulating items */
		for(curitem = curav->i; curitem && curitem->spc != curspc; ) {

		    	/* Not there yet, push the enc. item onto item stack.
			   Try to find one with a matching vendor nr. first. */
			for(i = curitem->spc->enc_items;
			    i && i->vnd != curav->i->vnd;
			    i = i->next);

			/* Couldn't match vendor, find the 'any vendor' item */
			if (!i) {
				for(i = curitem->spc->enc_items;
				    i && i->vnd != META_ORD_ERR;
				    i = i->next);
			}

			/* Add the found item to the encapsulating item stack */
			curitem = i;
			itmstk[itmsp++] = curitem;
		}

		/* See if we found an encapsulation route. If not, go one 
		   (or more) spaces back on the space stack and retry. */
		if (!curitem) {

			/* If we're not at the bottom yet, pop current
			   destination list and space from the stack and remove
			   all items from the item stack. If we _are_ at the
			   bottom, we apparently cannot handle this item. */
			if (dstsp) {

				/* Empty stack of (useless) obtained items */
				itmsp = 0;

				/* Keep going back if the current space
				   is a 'single attribute' space. The spaces
				   on the stack got above us on it because the
				   (encapsulating) items in those space brought
				   us here. (How can I say this clearly? ;-)).
				   In any case, all those spaces already have
				   their single attribute present in them. */
				do {
					curspc = spcstk[--dstsp];
					dstlst = dststk[dstsp];
				}
				while(curspc->single && dstsp);

				/* Retry now we're one or more levels back */
				continue;
			}

			msg(F_SEND, L_ERR, "meta_buildtree: ERROR: no encaps"
					   "ulation route possible from item "
					   "%d, %s (space %d, %s) to space %d, "
					   "%s!\n",
			    curav->i->nr, curav->i->name, curav->i->spc->nr,
			    curav->i->spc->name, groundspc->nr,groundspc->name);
			return;
		}

		/* We did succeed. Now follow the enc. route that we found,
		   adding the encapsulating items to the destination list,
		   while saving the destination list and space on its stack. */
		while(itmsp) {

			/* Allocate new AV item */
			newav = (META_AV *)malloc(sizeof(META_AV));
			if (!newav) {
				msg(F_MISC, L_ERR, "meta_buildtree: ERROR: "
						   "could not allocate new AV "
						   "item!\n");
				return;
			} 

			/* Pop item for this AV from encapsulating item stack */
			memset(newav, 0, sizeof(META_AV));
			newav->i = itmstk[--itmsp];

			/* Add this item to the current destination list */
			*dstlst = newav;
			dstlst = &(newav->next);

			/* Save the destination list tail (this AV item's next
			   pointer) and the current space */
			dststk[dstsp] = dstlst;
			spcstk[dstsp++] = curspc;

			/* Set the destination list to this item's subitem 
			   list and the current space to this item's subspace */
			dstlst = &(newav->sub);
			curspc = newav->i->subspace;
		}

		/* Now we can retry again. */
	}
}


/* Build a message holding an AV list data in binary or ASCII form */

ssize_t meta_avtomsg(META *m, META_AV *avlst, 
		     char *buf, ssize_t buflen, 
		     int flags, META_AV *acl)
{
	META_AV *av, *aclav;
	META_ORD l, tmpord;
	META_VND *vnd;
	META_VAL *v;
	char *o, *s;
	int t;

	/* Test ascii or binary outside of loop */
	if ((flags & AVMSG_ASCII) == 0) {

		/* Binary interface. Skip first 8 bytes and insert hdr later */
		for(av = avlst, o = buf + 8; av; av = av->next) {

			/* Check if av occurs in ACL, if any */
			if (acl) {
				for(aclav = acl; 
				    aclav && aclav->i != av->i;
				    aclav = aclav->next);
				if (!aclav) continue;
			}

			/* Determine output length first (8 if anonymous ord) */
			if (av->p) l = av->l;
			else l = av->i ? (av->i->val_size > 0 ? av->i->val_size
							      : 0)
				       : 8;

			/* Substract rounded up length + header size from 
			   buffer size and check if OK */
			buflen -= ((l + 3) & ~3) + 16; if (buflen < 0) break;

			/* Put space number in network order in first 4
			   octets. This could be optimized perhaps by 
			   introducing a putint32_aligned or simply using 
			   htonl (which I have avoided so far). */
			putord(o, 4, av->i && av->i->spc ? av->i->spc->nr : -1);
			o += 4;

			/* Put vendor number in next 4 octets */
			putord(o, 4, av->i ? av->i->vnd : -1); o += 4;

			/* Put attribute number in next 4 octets */
			putord(o, 4, av->i ? av->i->nr : -1); o += 4;

			/* Put length in next 4 octets */
			putord(o, 4, l); o += 4;

			/* Put string (av->p) or ordinal data (av->l) in */
			if (av->p) {
				/* Pad on right side with zeros */
				memcpy(o, av->p, l);
				if (l & 3) memset(o + l, 0, 4 - (l & 3));
			}
			else {
				/* Round size of field up get zeros on left */
				putord(o, (l + 3) & ~3, av->l);
			}
			o += (l + 3) & ~3; 

			if ((flags & AVMSG_ONESHOT) == 0) continue; 
			break;
		}

		/* Get the total length */
		l = o - buf;

		/* Put magic and total length in and return the total length */
		putord(buf, 4, C_BINMSG_MAGIC);
		putord(buf + 4, 4, l);
		return l;
	}

	/* Ascii interface */
	for(av = avlst, o = buf; av; av = av->next) {

		/* Check if av occurs in ACL, if any */
		if (acl) {
			for(aclav = acl; 
			    aclav && aclav->i != av->i;
			    aclav = aclav->next);
			if (!aclav) continue;
		}

		/* Test if we're adding tabs in front */
		if (flags & AVMSG_ADDTAB) {
			buflen--; if (buflen < 0) break; 
			*o++ = '\t';
		}

		/* Test if we're writing full attribute names */
		if ((flags & AVMSG_SHORTATTR) == 0) {

			/* Put space name in */
			s = av->i && av->i->spc ? av->i->spc->name : "UNKNOWN";
			l = strlen(s);
			buflen -= l + 1; if (buflen < 0) break;
			memcpy(o, s, l); o += l; *o++ = ':';

			/* Put vendor name in. I now regret a bit I was so 
			   strict in that vendors may have nothing but a PEC, 
			   as getvndbynr() here is the sole reason we need 
			   a pointer to the META object in this function. */
			s = av->i && (vnd = meta_getvndbynr(m, av->i->vnd)) ?
				vnd->name : "UNKNOWN";
			l = strlen(s);
			buflen -= l + 1; if (buflen < 0) break;
			memcpy(o, s, l); o += l; *o++ = ':';
		}

		/* Put attribute name and an equal sign in */
		s = av->i ? av->i->name : "UNKNOWN";
		l = strlen(s);
		buflen -= l; if (buflen < 0) break;
		memcpy(o, s, l); o += l;
		
		/* Put equal sign in, possibly surrounded by spaces */
		if (flags & AVMSG_ADDSPACES) {
			buflen -= 3; if (buflen < 0) break; 
			memcpy(o, " = ", 3); o += 3;
		}
		else {
			buflen--; if (buflen < 0) break; 
			*o++ = '=';
		}

		/* Test if we're showing the value's type as well */
		if (flags & AVMSG_ADDTYPE) {
			s = av->i ? meta_gettypebynr(av->i->val_type):"UNKNOWN";
			l = strlen(s);
			buflen -= l + 1; if (buflen < 0) break;
			memcpy(o, s, l); o += l; *o++ = ':';
		}

		/* Determine length first */
		if (av->p) l = av->l;
		else l = av->i ? (av->i->val_size > 0 ? av->i->val_size
						      : 0)
			       : sizeof(META_ORD);

		/* Test if we're writing hex encoded values (or a temp) */
		if (flags & AVMSG_HEXVALUE || (!av->i && !av->p)) {

			/* Yes. Check if twice the length plus a LF will fit */
			buflen -= (l << 1) + 1; if (buflen < 0) break;

			/* If ordinal value, convert to raw first */
			s = av->p;
			if (!s) { 
				s = (char *)&tmpord; 
				putord(s, l, av->l);
			}

			/* Write out the raw data in hex and add the LF */
			hex(o, s, l); o += (l << 1); *o++ = '\n';

			/* We're done, so continue or break loop here */
			if ((flags & AVMSG_ONESHOT) == 0) continue; 
			break;
		}

		/* Not hex encoded but as humanly readable as possible */
		t = av->i ? av->i->val_type : MT_STRING;
		switch(t) {

		  case MT_INTEGER:
			if (flags & AVMSG_NAMEDCONST &&
			    (v = meta_getvalbynr(m, av->i, av->l))) { 
				l = strlen(v->name);
				buflen -= l; if (buflen < 0) break;
				memcpy(o, v->name, l); o += l;
				break;
			}
			/* Note: fallthrough to decimal if not named */

		  case MT_DATE:
		  	tmpord = meta_ordtoa(o, buflen, 0, 10, av->l);
			o += tmpord; buflen -= tmpord; 
			break;

		  case MT_IPADDR:
		  	tmpord = meta_iptoa(o, buflen, av->l);
			o += tmpord; buflen -= tmpord; 
			break;
		  	
		  case MT_STRING:
		  default:
			/* If actually ordinal, convert to raw first */
		  	s = av->p;
			if (!s) { 
				s = (char *)&tmpord; 
				putord(s, l, av->l);
			}

			/* Write quote */
			buflen--; if (buflen < 0) break; 
			*o++ = '"';

			/* Write characters, encoding non-prt. chars as hex */
			tmpord = meta_atoprt(s, l, 0, 0, 0, 
					     flags & AVMSG_DBLBACKSLASH, 
					     o, buflen);
			o += tmpord; buflen -= tmpord;

			/* Write quote */
			buflen--; if (buflen < 0) break; 
			*o++ = '"';
		}

		/* Write LF */
		buflen--; if (buflen < 0) break; 
		*o++ = '\n';

		/* Exit here and now if this was a one-attribute operation */
		if ((flags & AVMSG_ONESHOT) == 0) continue; 
		return o - buf;
	}

	/* Write extra LF if not one-shot */
	if (buflen > 0) *o++ = '\n';

	return o - buf;
}


/* Add an A/V item to a list before or after the item indicated by rel, or
   at the beginning or the end of the list if no rel item was specified. */

void meta_addav(META_AV **head, META_AV **tail, META_AV *rel, int before,
		META_AV *av)
{
	if (before) {

		/* Add before */
		if (rel) {

			/* Before existing item */
			av->prev = rel->prev;
			av->next = rel;

			/* update previous item's next field, or the list head
			   if there is none */
			if (rel->prev) rel->prev->next = av;
			else *head = av;

			/* update next (existing) item's previous field */
			rel->prev = av;
		}
		else {
			/* At the start of the list */
			av->prev = 0;
			av->next = *head;
			*head = av;

			/* update next item's previous field */
			if (av->next) av->next->prev = av;
		}

		/* update tail if list was empty */
		if (!*tail) *tail = av;
		return;
	}

	/* Add after */
	if (rel) {

		/* After existing item */
		av->prev = rel;
		av->next = rel->next;

		/* update next item's previous field, or list's tail if none */
		if (rel->next) rel->next->prev = av;
		else *tail = av;

		/* update previous item's next field */
		rel->next = av;
	}
	else {
		/* At the end of the list */
		av->prev = *tail;
		av->next = 0;
		*tail = av;

		/* update previous item's next field */
		if (av->prev) av->prev->next = av;
	}

	/* update head if list was empty */
	if (!*head) *head = av;
}


/* Free temporary data that may be held by an AV item */

void meta_freeavdata(META_AV *av)
{
	if ((av->flags & AV_FREE_P) && av->p) {
		free(av->p); av->p = 0; av->flags &= ~AV_FREE_P;
	}
}


/* Free a whole list of AV items and possibly their data */

void meta_freeavlist(META_AV *avlist)
{
	META_AV *av;

	while(avlist) {

		/* Recurse, freeing sublist if any */
		if (avlist->sub) meta_freeavlist(avlist->sub);

		/* Save next, free, and continue with saved list item */
		av = avlist->next;
		meta_freeavdata(avlist);
		free(avlist);
		avlist = av;
	}
}


void meta_printav(META *m, META_AV *av, int depth)
{
	static char buf[32768];
	ssize_t buflen, ret;
	char *o;
	int n;

	buflen = sizeof(buf);
	ret = 0;
	o = buf;
	for(n = 0; n < depth && buflen > 0; n++) {
		*o++ = '\t'; *o++ ='|';
		buflen -= 2; ret += 2;
	}

	n = meta_avtomsg(m, av, o, buflen, AVMSG_ONESHOT + AVMSG_ASCII + 
			 AVMSG_ADDTAB + AVMSG_ADDSPACES + AVMSG_NAMEDCONST, 0);
	buflen -= n; ret += n;

	write(2, buf, ret);
}


void meta_printavlist(META *m, META_AV *avlst, int depth)
{
	META_AV *av;

	for(av = avlst; av; av = av->next) {
		meta_printav(m, av, depth);
		if (av->sub) meta_printavlist(m, av->sub, depth + 1);
	}
}

