/*
 * FASTRING - support for fast ring buffers using the mirror page technique
 *
 * Author:
 * Emile van Bergen, emile@e-advies.nl
 *
 * Permission to redistribute an original or modified version of this program
 * in source, intermediate or object code form is hereby granted exclusively
 * under the terms of the GNU General Public License, version 2. Please see the
 * file COPYING for details, or refer to http://www.gnu.org/copyleft/gpl.html.
 *
 * Theory of operation is as follows. The requirement for functions that 
 * access the ring buffer to support wrapping can be completely avoided by
 * following the buffer memory with a writable image of itself. Standard
 * memory access functions can operate directly on messages of the size of the
 * original buffer, without wrapping or copying, regardless of the value of the
 * read- and write pointers.
 *
 * The offered primitives by this module are: functions to get the amount of
 * free space for writing in the ring and to get the number of bytes available
 * for reading; direct access to the absolute read- and write pointers; and
 * functions to update the read and write pointers after getting and putting.
 *
 * Because reading data and updating the read pointer are done separately, the
 * distinction between peeking at the data and getting the data is removed; you
 * just decide whether or not to call the 'get' macro.
 *
 * The 'put' macro also emulates the mirrored memory in case there is no
 * operating system support for mapping the same memory twice in the program's
 * address space.
 *
 * It is required that you don't write more data than there's room; no bounds
 * checking is done by the 'put' function that updates the pointer as the 
 * damage would already be done anyway. The same goes for reading; you need
 * to call the maxget macro to know how many bytes you may access at the
 * read pointer. 
 *
 * History: 2003/05/08 - EvB - Created
 */

char fastring_id[] = "FASTRING - Copyright (C) 2003 Emile van Bergen.";


#include <unistd.h>		/* for read, write */
#include <stdlib.h>		/* for read, write */
#include <malloc.h>		/* for malloc, free */
#include <string.h>		/* for memset */

#include <fastring.h>


RING *ring_new(size_t size)
{
    RING *ret;

    ret = (RING *)malloc(sizeof(RING)); if (!ret) return 0;
    memset(ret, 0, sizeof(RING));
    if (mirr_new(size, &ret->mp) == 0) { free(ret); return 0; }
    ret->r = ret->w = ret->mp.p;
    return ret;
}

void ring_del(RING *r)
{
    mirr_del(&r->mp);
    free(r);
}
