/*
 * MISC - Miscellaneous utility functions
 *
 * Author:
 * Emile van Bergen, emile@evbergen.xs4all.nl
 *
 * Permission to redistribute an original or modified version of this program
 * in source, intermediate or object code form is hereby granted exclusively
 * under the terms of the GNU General Public License, version 2. Please see the
 * file COPYING for details, or refer to http://www.gnu.org/copyleft/gpl.html.
 *
 * History:
 * 2001/06/25 - EvB - Moved hex and memrchr here from lang_vm.c
 * 2002/04/07 - EvB - Added encryption/decryption functions as used in
 * 		      RFC 2868 (tunneling attrs) and RFC 2548 (MS VSAs)
 * 2002/02/23 - EvB - Added encrypt_attr_pap 
 * 2003/05/03 - EvB - Moved get_random_data here
 * 2003/05/08 - EvB - Added memspn, memcspn
 * 2004/01/13 - EvB - Added STR, cptstr, cpstr, toupperstr, tolowerstr
 * 2004/05/11 - EvB - Split off STR and friends
 */

char misc_id[] = "MISC - Copyright (C) 2001 Emile van Bergen.";


 /*
  * INCLUDES & DEFINES
  */


#include <unistd.h>		/* For read */
#include <stdlib.h>		/* For rand */
#include <malloc.h>		/* For malloc */
#include <string.h>		/* For memset / memcpy / strlen */
#include <errno.h>		/* For errno */
#include <ctype.h>		/* For toupper / tolower */

#include <constants.h>
#include <metadata.h>		/* For putord / getord */
#include <debug.h>
#include <md5.h>

#include <misc.h>


/*
 * FUNCTIONS
 */


/* A malloc that always succeeds. A failed sbrk() or read() I/O error should
   lead to a signal, IMHO, whereas things like SIGPIPE are transient and 
   potentially curable by the program and should therefore use return codes. 
   Oh well, it's easy to fix with this function. */

void *safe_malloc(size_t l)
{
    void *ret;

    ret = malloc(l); if (ret) return ret;
    msg(F_MISC, L_ERR, "FATAL: Out of memory!");
    _exit(0x4f);
}


/* A read that never returns -1. EINTR is handled, the rest ends the program. 
   Not suitable for nonblocking I/O, unless you always select() first. */

int safe_read(int fd, void *p, size_t l)
{
    int ret;

    do {
	ret = read(fd, p, l);
	if (ret != -1) return ret;
    } while (errno == EINTR);
    msg(F_MISC, L_ERR, "FATAL: I/O error: %s", strerror(errno));
    _exit(1);
}


void hex(char *buf, const char *src, ssize_t len)
{
	static char *hextbl = "0123456789abcdef";

	for( ; len > 0; len--) {
		*buf++ = hextbl[(*src >> 4) & 0xf];
		*buf++ = hextbl[*src++ & 0xf];
	}
}


/* Get reasonably good random data */

void get_random_data(char *p, ssize_t len)
{
#ifdef RANDOM_DEV
	int fd;
	ssize_t n;

	if ((fd = open(RANDOM_DEV, O_RDONLY)) != -1 &&
	    (n = read(fd, p, len)) == len &&
	    close(fd) != -1) return;

	msg(F_MISC, L_NOTICE, "Warning: No data from " RANDOM_DEV ", using rand()\n");
#endif
	while(len-- > 0) *p++ = (char)(256.0 * rand() / (RAND_MAX + 1.0));
}

