/*
 * METAOPS - Implements the operations that can be performed performed on 
 *	     individual or lists of data items described by metadata specs
 *
 * Author:
 * Emile van Bergen, emile@evbergen.xs4all.nl
 *
 * Permission to redistribute an original or modified version of this program
 * in source, intermediate or object code form is hereby granted exclusively
 * under the terms of the GNU General Public License, version 2. Please see the
 * file COPYING for details, or refer to http://www.gnu.org/copyleft/gpl.html.
 *
 * History:
 * 2001/03/17 - EvB - Created
 * 2001/05/08 - EvB - Added ord value and temporary flag to META_AV
 * 2001/05/16 - EvB - Added printav / printavlist debugging functions here
 * 2001/06/04 - EvB - Converted META_AV into a double linked list, to ease
 * 		      their use in the VM (adding before / after etc.)
 * 2001/06/25 - EvB - Added meta_avtomsg / meta_msgtoav
 * 2001/09/11 - EvB - Added meta_addav here; used by job_new, chan_handle_read
 *		      and vm_run (we should, at least).
 */


#ifndef __METAOPS_H
#define __METAOPS_H	1


/*
 * INCLUDES & DEFINES
 */


#include <constants.h>
#include <metatype.h>
#include <platform.h>


/* Flags used in the META_AV structure */

#define AV_UNRESOLVED	0x100	/* p and/or l not initialised from i yet */
#define AV_DEL		0x200	/* treat this item as deleted */
#define AV_FREE_P	0x400	/* p must be freed when freeing this AV */


/* Values for (bitfield) members of avref_t */

/* Flags for 'have' member. Indicates whether the list, inst and op members are
 * defined or whether or a situation dependent default is to be used. */

#define AVR_HAVE_LIST	1
#define AVR_HAVE_INST	2
#define AVR_HAVE_OP	4

/* Values for the 'list' member. Specifies referenced AV pair list.  Used in VM
 * attribute references on the stack and in interface ACLs. Don't ever change
 * the value of AV_LIST_REP, as this value is hardcoded in all kinds of
 * arithmetic, including array dereferences. */

#define AVR_LIST_REQ	0	/* request list (default in most cases) */
#define AVR_LIST_REP	1	/* reply list */

/* Values for the 'inst' member. Specifies referenced instance on target pair
 * list. Used in VM attribute references and interface ACLs. Not all values are
 * supported in all contexts. */

#define AVR_INST_LAST	0	/* use last instance (default) */
#define AVR_INST_FIRST	1	/* use first instance */
#define AVR_INST_NO	2	/* no instance (used with ACLs) */

/* Values for the 'inst' member. Specifies operation to be performed on target
 * pair list. Used in VM attribute references and interface ACLs. */

#define AVR_OP_ADD	0	/* add instance (default) */
#define AVR_OP_REPLACE	1	/* replace instance */
#define AVR_OP_IFNEW	2	/* add only if it doesn't exist yet */
#define AVR_OP_DEL	3	/* delete instance */


/* Values for flags parameter in meta_avtomsg - see constants.h for descr. */

#define AVMSG_ONESHOT		C_DV_FLAGS_RESV0 	/* send only one item */

#define AVMSG_ASCII		C_DV_FLAGS_ASCII
#define AVMSG_ADDTAB		C_DV_FLAGS_ADDTAB
#define AVMSG_ADDSPACES		C_DV_FLAGS_ADDSPACES
#define AVMSG_ADDTYPE		C_DV_FLAGS_ADDTYPE
#define AVMSG_HEXVALUE		C_DV_FLAGS_HEXVALUE
#define AVMSG_DBLBACKSLASH	C_DV_FLAGS_DBLBACKSLASH
#define AVMSG_NAMEDCONST	C_DV_FLAGS_NAMEDCONST
#define AVMSG_SHORTATTR		C_DV_FLAGS_SHORTATTR
#define AVMSG_ADDIGNORE		C_DV_FLAGS_ADDIGNORE


/*
 * TYPES
 */


/* Used together with META_AV flags. Note name convention: lowercase_t is
 * used for types are intended to be passed and returned by value; UPPERCASE is
 * used for larger structures. We never define types as pointers. */

typedef struct meta_avref {
	unsigned list:1;	/* Referenced list (REQ or REP) */
	unsigned inst:2;	/* Referenced instance (first, last, none) */
	unsigned op:2;		/* Operation (add, addifnew, replace, del) */

	unsigned have:3;	/* Bitmap specifying which members are defined
				   and which should be taken as default */
} avref_t;


typedef struct meta_av {

	struct meta_av *next;	/* Next item */
	struct meta_av *prev;	/* Previous item - not valid after buildtree */
	struct meta_av *sub;	/* In trees generated by buildtree: head of 
				   child list. In VM stack items: corresponding
				   item on request/reply list. */

	META_ITEM *i;		/* Dictionary item */
	char *p;		/* Data */
	META_ORD l;		/* Length or, if p == 0, data in host order */ 
	int flags;		/* Flags */
	avref_t ref;		/* Used in AV references in stack items, ACLs
				   and module interface receive lists. */

	struct meta_av *treeitem;  /* After calling buildtree, pairs on the
				      source list have this pointing to their
				      counterparts in the dest. tree. */

	char *encp;		/* Only valid in A/V pairs on avlst after
				   calling meta_encode; refers to location of
				   value in encoded packet. Always non-NULL if
				   the A/V pair was encoded at all, regardless
				   of type and/or value of p. Used for signing
				   encoded packets in already encoded 
				   attributes. */

	int rest;		/* Used by buildtree's getcopy's hack to split
				   long attributes according to dictionary */
} META_AV;


/*
 * PROTOTYPES
 */


/* Get a dictionary item and prefix flags based on a spec string */

META_ITEM *meta_getitembyextspec(META *m, char *spec, ssize_t specl, 
				 avref_t *retref);

/* Decoding */

META_AV *meta_decode(META *m, META_SPC *s, META_ORD defvnd, 
		     char *blk, META_ORD blksize, 
		     META_AV **rettail, int copystrings);

/* Encoding */

void meta_buildtree(META_AV *srclst, META_AV **dstlst, META_SPC *groundspc);

META_ORD meta_encode(META_SPC *s, char *blk, META_ORD blksize, META_AV *avlst,
		     META_ORD *retvnd);


/* From/to ASCII/binary messages */

int meta_ascmsgtoav(META *m, char *buf, ssize_t len,
		    META_AV **head, META_AV **tail, int flags);

int meta_binmsgtoav(META *m, U_INT32_T *buf, ssize_t len,
		    META_AV **head, META_AV **tail, int nocopy);

ssize_t meta_avtomsg(META *m, META_AV *avlst, char *buf, ssize_t buflen, 
		     int flags, U_INT32_T binmagic);

/* Misc */

void meta_addav(META_AV **head, META_AV **tail, META_AV *rel, int before,
		META_AV *av);
META_AV *meta_addnewav(META_AV **head, META_AV **tail, META_AV *rel, int before,
		META_ITEM *i, char *p, META_ORD l);
void meta_remav(META_AV **head, META_AV **tail, META_AV *av);
void meta_freeavdata(META_AV *av);
void meta_freeavlist(META_AV *avlist);
void meta_freeavtree(META_AV *avtree);

/* Debugging */

void meta_printav(META *m, META_AV *av, int depth);
void meta_printavlist(META *m, META_AV *avlst, int depth);

#endif
