/*
 * EAPTLS - EAPServer TLS module. This can be used to implement EAP-TLS by 
 *          verifying the client certificate, or as the foundation for
 *	    EAP-TTLS and EAP-PEAP by exporting the TLS payload (or
 *          InnerApplication records) as RADIUS A/V pairs or EAP packets,
 *          respectively.
 *
 * Author:
 * Emile van Bergen, emile@evbergen.xs4all.nl
 *
 * Permission to redistribute an original or modified version of this program
 * in source, intermediate or object code form is hereby granted exclusively
 * under the terms of the GNU General Public License, version 2. Please see the
 * file COPYING for details, or refer to http://www.gnu.org/copyleft/gpl.html.
 *
 * History:
 */

char eaptls_id[] = "EAPTLS - Copyright (C) 2005 Emile van Bergen.";


/*
 * INCLUDES & DEFINES
 */


#include <netinet/in.h>
#include <string.h>
#include <unistd.h>
#include <stdlib.h>
#include <stdio.h>

#include <evblib/sysdefs/byteorder.h>
#include <evblib/strio/strio.h>
#include <evblib/fastring/fastring.h>
#include <evblib/misc/misc.h>

#include <gnutls/gnutls.h>

#include <metadata.h>		/* For META_ORD */
#include <constants.h>		/* Server constants */


/* Calculates how far ahead a is from b on a ring of size mod */

#define MODAHEAD(a, b, mod)	(((mod) + (a) - (b)) % (mod))


/* Some constants */

#define MODULE_VERSION		"v0.1"

#define MAX_CONVERSATIONS	128

#define TLS_BUFFSER_SIZE	32768	    /* Two TLS records */

#define RET_NOMEM		0x40
#define RET_NOCONV		0x41
#define RET_NODATA		0x42


/* Dictionary items not in common/constants.h */

#define C_DI_FRAMED_MTU		12


/*
 * TYPES
 */


typedef struct eaptls_conversation {
    int inuse;
    time_t exp;
    gnutls_session_t session;
    char to_tls[TLS_BUFFER_SIZE], *tor, *tow;
    char from_tls[TLS_BUFFER_SIZE], *fromr, *fromw;
} eaptls_conv_t;


/* 
 * GLOBALS
 */


META_ORD rxspace, txspace;
int debug;


/*
 * CALLBACKS (BRR)
 */


/* Pull data from conversation's to_tls buffer, which is a ring that never
 * wraps, because everything is always pulled before that occurs.  
 * This function ensures tor never passes tow; you must ensure yourself tow
 * never passes to_tls + sizeof(to_tls) */

ssize_t tlspull(gnutls_transport_ptr_t td, void *data, size_t len)
{
    eaptls_conv_t *conv = (eaptls_conv_t *)td;
    if (!conv->inuse) { errno = EBADF; return -1; }

    /* Yet another copy, for no reason other than Gnutls' design. And
     * then Gnutls proceeds to make a few of its own. Sigh. */

    if (len > conv->tow - conv->tor) { errno = EGAGAIN; return -1; }
    memcpy(data, conv->tor, len);
    conv->tor += len;

    /* Reset R and W if everything has been pulled */

    if (conv->tor == conv->tow) conv->tor = conv->tow = 0;
    return len;
}


/* Push data into conversation's from_tls buffer, which is a ring that
 * never wraps, because we always empty it as soon as anything appears here. 
 * If you empty it, don't forget to reset the R and W pointers. */

ssize_t tlspush(gnutls_transport_ptr_t td, const void *data, size_t len)
{
    eaptls_conv_t *conv = (eaptls_conv_t *)td;
    if (!conv->inuse) { errno = EBADF; return -1; }
    
    /* Check if there's room for what Gnutls wants to give us. There
     * always should be, because Gnutls never writes more than two full
     * sized records for one top level API call, as far as we know. 
     * Otherwise, fail the call completely, so we'll get notified. */

    if (conv->fromw + len > conv->from_tls + sizeof(conv->from_tls)) {
	errno = ENOSYS; return -1;
    }

    /* Yet another copy, for no reason other than Gnutls' design. And
     * then Gnutls proceeds to make a few of its own. Sigh. */

    memcpy(conv->fromw, data, len);
    conv->fromw += len;
    return len;
}


/*
 * FUNCTIONS
 */


void usage()
{
    fprintf(stderr, 
"Usage: eaptls [-d] [-s attribute space base number]\n"
"       eaptls -v\n"
"       eaptls -h\n");
    _exit(1);
}


void parseoptions(int argc, char **argv)
{
    int c;

    /* Handle options */
    while((c = getopt(argc, argv, "dhvr:s:")) != -1) {
        switch(c) {
          case 's': rxspace = txspace = atoi(optarg); 
		    if (!txspace) {
			fprintf(stderr, "radldap: Invalid space number '%s'!\n",
				optarg);
			usage();
		    }
		    rxspace++;
		    break;
          case 'd': debug++; break;
          case 'v': fprintf(stderr, "\nEAPTLS module " MODULE_VERSION ". "
               "Copyright (C) 2005 Emile van Bergen / E-Advies.\n\n"
"Permission to redistribute an original or modified version of this program\n"
"in source, intermediate or object code form is hereby granted exclusively\n"
"under the terms of the GNU General Public License, version 2. Please see the\n"
"file COPYING for details, or refer to http://www.gnu.org/copyleft/gpl.html.\n"
"\n");
          case 'h':
          case '?':
            usage();
        }
    }
}


static inline int addint(uint32_t **p, uint32_t atrnr, uint32_t val)
{
    uint32_t *o = *p;

    *o++ = net32(C_DS_INTERNAL);
    *o++ = net32(META_VND_ANY);
    *o++ = net32(atrnr);
    *o++ = net32(4);
    *o++ = net32(val);
    *p = o;
    return 20;
}


#if 0
#define ADDINT(o,nr,val)		\
   (*(*(o))++ = net32(C_DS_INTERNAL),	\
    *(*(o))++ = net32(META_VND_ANY),	\
    *(*(o))++ = net32(nr),		\
    *(*(o))++ = net32(sizeof(val)),     \
    *(*(o))++ = net32(val),		\
    16 + (sizeof(val) + 3) & ~3)
#endif


/*
 * MAIN
 */


int main(int argc, char **argv)
{
    static gnutls_certificate_credentials_t x509_cred;
    static gnutls_dh_params_t dh_params;
    static uint32_t msgbuf[(C_MAX_MSGSIZE >> 2) + 1];
    static eaptls_conv_t convring[MAX_CONVERSATIONS];
    uint32_t spc, vnd, atr, len, *i, *e, *o, tls_flags, mtu;
    uint32_t convr, convw, convnr;
    gnutls_session_t convsess;
    gnutls_datum dat;
    STR_T s, tls_data;

    /* Initialise static data */

    memset(convring, 0, sizeof(convring));

    /* Parse options, setup TLS */

    parseoptions(argc, argv);
    gnutls_global_init();
    gnutls_certificate_allocate_credentials(&x509_cred);
    
    /* Get CAs, CRL, our own keypair */

    gnutls_certificate_set_x509_trust_file(x509_cred, 
	"modules/eaptls/ca.pem", GNUTLS_X509_FMT_PEM);
    gnutls_certificate_set_x509_crl_file(x509_cred, 
	"modules/eaptls/crl.pem", GNUTLS_X509_FMT_PEM);
    gnutls_certificate_set_x509_key_file(x509_cred, 
	"modules/eaptls/cert.pem",
	"modules/eaptls/key.pem", GNUTLS_X509_FMT_PEM);

    /* Get DH parameters - we should read those inside the loop, from a 
     * file that is created upon installation and periodically re-created
     * by cron and reread here each time it's found to be newer than last
     * time it was read. That ensures security even if the module stays
     * running forever, and does not create any delays here. For testing
     * this will do though. */

    s = readstr("modules/eaptls/dh.pem");
    if (s.l < 0) {
	fprintf(stderr, "Can't read DH parameters: %s!\n", strerror(-s.l));
	return 1;
    }
    dat.data = s.p; dat.size = s.l;
    gnutls_dh_params_init(&dh_params);
    gnutls_dh_params_import_pkcs3(dh_params, &dat, GNUTLS_X509_FMT_PEM);
    free(s.p);
    gnutls_certificate_set_dh_params(x509_cred, dh_params);

    /* Request loop */

    convr = convw = 0;
    if (debug) errputcs("eaptls: Ready for requests.\n");
    for(;;) {

        /*
         * Get the request from OpenRADIUS
         */

        /* Read header */
        if (read(0, msgbuf, 8) != 8) { perror("eaptls: read"); break; }
        if (net32(*msgbuf) != 0xbeefdead) {
            fprintf(stderr, "eaptls: Invalid magic 0x%08x!\n", net32(*msgbuf)); 
            break;
        }
        len = net32(msgbuf[1]);
        if (len < 8 || len > sizeof(msgbuf) - 4) {
            fprintf(stderr, "eaptls: Invalid length %d!\n", len); 
            break;
        }

        /* Read rest of message */
        if (read(0, msgbuf + 2, len - 8) != len - 8) {
            perror("eaptls: read"); 
            break;
        }

	/* Initialize defaults */

        convnr = -1;
	tls_flags = ~0;
	tls_data.l = 0;
	mtu = 1230;

        /*
         * Loop through the attributes. The attribute spaces for
         * the Request- and Response-[T]TLS/PEAP-* attributes 
         * depend on the EAP type and are specified on the command
         * line. Note: we parse responses, we send requests. Attribute
         * numbers 0 and 1 correspond to the Flags and TLS Length/Data
         * fields of the EAP TLS packets.
         */

        e = msgbuf + (len >> 2);
        for(i = msgbuf + 2; i < e; i += ((len + 3) >> 2) + 4) {

            /* Get space/vendor/attribute/length tuple */

            spc = net32(i[0]); vnd = net32(i[1]);
            atr = net32(i[2]); len = net32(i[3]);
            if (debug) {
                fprintf(stderr, "eaptls: got space %d, vendor "
                        "%d, attribute %d, len %d\n", spc, vnd, atr, len);
            }

            /* TLS-Data, -Flags attributes from EAP type-specific space */

            if (spc == rxspace && (vnd == 0 || vnd == META_ORD_ERR)) {
                switch (atr) {
                    case 0: tls_flags = net32(i[4]); break;
                    case 1: tls_data.p = (char *)&(i[4]); 
			    tls_data.l = len;
                }
                continue;
            }

            /* Transaction-Id (indexes timer ring containing conversations) */

            if (spc == C_DS_INTERNAL && atr == C_DI_TID) {
                convnr = net32(i[4]);
                continue;
            }

	    /* Framed-MTU, used as maximum amount of data to get from TLS */

            if (spc == C_DS_RAD_ATR && atr == C_DI_FRAMED_MTU) {
                mtu = net32(i[4]);
                continue;
            }
        }

        /* Initialise reply buffer */

        o = msgbuf; *o++ = net32(0xdeadbeef); o++;
        e = msgbuf + sizeof(msgbuf) / sizeof(msgbuf[0]);
        len = 8; 

	/* See if we actually got any TLS data */

	if (tls_flags == ~0) {
	    errputcs("eaptls: no TLS attributes from core - wrong space?\n");
	    len += addint(&o, C_DI_INT, RET_NODATA);
	    goto reply;
	}

        /* See if the request's conversation number is valid */

        if (convnr > MAX_CONVERSATIONS || !convring[convnr].inuse) {

            /* If we don't have room for a new conversation, fail 
               now */

            if (MODAHEAD(convr, convw + 1, MAX_CONVERSATIONS) <= 0) {
		fputs("eaptls: no room for new conversation!\n", stderr);
                len += addint(&o, C_DI_INT, RET_NOCONV);
                goto reply;
            }

            /* Create a new EAP-TLS conversation. A conversation is
             * a series of requests and responses. It does not map
             * 1:1 to a TLS session, because a new conversation may
             * resume an existing TLS session. */

            convnr = convw++;
	    fprintf(stderr, "eaptls: new conversation %u\n", convnr);
            convring[convnr].inuse = 1;

            gnutls_init(&convring[convnr].session, GNUTLS_SERVER);
            convsess = convring[convnr].session;
            gnutls_set_default_priority(convsess);
        }

	/* Feed received data to TLS */

	if (debug) {
	    fprintf(stderr, "eaptls: Got %4d bytes of TLS data, flags 0x%02x, conversation %d, MTU %d\n", tls_data.l, tls_flags, convnr, mtu);
	    hexdumpfd(2, tls_data.p, tls_data.l, 0);
	}

	/* Add the given or new conversation number */

	len += addint(&o, C_DI_TID, convnr);

	/* Put flags from TLS in attribute 0 in txspace */

	tls_flags = 0;
	*o++ = net32(txspace); *o++ = net32(META_VND_ANY); *o++ = net32(0);
	*o++ = net32(1);
	*o++ = net32(tls_flags);
	len += 20;

	/* Put data from TLS in attribute 1 in txspace */

	tls_data = cstr("Hello, world!");
	*o++ = net32(txspace); *o++ = net32(META_VND_ANY); *o++ = net32(1);
	*o++ = net32(tls_data.l);
	memcpy(o, tls_data.p, tls_data.l); o += (tls_data.l + 3) & ~3;
	len += 16 + ((tls_data.l + 3) & ~3);

        /*
         * Send a reply with size 'len'
         */
reply:
	if (debug) {
	    fprintf(stderr, "eaptls: replying with %d bytes of data:\n", len);
	    hexdumpfd(2, msgbuf, len, 0);
	}
        msgbuf[1] = net32(len);
        if (write(1, msgbuf, len) != len) { perror("eaptls: write"); 
                            break; }
    }

    gnutls_certificate_free_credentials(x509_cred);
    gnutls_global_deinit();

    return 1;
}


/*
 * vim:softtabstop=4:sw=4
 */

