/*
 * SRVTYPES - Server data structures
 *
 * Author:
 * Emile van Bergen, emile@evbergen.xs4all.nl
 *
 * Permission to redistribute an original or modified version of this program
 * in source, intermediate or object code form is hereby granted exclusively
 * under the terms of the GNU General Public License, version 2. Please see the
 * file COPYING for details, or refer to http://www.gnu.org/copyleft/gpl.html.
 *
 * History:
 * 2001/06/26 - EvB - Created
 * 2004/07/01 - EvB - Adapted to allow channels to have a window size > 1
 *                    making asynchronous modules possible
 */


#ifndef __SRVTYPES_H
#define __SRVTYPES_H 	1


/*
 * INCLUDES & DEFINES
 */


#include <sys/types.h>	/* For ssize_t, FD_SETSIZE and for socket.h */
#include <sys/socket.h>	/* Also for netinet/in.h */
#include <netinet/in.h>	/* For sockaddr_in */

#include <subprocs.h>	/* For PROC */
#include <metaops.h>	/* For META_ORD, META_NAME (metadata), META_AV */


/*
 * TYPES
 */


/* A source socket as given in the configuration file */

typedef struct sock {
	struct sock *next;
	struct conf *c;		/* inited */
	META_ORD ip, port;	/* inited */
	int fd;			/* not opened */
} SOCK;


/* A channel to a module subprocess used for sending requests to the module
 * and receiving responses. Not usable the other way around; too interwoven
 * with proc state machine. This is basically the message layer above the
 * subproc's stream layer. */

#define job_ring_empty(ch)	((ch)->job_r == (ch)->job_w)
#define job_ring_maxput(ch)	((C_CHAN_JOBRINGSIZE + (ch)->job_r - ((ch)->job_w + 1)) % C_CHAN_JOBRINGSIZE)
#define job_ring_maxget(ch)	((C_CHAN_JOBRINGSIZE + (ch)->job_w - (ch)->job_r) % C_CHAN_JOBRINGSIZE)

typedef struct chan {
	struct chan *next;
	struct iface *iface;	/* inited */
	META_AV *reph, *rept;	/* received reply list */
	int job_r, job_w;	/* read and write indexes into job ring */
	struct job *jobring[C_CHAN_JOBRINGSIZE];
	PROC *proc;		/* inited - needed to give the right proc a */
				/* kick if it's idle and a new req. appears;
				   also to reset the watchdog timer, which
				   now supervises the advancing of the job
				   ring read pointer instead of reception of 
				   single messages. */
} CHAN;


/* A request list, reply list, VM and reply fd and address; life time is
 * reception (job_new) through job_run's reply after vm_run returning VM_HALT */

typedef enum {
	RT_SOCKET = 0, RT_RING, RT_CHAN	
} job_replytype_t;

typedef union {
	struct sockaddr_in sockaddr;
	RING *ring;
	CHAN *chan;
} job_replyaddr_t;

typedef struct job {
	struct job *next, *prev;
	struct vm *vm;
	char *pkt;
	ssize_t pktlen;
	int replyfd;		/* file descriptor we'll reply to */
	job_replytype_t replytype;	/* see RT_ constants above */
	job_replyaddr_t replyaddr;	/* address we'll reply to */
	time_t expiry;		/* expiry time if assigned to a channel. Used
				   to guard module subprocesses. */
} JOB;


/* Callable interface as defined in the configuration file; specifies a 
 * number of channels to subprocs to handle calls, the last used channel for
 * distributing calls round robin, a send- and receive ACL and the interface
 * flags (ASCII, binary, etc). */

typedef struct iface {
	struct iface *next;
	struct conf *c;		/* inited */
	META_NAME name;		/* inited */
	ssize_t namel;		/* inited */
	JOB *sendqh, *sendqt;
	int sendqlen;
	struct meta_av *sendacl, *recvacl;	/* inited */
	CHAN *chans, *rrch;	/* inited */
	int xfertimeout, flags;	/* inited - only really used to pass to procs */
	int window;		/* inited */
	META_ITEM *jobticket;	/* inited */
	META_ITEM *pidattr;	/* inited */
} IFACE;


/* Global data; in a struct that's passed around, so that the functions that
 * need access to it are visible as such. */

typedef struct conf {
	SOCK *sources;		/* inited */
	IFACE *ifaces;		/* inited */
	PROC *procs;		/* inited - list of procs of all interfaces */
	struct insn *confexpr;	/* inited - needed around for const pairs */
	ssize_t confexprlen;	/* inited */
	struct insn *expr;	/* inited */
	ssize_t exprlen;	/* inited */
	META_AV *inithead[2];	/* inited - initial req- and rep lists */
	META_AV *inittail[2];	/* inited - initial req- and rep lists */

	/* These are convenience pointers, to save a lot of getitembynr calls */
	META *m;
	META_SPC *ds_ground, *ds_rad_pkt, *ds_rad_atr, *ds_internal;
	META_ITEM *di_code, *di_authenticator, *di_msg_auth;
	META_ITEM *di_source, *di_ip_source, *di_udp_source;
	META_ITEM *di_dest, *di_ip_dest, *di_udp_dest; 
	META_ITEM *di_timestamp, *di_secret, *di_log_line;
	META_ITEM *di_server_pid, *di_request_nr;
} CONF;


#endif

