/*
 * EAPTLS - EAPServer TLS module. This can be used to implement EAP-TLS by 
 *          verifying the client certificate, or as the foundation for
 *	    EAP-TTLS and EAP-PEAP by exporting the TLS payload (or
 *          InnerApplication records) as RADIUS A/V pairs or EAP packets,
 *          respectively.
 *
 * Author:
 * Emile van Bergen, emile@evbergen.xs4all.nl
 *
 * Permission to redistribute an original or modified version of this program
 * in source, intermediate or object code form is hereby granted exclusively
 * under the terms of the GNU General Public License, version 2. Please see the
 * file COPYING for details, or refer to http://www.gnu.org/copyleft/gpl.html.
 *
 * History:
 */

char eaptls_id[] = "EAPTLS - Copyright (C) 2005 Emile van Bergen.";


/*
 * INCLUDES & DEFINES
 */


#include <sys/uio.h>
#include <netinet/in.h>
#include <string.h>
#include <unistd.h>
#include <stdlib.h>
#include <stdio.h>

#include <evblib/sysdefs/byteorder.h>
#include <evblib/buffer/buffer.h>
#include <evblib/strio/strio.h>
#include <evblib/misc/misc.h>

#include <gnutls/gnutls.h>
#include <gnutls/extra.h>

#include <metadata.h>		/* For META_ORD */
#include <constants.h>		/* Server constants */


/* Calculates how far ahead a is from b on a ring of size mod */

#define MODAHEAD(a, b, mod)	(((mod) + (a) - (b)) % (mod))


/* Some constants */

#define MODULE_VERSION		"v0.1"

#define MAX_CONVERSATIONS	128

#define TLS_BUFFER_SIZE		32768	    /* Two TLS max. size records */


#define RET_ERROR	    	0x40

#define RET_NOMEM		0x40
#define RET_NOCONV		0x41
#define RET_NODATA		0x42
#define RET_NOBUF		0x43


/* Dictionary items not in common/constants.h */

#define C_DI_FRAMED_MTU		12


/*
 * TYPES
 */


typedef enum conv_state {
    TLS_HANDSHAKE, TLS_IA, TLS_APP, TLS_CLOSE
} conv_state_t;


typedef struct conv {
    int inuse;
    conv_state_t state;
    time_t exp;
    gnutls_session_t session;
    BUF_T fromtls, totls;	    /* transport buffers (below TLS) */
} CONV_T;


/* 
 * GLOBALS
 */


char *buf[TLS_BUFFER_SIZE];	    /* scratch buffer */

META_ORD rxspace, txspace;
int debug;


/*
 * CALLBACKS (BRR)
 */


/* Pull data from conversation's to_tls buffer, which is a ring that never
 * wraps, because everything is always pulled before that occurs. */

ssize_t tlspull(gnutls_transport_ptr_t td, void *data, size_t len)
{
    CONV_T *conv = (CONV_T *)td;
    if (!conv->inuse) { errno = EBADF; return -1; }

    /* Another copy, for no reason outside Gnutls' design. And then Gnutls
     * proceeds to make a few more of its own. Sigh. */

    if (len > buf_maxget(&conv->totls)) { errno = EAGAIN; return -1; }
    memcpy(data, conv->totls.r, len);
    buf_get(&conv->totls, len);
    return len;
}


/* Push data into conversation's from_tls buffer, which is a ring that
 * never wraps, because we always empty it as soon as anything appears here. */

ssize_t tlspush(gnutls_transport_ptr_t td, const void *data, size_t len)
{
    CONV_T *conv = (CONV_T *)td;
    if (!conv->inuse) { errno = EBADF; return -1; }
    
    /* Check if there's room for what Gnutls wants to give us. There
     * always should be, because Gnutls never writes more than two full
     * sized records for one top level API call, as far as we know. 
     * Otherwise, fail the call completely, so we'll get notified. */

    if (len > buf_maxput(&conv->fromtls)) { errno = ENOSYS; return -1; }

    /* Another copy, for no reason outside Gnutls' design. And then Gnutls
     * proceeds to make a few more of its own. Sigh. */

    memcpy(conv->fromtls.w, data, len);
    buf_put(&conv->fromtls, len);
    return len;
}


/* Output GnuTLS debugging */

void tlslog(int level, const char *s)
{
    fprintf(stderr, "[%d] %s", level, s);
}


/*
 * FUNCTIONS
 */


void usage()
{
    fprintf(stderr, 
"Usage: eaptls [-d] [-s attribute space base number]\n"
"       eaptls -v\n"
"       eaptls -h\n");
    _exit(1);
}


void parseoptions(int argc, char **argv)
{
    int c;

    /* Handle options */
    while((c = getopt(argc, argv, "dhvr:s:")) != -1) {
        switch(c) {
          case 's': rxspace = txspace = atoi(optarg); 
		    if (!txspace) {
			fprintf(stderr, "radldap: Invalid space number '%s'!\n",
				optarg);
			usage();
		    }
		    rxspace++;
		    break;
          case 'd': debug++; break;
          case 'v': fprintf(stderr, "\nEAPTLS module " MODULE_VERSION ". "
               "Copyright (C) 2005 Emile van Bergen / E-Advies.\n\n"
"Permission to redistribute an original or modified version of this program\n"
"in source, intermediate or object code form is hereby granted exclusively\n"
"under the terms of the GNU General Public License, version 2. Please see the\n"
"file COPYING for details, or refer to http://www.gnu.org/copyleft/gpl.html.\n"
"\n");
          case 'h':
          case '?':
            usage();
        }
    }
}


static inline int addint(uint32_t **p, uint32_t atrnr, uint32_t val)
{
    uint32_t *o = *p;

    *o++ = net32(C_DS_INTERNAL);
    *o++ = net32(META_VND_ANY);
    *o++ = net32(atrnr);
    *o++ = net32(4);
    *o++ = net32(val);
    *p = o;
    return 20;
}


#if 0
#define ADDINT(o,nr,val)		\
   (*(*(o))++ = net32(C_DS_INTERNAL),	\
    *(*(o))++ = net32(META_VND_ANY),	\
    *(*(o))++ = net32(nr),		\
    *(*(o))++ = net32(sizeof(val)),     \
    *(*(o))++ = net32(val),		\
    16 + (sizeof(val) + 3) & ~3)
#endif


/* Proceed with TLS conversation according to its state; may add
 * attributes to module response. */

uint32_t tls_state_machine(uint32_t **rep, CONV_T *conv)
{
    static 
    int ret, gr;

    ret = gr = 0;
    if (debug) 
	fprintf(stderr, "eaptls: tls_state_machine, state %d\n", conv->state);

    /* break on error, return on success */

    switch(conv->state) {
      case TLS_HANDSHAKE:
	gr = gnutls_handshake(conv->session);
	if (debug) 
	    fprintf(stderr, "eaptls: gnutls_handshake returned %d: %s\n",
		    gr, gnutls_strerror(gr));
	if (gr >= 0) { 
	    conv->state = gnutls_ia_handshake_p(conv->session) ? 
			  TLS_IA : TLS_APP; 
	    if (conv->state == TLS_APP) {
		gr = gnutls_record_send(conv->session, "ABCD", 4);
		if (debug) fprintf(stderr, "eaptls: gnutls_record_send returned %d: %s\n", gr, gnutls_strerror(gr));
	    }
	    return ret;
	}
	if (gr == GNUTLS_E_INTERRUPTED || GNUTLS_E_AGAIN) return ret;
	break;

      case TLS_APP:
	gr = gnutls_record_recv(conv->session, buf, sizeof(buf));
	if (debug) 
	    fprintf(stderr, "eaptls: gnutls_record_recv returned %d: %s\n", 
		    gr, gnutls_strerror(gr));
	if (gr < 0) break;
	if (!gr) return ret;
	if (debug) hexdumpfd(2, buf, gr, 0);
	gr = gnutls_record_send(conv->session, buf, gr);  /* we simply echo */
	if (debug) 
	    fprintf(stderr, "eaptls: gnutls_record_send returned %d: %s\n", 
		    gr, gnutls_strerror(gr));
	return ret;

      default:
	fprintf(stderr, "eaptls: Unknown state %d!\n", conv->state);
	/* FIXME: ditch session */
	return ret;
    }

    /* Report GnuTLS error */

    fprintf(stderr, "eaptls: "); gnutls_perror(gr);
    return ret;
}


/*
 * MAIN
 */


int main(int argc, char **argv)
{
    static gnutls_certificate_credentials_t x509_cred;
    static gnutls_dh_params_t dh_params;
    static uint32_t msgbuf[(C_MAX_MSGSIZE >> 2) + 1];
    static CONV_T convring[MAX_CONVERSATIONS];
    CONV_T *conv;
    uint32_t spc, vnd, atr, len, *i, *e, *o; 
    uint32_t convr, convw, convnr;
    uint32_t tls_flags, mtu, retint;
    gnutls_datum dat;
    STR_T s, tls_data;
    struct iovec iov[4];
    ssize_t l;
    int parts, n;

    /* Initialise static data, parse options */

    memset(convring, 0, sizeof(convring));
    parseoptions(argc, argv);

    /* Setup GnuTLS */

    gnutls_global_init();
    gnutls_global_init_extra();
    gnutls_global_set_log_function(tlslog);
    gnutls_global_set_log_level(4);
    gnutls_certificate_allocate_credentials(&x509_cred);
    
    /* Get CAs, CRL, our own keypair */

    gnutls_certificate_set_x509_trust_file(x509_cred, 
	"modules/eaptls/ca.pem", GNUTLS_X509_FMT_PEM);
    gnutls_certificate_set_x509_crl_file(x509_cred, 
	"modules/eaptls/crl.pem", GNUTLS_X509_FMT_PEM);
    gnutls_certificate_set_x509_key_file(x509_cred, 
	"modules/eaptls/cert.pem",
	"modules/eaptls/key.pem", GNUTLS_X509_FMT_PEM);

    /* Get DH parameters - we should read those inside the loop, from a 
     * file that is created upon installation and periodically re-created
     * by cron and reread here each time it's found to be newer than last
     * time it was read. That ensures security even if the module stays
     * running forever, and does not create any delays here. For testing
     * this will do though. */

    s = readstr("modules/eaptls/dh.pem");
    if (s.l < 0) {
	fprintf(stderr, "Can't read DH parameters: %s!\n", strerror(-s.l));
	return 1;
    }
    dat.data = s.p; dat.size = s.l;
    gnutls_dh_params_init(&dh_params);
    gnutls_dh_params_import_pkcs3(dh_params, &dat, GNUTLS_X509_FMT_PEM);
    free(s.p);
    gnutls_certificate_set_dh_params(x509_cred, dh_params);

    /* Request loop */

    convr = convw = 0;
    if (debug) errputcs("eaptls: Ready for requests.\n");
    for(;;) {

        /*
         * Get the request from OpenRADIUS
         */

        /* Read header */
        if (read(0, msgbuf, 8) != 8) { perror("eaptls: read"); break; }
        if (net32(*msgbuf) != 0xbeefdead) {
            fprintf(stderr, "eaptls: Invalid magic 0x%08x!\n", net32(*msgbuf)); 
            break;
        }
        len = net32(msgbuf[1]);
        if (len < 8 || len > sizeof(msgbuf) - 4) {
            fprintf(stderr, "eaptls: Invalid length %d!\n", len); 
            break;
        }

        /* Read rest of message */
        if (read(0, msgbuf + 2, len - 8) != len - 8) {
            perror("eaptls: read"); 
            break;
        }

	/* Initialize defaults */

	retint = convnr = tls_flags = ~0;
	tls_data.l = 0;
	mtu = 1230;

        /*
         * Loop through the attributes. The attribute spaces for
         * the Request- and Response-[T]TLS/PEAP-* attributes 
         * depend on the EAP type and are specified on the command
         * line. Note: we parse responses, we send requests. Attribute
         * numbers 0 and 1 correspond to the Flags and TLS Length/Data
         * fields of the EAP TLS packets.
         */

        e = msgbuf + (len >> 2);
        for(i = msgbuf + 2; i < e; i += ((len + 3) >> 2) + 4) {

            /* Get space/vendor/attribute/length tuple */

            spc = net32(i[0]); vnd = net32(i[1]);
            atr = net32(i[2]); len = net32(i[3]);
            if (debug) {
                fprintf(stderr, "eaptls: got space %d, vendor "
                        "%d, attribute %d, len %d\n", spc, vnd, atr, len);
            }

            /* TLS-Data, -Flags attributes from EAP type-specific space */

            if (spc == rxspace && (vnd == 0 || vnd == META_ORD_ERR)) {
                switch (atr) {
                    case 0: tls_flags = net32(i[4]); break;
                    case 1: tls_data.p = (char *)&(i[4]); 
			    tls_data.l = len;
                }
                continue;
            }

            /* Transaction-Id (indexes timer ring containing conversations) */

            if (spc == C_DS_INTERNAL && atr == C_DI_TID) {
                convnr = net32(i[4]);
                continue;
            }

	    /* Framed-MTU, used as maximum amount of data to get from TLS */

            if (spc == C_DS_RAD_ATR && atr == C_DI_FRAMED_MTU) {
                mtu = net32(i[4]);
                continue;
            }
        }

        /* Initialise reply buffer */

        o = msgbuf; *o++ = net32(0xdeadbeef); o++;
        e = msgbuf + sizeof(msgbuf) / sizeof(msgbuf[0]);
        len = 8; 

	/* See if we actually got any TLS data */

	if (tls_flags == ~0) {
	    errputcs("eaptls: no TLS attributes from core - wrong space?\n");
	    retint = RET_NODATA; goto reply;
	}

        /* See if the request's conversation number is valid */

        if (convnr > MAX_CONVERSATIONS || !convring[convnr].inuse) {

            /* If we don't have room for a new conversation, fail now */

            if (MODAHEAD(convr, convw + 1, MAX_CONVERSATIONS) <= 0) {
		errputcs("eaptls: no room for new conversation!\n");
		retint = RET_NOCONV; goto reply;
            }

            /* Create a new EAP-TLS conversation. A conversation is
             * a series of requests and responses. It does not map
             * 1:1 to a TLS session, because a new conversation may
             * resume an existing TLS session. */

            convnr = convw++;
	    conv = &convring[convnr];
	    conv->inuse = 1;
	    conv->state = TLS_HANDSHAKE;
	    buf_init(&conv->fromtls, TLS_BUFFER_SIZE);
	    buf_init(&conv->totls, TLS_BUFFER_SIZE);

	    /* FIXME: add expiry time */

	    /* Create Gnutls session - may be copied from existing one during
	     * handshake */

            gnutls_init(&conv->session, GNUTLS_SERVER);
            gnutls_set_default_priority(conv->session);
	    gnutls_credentials_set(conv->session, GNUTLS_CRD_CERTIFICATE, 
				   x509_cred);
	    gnutls_transport_set_lowat(conv->session, 0);
	    gnutls_transport_set_push_function(conv->session, tlspush);
	    gnutls_transport_set_pull_function(conv->session, tlspull);
	    gnutls_transport_set_ptr(conv->session, 
				     (gnutls_transport_ptr_t)conv);

	    fprintf(stderr, "eaptls: New conversation %u\n", convnr);
        }

	/* Feed received data to TLS */

	if (debug) {
	    fprintf(stderr, "eaptls: Got %4d bytes of TLS data, flags 0x%02x, MTU %d, conversation %d\n", tls_data.l, tls_flags, mtu, convnr);
	    hexdumpfd(2, tls_data.p, tls_data.l, 0);
	}
	l = buf_maxput(&conv->totls);
	if (tls_data.l > l) {
	    fprintf(stderr, "eaptls: No room for %d bytes in buffer; has %d left!\n", tls_data.l, l);

	    /* We should probably dump this session now */
	    retint = RET_NOBUF; goto reply;
	}
	memcpy(conv->totls.w, tls_data.p, tls_data.l);
	buf_put(&conv->totls, tls_data.l);

	/* Handle TLS according to state machine; may add attributes */

	len += tls_state_machine(&o, conv);

	/* Add the given or new conversation number */

	len += addint(&o, C_DI_TID, convnr);

	/* Put flags from TLS in attribute 0 in txspace */

	tls_flags = 0;
	*o++ = net32(txspace); *o++ = net32(META_VND_ANY); *o++ = net32(0);
	*o++ = net32(1);
	*o++ = net32(tls_flags);
	len += 20;

	/* We're OK now */

	retint = 0;

        /*
         * Send a reply with size 'len'; an 'int' attribute containing the 
	 * status in retint is added, together with the TLS data if there's
	 * any to send and we're not reporting an error.
         */
reply:
	/* Add reply status */

	len += addint(&o, C_DI_INT, retint);

	/* Add header for TLS data attribute, if we'll send any */

	l = 0;
	if (retint < RET_ERROR && (l = buf_maxget(&conv->fromtls))) {
	    if (l > mtu - 8) l = mtu - 8;
	    *o++ = net32(txspace); *o++ = net32(META_VND_ANY); *o++ = net32(1);
	    *o++ = net32(l);
	    len += 16;
	}

	/* Prepare response */

	parts = 0;
	iov[parts].iov_base = msgbuf;			    /* Header */
	iov[parts].iov_len = len;
	parts++;
	if (l) {
	    iov[parts].iov_base = conv->fromtls.r;	    /* TLS data */
	    iov[parts].iov_len = l;
	    parts++;
	    len += l;
	    l = (4 - l) & 3;
	    if (l) {
		iov[parts].iov_base = "\x00\x00\x00\x00";   /* Padding */
		iov[parts].iov_len = l;
		parts++;
		len += l;
	    }
	}

	/* Send it */

        msgbuf[1] = net32(len);
	if (debug) {
	    fprintf(stderr, "eaptls: replying, %d bytes in %d part(s):\n", len, parts);
	    for(n = 0; n < parts; n++)
		hexdumpfd(2, iov[n].iov_base, iov[n].iov_len, n << 12);
	}

        if (writev(1, iov, parts) != len) { perror("eaptls: write"); break; }
	if (parts >= 2) buf_get(&conv->fromtls, iov[1].iov_len);
    }

    gnutls_certificate_free_credentials(x509_cred);
    gnutls_global_deinit();

    return 1;
}


/*
 * vim:softtabstop=4:sw=4
 */

