/*
 * EAPTLS - EAPServer TLS module. This can be used to implement EAP-TLS by 
 *          verifying the client certificate, or as the foundation for
 *	    EAP-TTLS and EAP-PEAP by exporting the TLS payload (or
 *          InnerApplication records) as RADIUS A/V pairs or EAP packets,
 *          respectively.
 *
 * Author:
 * Emile van Bergen, emile@evbergen.xs4all.nl
 *
 * Permission to redistribute an original or modified version of this program
 * in source, intermediate or object code form is hereby granted exclusively
 * under the terms of the GNU General Public License, version 2. Please see the
 * file COPYING for details, or refer to http://www.gnu.org/copyleft/gpl.html.
 *
 * History:
 */

char eaptls_id[] = "EAPTLS - Copyright (C) 2005 Emile van Bergen.";


/*
 * INCLUDES & DEFINES
 */


#include <sys/uio.h>
#include <netinet/in.h>
#include <string.h>
#include <unistd.h>
#include <stdlib.h>
#include <stdio.h>

#include <evblib/sysdefs/byteorder.h>
#include <evblib/buffer/buffer.h>
#include <evblib/strio/strio.h>
#include <evblib/misc/misc.h>

#include <gnutls/gnutls.h>
#include <gnutls/x509.h>
#include <gnutls/extra.h>

#include <metadata.h>		/* For META_ORD */
#include <constants.h>		/* Server constants */


/* Calculates how far ahead a is from b on a ring of size mod */

#define MODAHEAD(a, b, mod)	(((mod) + (a) - (b)) % (mod))
#define MODADD(a, b, mod)	(((a) + (b)) % (mod))


/* Some constants */

#define MODULE_VERSION		"v0.1"

#define MAX_CONVERSATIONS	128

#define TLS_BUFFER_SIZE		32768	    /* Two TLS max. size records */
#define TLS_KEYMAT_SIZE		64

#define RET_ERROR	    	0x40

#define RET_NOMEM		0x40
#define RET_NOCONV		0x41
#define RET_NODATA		0x42
#define RET_NOBUF		0x43


/* Dictionary items not in common/constants.h */

#define C_DI_FRAMED_MTU		12	/* In C_DS_RAD_ATTR */

#define C_DI_TLS_ACTION		51	/* In C_DS_INTERNAL */
#define C_DI_TLS_PAYLOAD	52	
#define C_DI_TLS_KEYMAT		53
#define C_DI_TLS_CLIENT_STATUS	54
#define C_DI_TLS_CLIENT_SUBJECT	55

#define C_DV_TLS_FLAGS_MORE	0x40
#define C_DV_TLS_FLAGS_LENGTH	0x80

#define C_DV_TLS_DROP_RES	1	/* Drop session, mark resumable */
#define C_DV_TLS_DROP_NONRES	2


/*
 * TYPES
 */


/* Typing (excuse the pun) convenience */

typedef uint32_t u32;		    /* Identical semantics */
typedef unsigned char u8;	    /* Identical semantics */


typedef enum conv_state {
    TLS_HANDSHAKE, TLS_IA, TLS_APP, DROPPED
} conv_state_t;


typedef struct conv {
    int inuse;
    conv_state_t state;
    time_t expiry;

    gnutls_session_t session;
    BUF_T fromtls, totls;	    /* transport buffers (below TLS) */
    BUF_T totls_app;		    /* application buffer (above TLS);
				       we don't buffer application dat
				       from TLS, as we send that to OR
				       straight away */ 

    unsigned int clientstatus;	    /* obtained after handshake if cl. auth */
    char keymatbuf[TLS_KEYMAT_SIZE];/* obtained after handshake */
    STR_T keymat;		    /* l initialized to desired size; p set
				       when keying material is obtained */
} CONV_T;


/* State machine actions */

typedef enum { RETURN, APP_GETKEY, APP_RECV, IA_RECV, 
	       PRERROR, ERROR, DROPSESS } action_t;


/* 
 * GLOBALS
 */


/* command line arguments */

META_ORD rx_space, tx_space, decpay_space;
enum { DECPAY_RAW = 1, DECPAY_PEAP = 2, DECPAY_DIAMETER = 4 } decpay_type;
int wantclientcert;
STR_T keymat_label = cstr("client EAP encryption");
int debug;


/* scratch buffer */

char buf[TLS_BUFFER_SIZE];	    


/*
 * CALLBACKS (BRR)
 */


/* Pull data from conversation's to_tls buffer, which is a ring that never
 * wraps, because everything is always pulled before that occurs. */

ssize_t tlspull(gnutls_transport_ptr_t td, void *data, size_t len)
{
    CONV_T *conv = (CONV_T *)td;
    if (!conv->inuse) { errno = EBADF; return -1; }

    /* Another copy, and then Gnutls proceeds to make a few more of its own.
     * Sigh. */

    if (len > buf_maxget(&conv->totls)) { errno = EAGAIN; return -1; }
    memcpy(data, conv->totls.r, len);
    buf_get(&conv->totls, len);
    return len;
}


/* Push data into conversation's from_tls buffer, which is a ring that
 * never wraps, because we always empty it as soon as anything appears here. */

ssize_t tlspush(gnutls_transport_ptr_t td, const void *data, size_t len)
{
    CONV_T *conv = (CONV_T *)td;
    if (!conv->inuse) { errno = EBADF; return -1; }
    
    /* Check if there's room for what Gnutls wants to give us. There
     * always should be, because Gnutls never writes more than two full
     * sized records for one top level API call, as far as we know. 
     * Otherwise, fail the call completely, so we'll get notified. */

    if (len > buf_maxput(&conv->fromtls)) { errno = ENOSYS; return -1; }

    /* Another copy, and then Gnutls proceeds to make a few more of its own.
     * Sigh. */

    memcpy(conv->fromtls.w, data, len);
    buf_put(&conv->fromtls, len);
    return len;
}


/* Output GnuTLS debugging */

void tlslog(int level, const char *s)
{
    fprintf(stderr, "[%d] %s", level, s);
}


/*
 * FUNCTIONS
 */


void usage()
{
    fprintf(stderr, 
"Usage: eaptls [-d] [-s attribute space base number]\n"
"       eaptls -v\n"
"       eaptls -h\n");
    _exit(1);
}


void parseoptions(int argc, char **argv)
{
    int c;

    /* Handle options */

    while((c = getopt(argc, argv, "dhvs:p:t:l:c")) != -1) {
        switch(c) {
	  case 't': switch(*optarg) {
			case 'r': decpay_type |= DECPAY_RAW; break;
			case 'p': decpay_type |= DECPAY_PEAP; break;
			case 't': 
			case 'd': decpay_type |= DECPAY_DIAMETER; break;
			default:  fprintf(stderr, "eaptls: Invalid payload "
					  "decode type '%c'!\n", *optarg);
		    }
		    break;
          case 's': rx_space = tx_space = atoi(optarg); 
		    if (!tx_space) {
			fprintf(stderr, "eaptls: Invalid space number '%s'!\n",
				optarg);
			usage();
		    }
		    rx_space++;
		    break;
	  case 'p': decpay_space = atoi(optarg);
		    if (!decpay_space) {
			fprintf(stderr, "eaptls: Invalid space number '%s'!\n",
				optarg);
			usage();
		    }
		    break;
	  case 'l': keymat_label.p = optarg;
		    keymat_label.l = strlen(optarg);
		    break;
	  case 'c': wantclientcert++; break;
          case 'd': debug++; break;
          case 'v': fprintf(stderr, "\nEAPTLS module " MODULE_VERSION ". "
               "Copyright (C) 2005 Emile van Bergen / E-Advies.\n\n"
"Permission to redistribute an original or modified version of this program\n"
"in source, intermediate or object code form is hereby granted exclusively\n"
"under the terms of the GNU General Public License, version 2. Please see the\n"
"file COPYING for details, or refer to http://www.gnu.org/copyleft/gpl.html.\n"
"\n");
          case 'h':
          case '?':
            usage();
        }
    }
}


int reply_addint(BUF_T *rep, u32 i)
{
    u32 l = 4;

    if (buf_maxput(rep) < l) return 0;
    *(u32 *)rep->w = net32(i);
    buf_put(rep, l);
    return l;
}


int reply_addattrhdr(BUF_T *rep, u32 spc, u32 vnd, u32 atr, u32 len)
{
    u32 *o, l = 16;

    if (buf_maxput(rep) < l) return 0;

    o = (u32 *)rep->w;
    *o++ = net32(spc);
    *o++ = net32(vnd);
    *o++ = net32(atr);
    *o++ = net32(len);
    buf_put(rep, l);
    return l;
}


int reply_addintattr(BUF_T *rep, u32 spc, u32 vnd, u32 atr, u32 val)
{
    u32 *o, l = 20;

    if (buf_maxput(rep) < l) return 0;

    o = (u32 *)rep->w;
    *o++ = net32(spc);
    *o++ = net32(vnd);
    *o++ = net32(atr);
    *o++ = net32(4);
    *o++ = net32(val);
    buf_put(rep, l);
    return l;
}


STR_T reply_startattr(BUF_T *rep)
{
    STR_T ret;

    ret.p = rep->w + 16;
    ret.l = buf_maxput(rep) - 16;
    return ret;
}


int reply_endattr(BUF_T *rep, u32 spc, u32 vnd, u32 atr, u32 len)
{
    u32 *o, l;

    l = (len + 19) & ~3;
    if (buf_maxput(rep) < l) return 0;

    o = (u32 *)rep->w;
    *o++ = net32(spc);
    *o++ = net32(vnd);
    *o++ = net32(atr);
    *o++ = net32(len);
    buf_put(rep, l);
    return l;
}


/* Decode TLS payload into attributes for reply buffer. */

void dec_diameter(BUF_T *repbuf, char *p, ssize_t l)
{
    static char hexbuf[1024];
    char *e = p + l;
    u32 atr, len, vnd;
    u8 flags;

    while(p < e) {
	atr = net32(*(u32 *)p); p += 4;
	len = net32(*(u32 *)p); p += 4;
	flags = len >> 24;
	len &= ((1<<24)-1);
	len -= 8;
	vnd = META_VND_ANY;
	if (flags & 0x80) { vnd = net32(*(u32 *)p); p += 4; len -= 4; }
	if ((signed)len < 0 || len + 16 > buf_maxput(repbuf)) break;

	if (debug) {
	    l = MIN(len, sizeof(hexbuf) >> 1);
	    hex(hexbuf, p, l);
	    fprintf(stderr, "eaptls: decoded attr %lu:%lu:%lu = %.*s (%d)\n",
		    (unsigned long)decpay_space, (unsigned long)vnd, 
		    (unsigned long)atr, l << 1, hexbuf, l);
	}
	reply_addattrhdr(repbuf, decpay_space, vnd, atr, len);
	memcpy(repbuf->w, p, len);
	len = (len + 3) & ~3;
	buf_put(repbuf, len);
	p += len;
    }
}


/* Verify peer (client) certificate. We leave verifying the hostname
 * to OR. We won't send any sensitive data between the handshake and our
 * first reply to OR anyway. This function should only return -1 if the
 * TLS protocol breaks and the connection should be aborted immediately,
 * not if there's anything wrong with the certificate. That should be 
 * returned in conv->clientstatus so that OR can act upon it. */

int verify_peer(BUF_T *repbuf, CONV_T *conv)
{
    gnutls_x509_crt_t cert;
    const gnutls_datum_t *certlist;
    STR_T s;
    int gr, certcount;

    /* This is the default if we exit early. */

    conv->clientstatus = GNUTLS_CERT_INVALID;

    /* Obtain client's certificate chain */

    certcount = 0;
    certlist = gnutls_certificate_get_peers(conv->session, &certcount);
    if (!certlist || certcount <= 0) { 
	if (debug) fprintf(stderr, "eaptls: verify_peer: No client certificates received!\n");
	return 0;
    }

    /* Import first certificate and get subject name into attribute. This is
     * done only once, and for security OR is REQUIRED to check the subject DN
     * against the stored DN for the user. */

    gnutls_x509_crt_init(&cert);
    gr = gnutls_x509_crt_import(cert, &certlist[0], GNUTLS_X509_FMT_DER);
    if (gr < 0) {
	if (debug) {
	    fprintf(stderr, "eaptls: verify_peer: Failed to import client certificate: ");
	    gnutls_perror(gr);
	    hexdumpfd(2, certlist[0].data, certlist[0].size, 0);
	}
	return 0;
    }
    s = reply_startattr(repbuf);
    gnutls_x509_crt_get_dn(cert, s.p, &s.l);
    reply_endattr(repbuf, C_DS_INTERNAL, META_VND_ANY, C_DI_TLS_CLIENT_SUBJECT,
		  s.l);
    gnutls_x509_crt_deinit(cert);

    /* Verify the chain up to one of our trusted roots and put verification
     * status in conversation */

    gr = gnutls_certificate_verify_peers2(conv->session, &conv->clientstatus);
    if (gr < 0) { 
	fprintf(stderr, "eaptls: verify_peer: ");
	gnutls_perror(gr);
	conv->clientstatus = GNUTLS_CERT_INVALID;
    }

    return 0;
}


/* Proceed with TLS conversation according to its state; may add
 * attributes to module response. */

int tls_state_machine(BUF_T *repbuf, CONV_T *conv, int app_action)
{
    action_t action;
    int gr;
    ssize_t l;
    STR_T s;

    gr = 0;
    action = RETURN;

    /* State switch */

    if (debug) 
	fprintf(stderr, "eaptls: tls_state_machine, state %d\n", conv->state);

    switch(conv->state) {
      case TLS_HANDSHAKE:
	gr = gnutls_handshake(conv->session);
	if (debug) fprintf(stderr, "eaptls: gnutls_handshake returned %d: %s\n", gr, gr <= 0 ? gnutls_strerror(gr) : "(n/a)");
	if (gr >= 0) {
	    if (wantclientcert) {
		gr = verify_peer(repbuf, conv);
		if (gr < 0) { action = ERROR; break; }
		if (debug) fprintf(stderr, "eaptls: verify_peer returned %d, certificate status %u\n", gr, conv->clientstatus);
	    }
	    action = gnutls_ia_handshake_p(conv->session) ? IA_RECV : 
		     APP_GETKEY;    /* after IA handshake in case of IA */
	    break;
	}
	if (gr == GNUTLS_E_INTERRUPTED || gr == GNUTLS_E_AGAIN) break;
	action = PRERROR;
	break;

      case TLS_APP:
	action = APP_RECV;
	l = buf_maxget(&conv->totls_app);
	if (!l) break;
	gr = gnutls_record_send(conv->session, conv->totls_app.r, l); 
	if (debug) fprintf(stderr, "eaptls: gnutls_record_send %d returned %d: %s\n", (int)l, gr, gr <= 0 ? gnutls_strerror(gr) : (gr < l ? "short write" : "all data sent"));
	if (gr > 0) { if (gr > l) gr = l; buf_get(&conv->totls_app, gr); }
	if (!gr || gr == GNUTLS_E_INTERRUPTED || gr == GNUTLS_E_AGAIN) break;
	action = PRERROR;
	break;

      default:
	fprintf(stderr, "eaptls: Unknown state %d!\n", conv->state);
	action = ERROR;
    }

    /* Application level action switch. Only process OR's action if the
     * previous step didn't already result in an error action. */

    if (action != ERROR && action != PRERROR) {
	switch(app_action) {

	  case C_DV_TLS_DROP_NONRES:			/* Drop nonresumable */
	    gnutls_db_remove_session(conv->session);
	    /* fallthrough */

	  case C_DV_TLS_DROP_RES:			/* Drop resumable */
	    action = DROPSESS;	    /* we don't care about remaining payload */
	}
    }

    /* Action switch */

    if (debug) 
	fprintf(stderr, "eaptls: tls_state_machine, action %d\n", action);
    switch(action) {

      /* Get keying material from handshake (and proceed with APP_RECV) */

      case APP_GETKEY:
	conv->keymat.p = conv->keymatbuf;
	gr = gnutls_prf(conv->session, keymat_label.l, keymat_label.p, 
			0, 0, 0, conv->keymat.l, conv->keymat.p);
	if (gr) {
	    fprintf(stderr, "eaptls: gnutls_prf failed: "); gnutls_perror(gr);
	    action = ERROR;
	    break;
	}
	if (debug) {
	    fprintf(stderr, "eaptls: generated %d bytes of keying material using label [%.*s]:\n", conv->keymat.l, keymat_label.l, keymat_label.p);
	    hexdumpfd(2, conv->keymat.p, conv->keymat.l, 0);
	}

	/* fall through to APP_RECV */

      /* Receive TLS data straight into reply packet buffer; add header
       * and advance buffer only if we received any data at all */

      case APP_RECV:
	conv->state = TLS_APP;
	s.p = buf; s.l = sizeof(buf);
	if (decpay_type & DECPAY_RAW) {
	    s = reply_startattr(repbuf);
	    if (s.l <= 0) {
		fprintf(stderr, "eaptls: BUG: no room in OR reply buffer\n");
		action = ERROR; 
		break;
	    }
	}
	gr = gnutls_record_recv(conv->session, s.p, s.l);
	if (debug) {
	    fprintf(stderr, "eaptls: gnutls_record_recv %d returned %d: %s\n", s.l, gr, gr < 0 ? gnutls_strerror(gr) : "data:");
	    hexdumpfd(2, s.p, gr, 0);
	}
	if (gr > 0) {
	    if (decpay_type & DECPAY_RAW) reply_endattr(repbuf, C_DS_INTERNAL, META_VND_ANY, C_DI_TLS_PAYLOAD, gr);
	    if (decpay_type & DECPAY_DIAMETER) dec_diameter(repbuf, s.p, gr);
	    break;
	}
	if (gr == GNUTLS_E_INTERRUPTED || gr == GNUTLS_E_AGAIN) break;
	action = PRERROR;
	break;

      case RETURN: 
	break;

      default:
	break;
    }

    /* Action switch, take 2 (errors) */

    switch(action) {
      case PRERROR:
	fprintf(stderr, "eaptls: "); gnutls_perror(gr);
      case ERROR:
      case DROPSESS:
	conv->state = DROPPED;
	break;
      default: 
	break;
    }
    return 0;
}



/*
 * MAIN
 */


int main(int argc, char **argv)
{
    static u32 rxbuf[(C_MAX_MSGSIZE >> 2) + 1];
    static CONV_T convring[MAX_CONVERSATIONS];
    static gnutls_certificate_credentials_t x509_cred;
    static gnutls_dh_params_t dh_params;
    CONV_T *conv;
    STR_T s, tls_data, rx_payload;
    BUF_T txbuf;
    u32 spc, vnd, atr, len, *i, *e;
    u32 convr, convw, convnr;
    u32 rx_flags, tx_flags, mtu, rx_action;
    gnutls_datum dat;
    struct iovec iov[4];
    int parts, n;
    ssize_t l;
    time_t t;

    /* Initialise static data, parse options */

    memset(convring, 0, sizeof(convring));
    if (!buf_init(&txbuf, 65536)) { perror("buf_init"); return 1; }
    parseoptions(argc, argv);

    /* Setup GnuTLS */

    gnutls_global_init();
    gnutls_global_init_extra();
    if (debug) { 
	gnutls_global_set_log_function(tlslog);
	gnutls_global_set_log_level(4);
    }
    gnutls_certificate_allocate_credentials(&x509_cred);
    
    /* Get CAs, CRL, our own keypair */

    gnutls_certificate_set_x509_trust_file(x509_cred, 
	"modules/eaptls/ca.pem", GNUTLS_X509_FMT_PEM);
    gnutls_certificate_set_x509_crl_file(x509_cred, 
	"modules/eaptls/crl.pem", GNUTLS_X509_FMT_PEM);
    gnutls_certificate_set_x509_key_file(x509_cred, 
	"modules/eaptls/cert.pem",
	"modules/eaptls/key.pem", GNUTLS_X509_FMT_PEM);

    /* Get DH parameters - we should read those inside the loop, from a 
     * file that is created upon installation and periodically re-created
     * by cron and reread here each time it's found to be newer than last
     * time it was read. That ensures security even if the module stays
     * running forever, and does not create any delays here. For testing
     * this will do though. */

    s = readstr("modules/eaptls/dh.pem");
    if (s.l < 0) {
	fprintf(stderr, "Can't read DH parameters: %s!\n", strerror(-s.l));
	return 1;
    }
    dat.data = s.p; dat.size = s.l;
    gnutls_dh_params_init(&dh_params);
    gnutls_dh_params_import_pkcs3(dh_params, &dat, GNUTLS_X509_FMT_PEM);
    free(s.p);
    gnutls_certificate_set_dh_params(x509_cred, dh_params);

    /* Request loop */

    convr = convw = 0;
    if (debug) errputcs("eaptls: Ready for requests.\n");
    for(;;) {

        /*
         * Get the request from OpenRADIUS
         */

        /* Read header */
        if (read(0, rxbuf, 8) != 8) { perror("eaptls: read"); break; }
        if (net32(*rxbuf) != 0xbeefdead) {
            fprintf(stderr, "eaptls: Invalid magic 0x%08x!\n", net32(*rxbuf)); 
            break;
        }
        len = net32(rxbuf[1]);
        if (len < 8 || len > sizeof(rxbuf) - 4) {
            fprintf(stderr, "eaptls: Invalid length %d!\n", len); 
            break;
        }

        /* Read rest of message */
        if (read(0, rxbuf + 2, len - 8) != len - 8) {
            perror("eaptls: read"); 
            break;
        }

	/* Initialize defaults */

	t = time(0);
	conv = 0;
	convnr = rx_flags = tx_flags = rx_action = ~0;
	tls_data.l = 0;
	rx_payload.l = 0;
	mtu = 1230;

        /*
         * Loop through the attributes. The attribute spaces for
         * the Request- and Response-[T]TLS/PEAP-* attributes 
         * depend on the EAP type and are specified on the command
         * line. Note: we parse responses, we send requests. Attribute
         * numbers 0 and 1 correspond to the Flags and TLS Length/Data
         * fields of the EAP TLS packets.
         */

        e = rxbuf + (len >> 2);
        for(i = rxbuf + 2; i < e; i += ((len + 3) >> 2) + 4) {

            /* Get space/vendor/attribute/length tuple */

            spc = net32(i[0]); vnd = net32(i[1]);
            atr = net32(i[2]); len = net32(i[3]);
            if (debug) {
                fprintf(stderr, "eaptls: got space %d, vendor "
                        "%d, attribute %d, len %d\n", spc, vnd, atr, len);
            }

            /* Get TLS-Data, -Flags attributes from EAP type-specific space */

            if (spc == rx_space && (vnd == 0 || vnd == META_ORD_ERR)) {
                switch (atr) {
                    case 0: rx_flags = net32(i[4]); break;
                    case 1: tls_data.p = (char *)&(i[4]); 
			    tls_data.l = len;
                }
                continue;
            }

	    /* Get Transaction-Id (indexes timer ring containing
	     * conversations); TLS-Payload (cleartext data to be transmitted
	     * using as TLS Application Data) */

	    if (spc == C_DS_INTERNAL) switch(atr) {
		case C_DI_TID:		convnr = net32(i[4]); 
					continue;
		case C_DI_TLS_ACTION:	rx_action = net32(i[4]); 
					continue;
		case C_DI_TLS_PAYLOAD:	rx_payload.p = (char *)&(i[4]); 
					rx_payload.l = len;
					continue;
	    }

	    /* Get Framed-MTU, used as maximum amount of data to get from TLS at
	     * a time */

            if (spc == C_DS_RAD_ATR && atr == C_DI_FRAMED_MTU) {
                mtu = net32(i[4]);
                continue;
            }
        }

        /* Initialise reply buffer */

	*(u32 *)txbuf.w = net32(0xdeadbeef);	/* length done later */
	buf_put(&txbuf, 8);

        /* Obtain new conversation number if none given or given one invalid */

        if (convnr > MAX_CONVERSATIONS || !convring[convnr].inuse) {

            /* If we don't have room for a new conversation, fail now */

            if (MODAHEAD(convr, convw + 1, MAX_CONVERSATIONS) <= 0) {
		fprintf(stderr, "eaptls: Exceeding maximum number of simultaneous conversations (%d)!\n", MAX_CONVERSATIONS);
		conv = 0;
		goto reply;
            }

	    /* If the client didn't provide any transport data (client hello),
	     * we can simply skip this. */

	    if (!tls_data.l) {
		fprintf(stderr, "eaptls: Didn't see ClientHello - not starting conversation without greeting, sorry.\n");
		conv = 0;
		goto reply;
	    }

	    /* Create a new EAP-TLS conversation. A conversation is a series of
	     * requests and responses. It does not map 1:1 to a TLS session; a
	     * new conversation may resume an existing TLS session. It does not
	     * map 1:1 to an user authentication either; one user
	     * authentication may involve multiple TLS conversations (tunnel
	     * and one or more inner conversations in case of TTLS and PEAP),
	     * but that's done on a higher layer. */

            convnr = convw; 
	    conv = &convring[convnr];
	    if (!buf_init(&conv->fromtls, TLS_BUFFER_SIZE) ||
	        !buf_init(&conv->totls, TLS_BUFFER_SIZE) ||
	        !buf_init(&conv->totls_app, TLS_BUFFER_SIZE)) {
		buf_done(&conv->fromtls);
		buf_done(&conv->totls);
		buf_done(&conv->totls_app);
		errputcs("eaptls: no memory!\n");
		conv = 0;
		goto reply;
	    }

	    convw = MODADD(convw, 1, MAX_CONVERSATIONS);
	    conv->inuse = 1;
	    conv->state = TLS_HANDSHAKE;
	    conv->expiry = t + 60;
	    conv->keymat.p = 0;
	    conv->keymat.l = sizeof(conv->keymatbuf);
	    conv->clientstatus = ~0;

	    /* Create Gnutls session - may be resumed from existing one during
	     * handshake */

            gnutls_init(&conv->session, GNUTLS_SERVER);
            gnutls_set_default_priority(conv->session);
	    gnutls_credentials_set(conv->session, GNUTLS_CRD_CERTIFICATE, 
				   x509_cred);
	    if (wantclientcert)
		gnutls_certificate_server_set_request(conv->session, 
						      GNUTLS_CERT_REQUEST);
	    gnutls_transport_set_lowat(conv->session, 0);
	    gnutls_transport_set_push_function(conv->session, tlspush);
	    gnutls_transport_set_pull_function(conv->session, tlspull);
	    gnutls_transport_set_ptr(conv->session, 
				     (gnutls_transport_ptr_t)conv);

	    if (debug) fprintf(stderr, "eaptls: New conversation %u, count now %u\n", convnr, MODAHEAD(convw, convr, MAX_CONVERSATIONS));
        }
	conv = &convring[convnr];

	/* Skip length field in TLS data if length flag set */

	if (rx_flags & C_DV_TLS_FLAGS_LENGTH) {
	    tls_data.p += 4;
	    tls_data.l -= 4;
	}

	/* Show what we got */

	if (debug) {
	    fprintf(stderr, "eaptls: got %4d bytes of transport data, flags 0x%02x, MTU %d, conversation %d:\n", tls_data.l, rx_flags, mtu, convnr);
	    hexdumpfd(2, tls_data.p, tls_data.l, 0);
	    fprintf(stderr, "eaptls: got %4d bytes of payload data, action %d:\n", rx_payload.l, rx_action);
	    hexdumpfd(2, rx_payload.p, rx_payload.l, 0);
	}

	/* Process it */

	if (tls_data.l) { 

	    /* Feed TLS data to TLS transport buffer, from where GnuTLS will
	     * pull it using the gnutls_transport_read callback */

	    l = buf_maxput(&conv->totls);
	    if (tls_data.l > l) {
		fprintf(stderr, "eaptls: No room for %d bytes in transport "
				"buffer; has %d left!\n", tls_data.l, l);
		conv->state = DROPPED;		/* do cleanup, also causes
						   OR to reject */
		goto reply;
	    }
	    memcpy(conv->totls.w, tls_data.p, tls_data.l);
	    buf_put(&conv->totls, tls_data.l);
	}
	if (rx_payload.l) {

	    /* Feed payload data to TLS application buffer, where it's kept
	     * until our state machine determines it's ready to send TLS
	     * Application or InnerApplication records and it's all sent
	     * in one go. */

	    l = buf_maxput(&conv->totls_app);
	    if (rx_payload.l > l) {
		fprintf(stderr, "eaptls: No room for %d bytes in payload "
				"buffer; has %d left!\n", rx_payload.l, l);
		conv->state = DROPPED;		/* do cleanup, also causes
						   OR to reject */
		goto reply;
	    }
	    memcpy(conv->totls_app.w, rx_payload.p, rx_payload.l);
	    buf_put(&conv->totls_app, rx_payload.l);
	}

	/* Work TLS engine according to our state machine, unless the 'more'
	 * flag was set by the sender. May add attributes to response. */

	if (!(rx_flags & C_DV_TLS_FLAGS_MORE))
	    tls_state_machine(&txbuf, conv, rx_action);

reply:
	/* If we have a conversation, put the relevant data in the reply
	 * buffer. 'len' will hold the amount to add directly from the
	 * TLS transport buffer. */

	len = 0;
	if (conv) {

	    /* Echo our conversation number, unless we're about to clean it */

	    if (conv->state != DROPPED) 
		reply_addintattr(&txbuf, C_DS_INTERNAL, META_VND_ANY, C_DI_TID, 
				 convnr);

	    /* Add the keying material, if we have any */

	    if (conv->keymat.p) {
		s = reply_startattr(&txbuf);
		if (s.l >= conv->keymat.l) {
		    memcpy(s.p, conv->keymat.p, conv->keymat.l);
		    reply_endattr(&txbuf, C_DS_INTERNAL, META_VND_ANY, 
				  C_DI_TLS_KEYMAT, conv->keymat.l);
		}
	    }

	    /* Add the client status, if we have any */

	    if (conv->clientstatus != ~0)
		reply_addintattr(&txbuf, C_DS_INTERNAL, META_VND_ANY,
				 C_DI_TLS_CLIENT_STATUS, conv->clientstatus);

	    /* If the MTU is at least 256 and the other side has finished
	     * and we have data in our buffer, add a header for it. This must
	     * be final, because the data itself will be written directly
	     * from the conversation's transport buffer. */

	    if (mtu >= 256 &&
		!(rx_flags & C_DV_TLS_FLAGS_MORE) && 
		(len = buf_maxget(&conv->fromtls))) {

		/* If this is the first part of our transmission, set
		 * length flag in tx_flags and add a length field at
		 * the start of the TLS data. */

		tx_flags = 0;
		if (conv->fromtls.r == conv->fromtls.bu.p) {
		    tx_flags = C_DV_TLS_FLAGS_LENGTH;
		}

		/* If we can send less than our whole buffer, set the
		 * more flag in tx_flags. */

		if (len > mtu - 12) {
		    len = mtu - 12;
		    tx_flags |= C_DV_TLS_FLAGS_MORE;
		}

		/* Add the TLS flags in txspace, attribute 0 */

		reply_addintattr(&txbuf, tx_space, META_VND_ANY, 0, tx_flags);

		/* Add a header for the TLS length and data in atr. 1 */

		if (tx_flags & C_DV_TLS_FLAGS_LENGTH) {
		    reply_addattrhdr(&txbuf, tx_space, META_VND_ANY, 1, len + 4);
		    reply_addint(&txbuf, conv->fromtls.w - conv->fromtls.bu.p);
		}
		else {
		    reply_addattrhdr(&txbuf, tx_space, META_VND_ANY, 1, len);
		}
	    }	
	}

	/* Start to prepare response */

	parts = 0; l = 0;

	/* Reply buffer incl. header for TLS transport data and padding,
	 * minus TLS transport data and padding */

	iov[parts].iov_base = txbuf.r;
	l += iov[parts].iov_len = buf_maxget(&txbuf);
	parts++; 

	if (len) {

	    /* TLS transport data */

	    iov[parts].iov_base = conv->fromtls.r;
	    l += iov[parts].iov_len = len;
	    parts++;

	    len = (4 - len) & 3;
	    if (len) {

		/* Padding */

		iov[parts].iov_base = "\x00\x00\x00\x00";  
		l += iov[parts].iov_len = len;
		parts++;
	    }
	}

	/* Send it; advance conversation's TLS transport buffer; 
	 * empty reply buffer */

	((u32 *)txbuf.r)[1] = net32(l);
	if (debug) {
	    fprintf(stderr, "eaptls: replying to parent with %d bytes in %d part(s):\n", l, parts);
	    for(n = 0; n < parts; n++)
		hexdumpfd(2, iov[n].iov_base, iov[n].iov_len, 0);
	}

        if (writev(1, iov, parts) != l) { perror("eaptls: write"); break; }
	if (conv && parts >= 2) buf_get(&conv->fromtls, iov[1].iov_len);
	buf_empty(&txbuf);

	/* If the conversation was dropped, clean it up its memory */

	if (conv && conv->state == DROPPED) {
	    if (debug) 
		fprintf(stderr, "eaptls: Conversation %d finished\n", convnr);
	    gnutls_deinit(conv->session);
	    buf_done(&conv->fromtls);
	    buf_done(&conv->totls);
	    buf_done(&conv->totls_app);
	    conv->inuse = 0;

	    /* Check if we happened to close the tail of the ring */

	    if (convnr == convr) {	    
		convr = MODADD(convr, 1, MAX_CONVERSATIONS);
		if (debug) fprintf(stderr, "eaptls: Conversation ring read pointer advanced to %d, %d conversations left\n", convr, MODAHEAD(convw, convr, MAX_CONVERSATIONS));
	    }
	}

	/* Clean up all expired conversations from the start of the ring */

	while(MODAHEAD(convw, convr, MAX_CONVERSATIONS) > 0 &&
	      convring[convr].expiry <= t) {

	    if (!convring[convr].inuse) {
		fprintf(stderr, "eaptls: BUG: Something failed to advance conversation ring read pointer!\n");
		convr = MODADD(convr, 1, MAX_CONVERSATIONS);
		continue;
	    }
	    if (debug)
		fprintf(stderr, "eaptls: Discarding expired conversation %d, %d conversations left\n", convr, MODAHEAD(convw, convr, MAX_CONVERSATIONS));
	    gnutls_deinit(convring[convr].session);
	    buf_done(&convring[convr].fromtls);
	    buf_done(&convring[convr].totls);
	    buf_done(&convring[convr].totls_app);
	    convring[convr].inuse = 0;

	    convr = MODADD(convr, 1, MAX_CONVERSATIONS);
	    if (debug) fprintf(stderr, "eaptls: Conversation ring read pointer advanced to %d, %d conversations left\n", convr, MODAHEAD(convw, convr, MAX_CONVERSATIONS));
	}
    }

    gnutls_certificate_free_credentials(x509_cred);
    gnutls_global_deinit();

    return 1;
}


/*
 * vim:softtabstop=4:sw=4
 */

