/*
 * acct.c	Accounting routines.
 *
 */
char acct_sccsid[] =
"@(#)acct.c	2.14 Copyright 1994 Livingston Enterprises Inc\n"
"		     Copyright 1999,2000 Cistron Internet Services B.V.";


#include	<sys/types.h>
#include	<sys/socket.h>
#include	<sys/time.h>
#include	<sys/file.h>
#include	<sys/stat.h>
#include	<netinet/in.h>

#include	<stdio.h>
#include	<stdlib.h>
#include	<netdb.h>
#include	<pwd.h>
#include	<time.h>
#include	<ctype.h>
#include	<unistd.h>
#include	<signal.h>
#include	<errno.h>
#include	<sys/wait.h>

#include	"radiusd.h"
#include	"radutmp.h"


#ifdef XTRADIUS
static char *exec_log_program;
#endif

/*
 *	FIXME: this should be configurable.
 */
int	doradwtmp = 1;

static char trans[64] =
   "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/";
#define ENC(c) trans[c]

static char porttypes[] = "ASITX";

#define LOCK_LEN sizeof(struct radutmp)

/*
 *	used for caching radutmp lookups.
 */
typedef struct nas_port {
	UINT4			nasaddr;
	int			port;
	off_t			offset;
	struct nas_port		*next;
} NAS_PORT;
static NAS_PORT *nas_port_list = NULL;


/*
 *	Lookup a NAS_PORT in the nas_port_list
 */
static NAS_PORT *nas_port_find(UINT4 nasaddr, int port)
{
	NAS_PORT	*cl;

	for(cl = nas_port_list; cl; cl = cl->next)
		if (nasaddr == cl->nasaddr &&
			port == cl->port)
			break;
	return cl;
}


/*
 *	UUencode 4 bits base64. We use this to turn a 4 byte field
 *	(an IP adres) into 6 bytes of ASCII. This is used for the
 *	wtmp file if we didn't find a short name in the naslist file.
 */
char *uue(void *in)
{
	int i;
	static unsigned char res[7];
	unsigned char *data = (char *)in;

	res[0] = ENC( data[0] >> 2 );
	res[1] = ENC( ((data[0] << 4) & 060) + ((data[1] >> 4) & 017) );
	res[2] = ENC( ((data[1] << 2) & 074) + ((data[2] >> 6) & 03) );
	res[3] = ENC( data[2] & 077 );

	res[4] = ENC( data[3] >> 2 );
	res[5] = ENC( (data[3] << 4) & 060 );
	res[6] = 0;

	for(i = 0; i < 6; i++) {
		if (res[i] == ' ') res[i] = '`';
		if (res[i] < 32 || res[i] > 127)
			printf("uue: protocol error ?!\n");
	}
	return res;
}


/*
 *	Convert a struct radutmp to a normal struct utmp
 *	as good as we possibly can.
 */
static void make_wtmp(struct radutmp *ut, struct utmp *wt, int status)
{
	char		buf[32];
	NAS		*cl;
	char		*s;

	/*
	 *	Fill out the UTMP struct for the radwtmp file.
	 *	(this one must be "last" - compatible).
	 */
#ifdef __linux__
	/*
	 *	Linux has a field for the client address.
	 */
	wt->ut_addr = ut->framed_address;
#endif
	/*
	 *	We use the tty field to store the terminal servers' port
	 *	and address so that the tty field is unique.
	 */
	s = "";
	if ((cl = nas_find(ntohl(ut->nas_address))) != NULL)
		s = cl->shortname;
	if (s == NULL || s[0] == 0) s = uue(&(ut->nas_address));
#if UT_LINESIZE > 9
	sprintf(buf, "%03d:%.20s", ut->nas_port, s);
#else
	sprintf(buf, "%02d%.20s", ut->nas_port, s);
#endif
	strncpy(wt->ut_line, buf, UT_LINESIZE);

	/*
	 *	We store the dynamic IP address in the hostname field.
	 */
#ifdef UT_HOSTSIZE
	if (ut->framed_address) {
		ipaddr2str(buf, ntohl(ut->framed_address));
		strncpy(wt->ut_host, buf, UT_HOSTSIZE);
	}
#endif
#ifdef __svr4__
	wt->ut_xtime = ut->time;
#else
	wt->ut_time = ut->time;
#endif
#ifdef USER_PROCESS
	/*
	 *	And we can use the ID field to store
	 *	the protocol.
	 */
	if (ut->proto == P_PPP)
		strcpy(wt->ut_id, "P");
	else if (ut->proto == P_SLIP)
		strcpy(wt->ut_id, "S");
	else
		strcpy(wt->ut_id, "T");
	wt->ut_type = status == PW_STATUS_STOP ? DEAD_PROCESS : USER_PROCESS;
#endif
	if (status == PW_STATUS_STOP)
		wt->ut_name[0] = 0;
}


/*
 *	Zap a user, or all users on a NAS, from the radutmp file.
 */
int radzap(UINT4 nasaddr, int port, char *user, time_t t)
{
	struct radutmp	u;
	struct utmp	wt;
	FILE		*fp;
	int		fd;
	UINT4		netaddr;

	if (t == 0) time(&t);
	fp = fopen(RADWTMP, "a");
	netaddr = htonl(nasaddr);

	if ((fd = open(RADUTMP, O_RDWR|O_CREAT, 0644)) >= 0) {
		int r;

		/*
		 *	Lock the utmp file, prefer lockf() over flock().
		 */
#if defined(F_LOCK) && !defined(BSD)
		(void)lockf(fd, F_LOCK, LOCK_LEN);
#else
		(void)flock(fd, LOCK_EX);
#endif
	 	/*
		 *	Find the entry for this NAS / portno combination.
		 */
		r = 0;
		while (read(fd, &u, sizeof(u)) == sizeof(u)) {
			if (((nasaddr != 0 && netaddr != u.nas_address) ||
			      (port >= 0   && port    != u.nas_port) ||
			      (user != NULL && strcmp(u.login, user) != 0) ||
			       u.type != P_LOGIN))
				continue;
			/*
			 *	Match. Zap it.
			 */
			if (lseek(fd, -(off_t)sizeof(u), SEEK_CUR) < 0) {
				log(L_ERR, "Accounting: radzap: negative lseek!\n");
				lseek(fd, (off_t)0, SEEK_SET);
			}
			u.type = P_IDLE;
			u.time = t;
			write(fd, &u, sizeof(u));

			/*
			 *	Add a logout entry to the wtmp file.
			 */
			if (fp != NULL)  {
				make_wtmp(&u, &wt, PW_STATUS_STOP);
				fwrite(&wt, sizeof(wt), 1, fp);
			}
		}
		close(fd);
	}
	if (fp) fclose(fp);

	return 0;
}


/*
 *	Store logins in the RADIUS utmp file.
 */
int rad_accounting_new(AUTH_REQ *authreq, int dowtmp)
{
	struct radutmp	ut, u;
	struct utmp	wt;
	VALUE_PAIR	*vp;
	int		rb_record = 0;
	int		status = -1;
	int		nas_address = 0;
	int		framed_address = 0;
	int		protocol = -1;
	FILE		*fp;
	time_t		t;
	int		fd;
	int		ret = 0;
	int		just_an_update = 0;
	int		port_seen = 0;
	int		nas_port_type = 0;
	int		off;
#ifdef XTRADIUS
	int		exec_account = 0;
	char	*exec_account_program = NULL;
	char	*exec_nas_program = NULL;
	char	uname[256];
	VALUE_PAIR *user_check = NULL;
	VALUE_PAIR *user_reply = NULL;
#endif

	/*
	 *	Which type is this.
	 */
	if ((vp = pairfind(authreq->request, PW_ACCT_STATUS_TYPE)) == NULL) {
		log(L_ERR, "Accounting: no Accounting-Status-Type record.");
		return -1;
	}
	status = vp->lvalue;
	if (status == PW_STATUS_ACCOUNTING_ON ||
	    status == PW_STATUS_ACCOUNTING_OFF) rb_record = 1;

	if (!rb_record &&
	    (vp = pairfind(authreq->request, PW_USER_NAME)) == NULL) do {
		int check1 = 0;
		int check2 = 0;

		/*
		 *	ComOS (up to and including 3.5.1b20) does not send
		 *	standard PW_STATUS_ACCOUNTING_XXX messages.
		 *
		 *	Check for:  o no Acct-Session-Time, or time of 0
		 *		    o Acct-Session-Id of "00000000".
		 *
		 *	We could also check for NAS-Port, that attribute
		 *	should NOT be present (but we don't right now).
	 	 */
		if ((vp = pairfind(authreq->request, PW_ACCT_SESSION_TIME))
		     == NULL || vp->lvalue == 0)
			check1 = 1;
		if ((vp = pairfind(authreq->request, PW_ACCT_SESSION_ID))
		     != NULL && vp->length == 8 &&
		     memcmp(vp->strvalue, "00000000", 8) == 0)
			check2 = 1;
		if (check1 == 0 || check2 == 0) {
#if 0 /* Cisco sometimes sends START records without username. */
			log(L_ERR, "Accounting: no username in record");
			return -1;
#else
			break;
#endif
		}
		log(L_INFO, "Accounting: converting reboot records.");
		if (status == PW_STATUS_STOP)
			status = PW_STATUS_ACCOUNTING_OFF;
		if (status == PW_STATUS_START)
			status = PW_STATUS_ACCOUNTING_ON;
		rb_record = 1;
	} while(0);

#ifdef NT_DOMAIN_HACK
        if (!rb_record && vp) {
		char buffer[AUTH_STRING_LEN];
		char *ptr;
		if ((ptr = strchr(vp->strvalue, '\\')) != NULL) {
			strncpy(buffer, ptr + 1, sizeof(buffer));
			buffer[sizeof(buffer) - 1] = 0;
			strcpy(vp->strvalue, buffer);
		}
	}
#endif

	/*
	 *	Add any specific attributes for this username.
	 */
	if (!rb_record && vp != NULL) {
		hints_setup(authreq->request);
		presuf_setup(authreq->request);
	}

#ifdef XTRADIUS
		/* XTRADIUS: AccountingON - AccountingOFF (NAS Reboot Records) haven't user name attribute 
			Let's fill it out :)
		*/

	if (pairfind(authreq->request, PW_USER_NAME) != NULL)
		strcpy(uname,(pairfind(authreq->request, PW_USER_NAME))->strvalue);
	else 
		strcpy(uname,"dummyusername");

	if (user_find(uname, authreq->request, &user_check, &user_reply) ==  0)
	{
		pairmove(&(authreq->request), &user_reply);
		pairfree(user_reply);
		pairfree(user_check);
	}
#endif


	time(&t);
	memset(&ut, 0, sizeof(ut));
	memset(&wt, 0, sizeof(wt));
	ut.porttype = 'A';

	/*
	 *	First, find the interesting attributes.
	 */
	for (vp = authreq->request; vp; vp = vp->next) {
		switch (vp->attribute) {
			case PW_USER_NAME:
				strncpy(ut.login, vp->strvalue, RUT_NAMESIZE);
				strncpy(wt.ut_name, vp->strvalue, UT_NAMESIZE);
				break;
			case PW_LOGIN_IP_HOST:
			case PW_FRAMED_IP_ADDRESS:
				framed_address = vp->lvalue;
				ut.framed_address = htonl(vp->lvalue);
				break;
			case PW_FRAMED_PROTOCOL:
				protocol = vp->lvalue;
				break;
			case PW_NAS_IP_ADDRESS:
				nas_address = vp->lvalue;
				ut.nas_address = htonl(vp->lvalue);
				break;
			case PW_NAS_PORT_ID:
				ut.nas_port = vp->lvalue;
				port_seen = 1;
				break;
			case PW_ACCT_DELAY_TIME:
				ut.delay = vp->lvalue;
				break;
			case PW_ACCT_SESSION_ID:
				/*
				 *	If length > 8, only store the
				 *	last 8 bytes.
				 */
				off = vp->length - sizeof(ut.session_id);
				/*
				 *	Ascend is br0ken - it adds a \0
				 *	to the end of any string.
				 *	Compensate.
				 */
				if (vp->length > 0 &&
				    vp->strvalue[vp->length - 1] == 0)
					off--;
				if (off < 0) off = 0;
				memcpy(ut.session_id, vp->strvalue + off,
					sizeof(ut.session_id));
				break;
			case PW_NAS_PORT_TYPE:
				if (vp->lvalue >= 0 && vp->lvalue <= 4)
					ut.porttype = porttypes[vp->lvalue];
				nas_port_type = vp->lvalue;
				break;
			case PW_CALLING_STATION_ID:
				strncpy(ut.caller_id, vp->strvalue,
					sizeof(ut.caller_id));
				ut.caller_id[sizeof(ut.caller_id)] = 0;
				break;
#ifdef XTRADIUS
			case PW_EXEC_PROGRAM_ACCOUNT:
				exec_account = 1;
				exec_account_program = strdup(vp->strvalue);
				break;
			case PW_EXEC_PROGRAM_NAS:
				exec_nas_program = strdup(vp->strvalue);
				break;
			case PW_EXEC_PROGRAM_LOG:
				exec_log_program = strdup(vp->strvalue);
				break;
#endif
		}
	}

	/*
	 *	If we didn't find out the NAS address, use the
	 *	originator's IP address.
	 */
	if (nas_address == 0) {
		nas_address = authreq->ipaddr;
		ut.nas_address = htonl(nas_address);
	}

	if (protocol == PW_PPP)
		ut.proto = 'P';
	else if (protocol == PW_SLIP)
		ut.proto = 'S';
	else
		ut.proto = 'T';
	ut.time = t - ut.delay;
	make_wtmp(&ut, &wt, status);

	/*
	 *	See if this was a portmaster reboot.
	 */
	if (status == PW_STATUS_ACCOUNTING_ON && nas_address) {
		log(L_INFO, "NAS %s restarted (Accounting-On packet seen)",
			nas_name(nas_address));
		radzap(nas_address, -1, NULL, ut.time);
#ifdef XTRADIUS
		if (exec_nas_program != NULL)
		{
			exec_nas_program = set_session_id(exec_nas_program, authreq->id);
			exec_nas_program = radius_xlate(exec_nas_program,authreq->request, authreq->request);
			radius_exec_program(exec_nas_program,authreq->request,&authreq->request,0,NULL);
			log(L_INFO, "NAS On executing %s",exec_nas_program);
		}
#endif
		return 0;
	}
	if (status == PW_STATUS_ACCOUNTING_OFF && nas_address) {
		log(L_INFO, "NAS %s rebooted (Accounting-Off packet seen)",
			nas_name(nas_address));
		radzap(nas_address, -1, NULL, ut.time);
#ifdef XTRADIUS
		if (exec_nas_program != NULL)
		{
			exec_nas_program = set_session_id(exec_nas_program, authreq->id);
			exec_nas_program = radius_xlate(exec_nas_program,authreq->request, authreq->request);
			radius_exec_program(exec_nas_program,authreq->request,&authreq->request,0,NULL);
			log(L_INFO, "NAS Off executing %s",exec_nas_program);
		}
#endif
		return 0;
	}

	/*
	 *	If we don't know this type of entry pretend we succeeded.
	 */
	if (status != PW_STATUS_START &&
	    status != PW_STATUS_STOP &&
	    status != PW_STATUS_ALIVE) {
		log(L_ERR, "NAS %s port %d unknown packet type %d)",
			nas_name(nas_address), ut.nas_port, status);
		return 0;
	}

	/*
	 *	Perhaps we don't want to store this record into
	 *	radutmp/radwtmp. We skip records:
	 *
	 *	- without a NAS-Port-Id (telnet / tcp access)
	 *	- with the username "!root" (console admin login)
	 *	- with Port-Type = Sync (leased line up/down)
	 */
	if (!port_seen ||
	    strncmp(ut.login, "!root", RUT_NAMESIZE) == 0
#if 0 /* I HATE Ascend - they label ISDN as sync */
	    || nas_port_type == PW_NAS_PORT_SYNC
#endif
	   )
		return 0;

	/*
	 *	Enter into the radutmp file.
	 */
	if ((fd = open(RADUTMP, O_RDWR|O_CREAT, 0644)) >= 0) {
		NAS_PORT *cache;
		int r;

		/*
		 *	Lock the utmp file, prefer lockf() over flock().
		 */
#if defined(F_LOCK) && !defined(BSD)
		(void)lockf(fd, F_LOCK, LOCK_LEN);
#else
		(void)flock(fd, LOCK_EX);
#endif
	 	/*
		 *	Find the entry for this NAS / portno combination.
		 */
		if ((cache = nas_port_find(ut.nas_address, ut.nas_port)) != NULL)
			lseek(fd, (off_t)cache->offset, SEEK_SET);

		r = 0;
		off = 0;
		while (read(fd, &u, sizeof(u)) == sizeof(u)) {
			off += sizeof(u);
			if (u.nas_address != ut.nas_address ||
			    u.nas_port    != ut.nas_port)
				continue;

			if (status == PW_STATUS_STOP &&
			    strncmp(ut.session_id, u.session_id, 
			     sizeof(u.session_id)) != 0) {
				/*
				 *	Don't complain if this is not a
				 *	login record (some clients can
				 *	send _only_ logout records).
				 */
				if (u.type == P_LOGIN)
					log(L_ERR,
		"Accounting: logout: entry for NAS %s port %d has wrong ID",
					nas_name(nas_address), u.nas_port);
				r = -1;
				break;
			}

			if (status == PW_STATUS_START &&
			    strncmp(ut.session_id, u.session_id, 
			     sizeof(u.session_id)) == 0  &&
			    u.time >= ut.time) {
				if (u.type == P_LOGIN) {
					log(L_INFO,
		"Accounting: login: entry for NAS %s port %d duplicate",
					nas_name(nas_address), u.nas_port);
					r = -1;
					dowtmp = 0;
					break;
				}
				log(L_ERR,
		"Accounting: login: entry for NAS %s port %d wrong order",
				nas_name(nas_address), u.nas_port);
				r = -1;
				break;
			}

			/*
			 *	FIXME: the ALIVE record could need
			 *	some more checking, but anyway I'd
			 *	rather rewrite this mess -- miquels.
			 */
			if (status == PW_STATUS_ALIVE &&
			    strncmp(ut.session_id, u.session_id, 
			     sizeof(u.session_id)) == 0  &&
			    u.type == P_LOGIN) {
				/*
				 *	Keep the original login time.
				 */
				ut.time = u.time;
				if (u.login[0] != 0)
					just_an_update = 1;
			}

			if (lseek(fd, -(off_t)sizeof(u), SEEK_CUR) < 0) {
				log(L_ERR, "Accounting: negative lseek!\n");
				lseek(fd, (off_t)0, SEEK_SET);
				off = 0;
			} else
				off -= sizeof(u);
			r = 1;
			break;
		}

		if (r >= 0 &&  (status == PW_STATUS_START ||
				status == PW_STATUS_ALIVE)) {
			if (cache == NULL) {
			   if ((cache = malloc(sizeof(NAS_PORT))) != NULL) {
				   cache->nasaddr = ut.nas_address;
				   cache->port = ut.nas_port;
				   cache->offset = off;
				   cache->next = nas_port_list;
				   nas_port_list = cache;
			   }
			}
			ut.type = P_LOGIN;
			write(fd, &ut, sizeof(u));
		}
		if (status == PW_STATUS_STOP) {
			if (r > 0) {
				u.type = P_IDLE;
				u.time = ut.time;
				u.delay = ut.delay;
				write(fd, &u, sizeof(u));
			} else if (r == 0) {
				log(L_ERR,
		"Accounting: logout: login entry for NAS %s port %d not found",
				nas_name(nas_address), ut.nas_port);
				r = -1;
			}
		}
		close(fd);
	} else {
		log(L_ERR, "Accounting: %s: %s", RADUTMP, strerror(errno));
		ret = -1;
	}

	/*
	 *	Don't write wtmp if we don't have a username, or
	 *	if this is an update record and the original record
	 *	was already written.
	 */
	if ((status != PW_STATUS_STOP && wt.ut_name[0] == 0) || just_an_update)
		dowtmp = 0;

	/*
	 *	Write a RADIUS wtmp log file.
	 */
	if (dowtmp && (fp = fopen(RADWTMP, "a")) != NULL) {
		ret = 0;
		fwrite(&wt, sizeof(wt), 1, fp);
		fclose(fp);
	}

#ifdef XTRADIUS
	if (exec_account)
	{
		exec_account_program = set_session_id(exec_account_program, authreq->id);
		exec_account_program = radius_xlate(exec_account_program,authreq->request, authreq->request);
		radius_exec_program(exec_account_program,authreq->request,&authreq->request,0,NULL);
	}
#endif

	return ret;
}



int rad_accounting_orig(AUTH_REQ *authreq, int authtype, char *f)
{
	FILE		*outfd;
	char		nasname[128];
	char		buffer[512];
	char		*s;
	VALUE_PAIR	*pair;
	UINT4		nas;
	NAS		*cl;
#ifdef __FreeBSD__
	time_t		curtime;
#else
	long		curtime;
#endif
	int		ret = 0;
	struct stat	st;
#ifdef XTRADIUS
	char 	*exec_log = NULL;
#endif

	/*
	 *	See if we have an accounting directory. If not,
	 *	return.
	 */
	if (stat(radacct_dir, &st) < 0)
		return 0;
	curtime = time(0);

	/*
	 *	Find out the name of this terminal server. We try
	 *	to find the PW_NAS_IP_ADDRESS in the naslist file.
	 *	If that fails, we look for the originating address.
	 *	Only if that fails we resort to a name lookup.
	 */
	cl = NULL;
	nas = authreq->ipaddr;
	if ((pair = pairfind(authreq->request, PW_NAS_IP_ADDRESS)) != NULL)
		nas = pair->lvalue;
	if (authreq->server_ipaddr)
		nas = authreq->server_ipaddr;

	if ((cl = nas_find(nas)) != NULL) {
		/* shortname and longname are < 128 bytes */
		if (cl->shortname[0])
			strcpy(nasname, cl->shortname);
		else
			strcpy(nasname, cl->longname);
	}

	if (cl == NULL) {
		s = ip_hostname(nas);
		if (strlen(s) >= sizeof(nasname) || strchr(s, '/'))
			return -1;
		strcpy(nasname, s);
	}

	/*
	 *	Create a directory for this nas.
	 */
	sprintf(buffer, "%.250s/%.250s", radacct_dir, nasname);
	(void) mkdir(buffer, 0755);

	/*
	 *	Write Detail file.
	 */
	sprintf(buffer, "%.200s/%.200s/%.100s", radacct_dir, nasname, f ? f : "detail");
	if((outfd = fopen(buffer, "a")) == (FILE *)NULL) {
		log(L_ERR, "Acct: Couldn't open file %s", buffer);
		ret = -1;
	} else {

		/* Post a timestamp */
		fputs(ctime(&curtime), outfd);

#ifdef XTRADIUS
		fclose(outfd);
		if (exec_log != NULL)
		{
			exec_log = set_detail_log(exec_log,buffer);
			exec_log = set_session_id(exec_log, authreq->id);
			exec_log = radius_xlate(exec_log,authreq->request,authreq->request);
			radius_exec_program(exec_log,authreq->request,&authreq->request,1,NULL);
		}
		outfd = fopen(buffer, "a");
#endif

		/* Write each attribute/value to the log file */
		pair = authreq->request;
		while(pair != (VALUE_PAIR *)NULL) {
			if (pair->attribute != PW_PASSWORD) {
				fputs("\t", outfd);
				fprint_attr_val(outfd, pair);
				fputs("\n", outfd);
			}
			pair = pair->next;
		}

		/*
		 *	Add non-protocol attibutes.
		 */

#ifdef __FreeBSD__
		fprintf(outfd, "\tTimestamp = %d\n", curtime);
#else
		fprintf(outfd, "\tTimestamp = %ld\n", curtime);
#endif
		switch(authtype) {
		    case 0:
			fputs("\tRequest-Authenticator = Verified\n", outfd);
			break;
		    case 1:
			fputs("\tRequest-Authenticator = None\n", outfd);
			break;
		    case 2:
			fputs("\tRequest-Authenticator = Unverified\n", outfd);
			break;
		    default:
			break;
		}
		fputs("\n", outfd);
		fclose(outfd);
	}

	return ret;
}


/*
 *	rad_accounting: call both the old and new style accounting functions.
 */
int rad_accounting(AUTH_REQ *authreq, int activefd)
{
	int reply = 0;
	int auth;
	char pw_digest[16];

	/*
	 *	See if we know this client, then check the
	 *	request authenticator.
	 */
	auth = calc_acctdigest(pw_digest, authreq);

	if (auth < 0) {
		authfree(authreq);
		return -1;
	}

	if (log_stripped_names) {
		/*
		 *	rad_accounting_new strips authreq for us. If
		 *	we run it first, the stripped info will also
		 *	get into the "details" file.
		 */
		if (rad_accounting_new(authreq, doradwtmp) == 0)
			reply = 1;
		if (rad_accounting_orig(authreq, auth, NULL) == 0)
			reply = 1;
	} else {
		/*
		 *	First log into the details file, before the
		 *	username gets stripped by rad_accounting_new.
		 */
		if (rad_accounting_orig(authreq, auth, NULL) == 0)
			reply = 1;
		if (rad_accounting_new(authreq, doradwtmp) == 0)
			reply = 1;
	}

	if (reply) {
		/*
		 *	Now send back an ACK to the NAS.
		 */
		rad_send_reply(PW_ACCOUNTING_RESPONSE,
			authreq, NULL, NULL, activefd);
	}

	authfree(authreq);

	return reply ? 0 : -1;
}


/*
 *	Timeout handler (10 secs)
 */
static int got_alrm;
static void alrm_handler()
{
	got_alrm = 1;
}

/*
 *	Check one terminal server to see if a user is logged in.
 */
static int rad_check_ts(struct radutmp *ut)
{
	int	pid, st, e;
	int	n;
	NAS	*nas;
	char	address[16];
	char	port[8];
	char	session_id[12];
	char	*s;
	void	(*handler)(int);

	/*
	 *	Find NAS type.
	 */
	if ((nas = nas_find(ntohl(ut->nas_address))) == NULL) {
		log(L_ERR, "Accounting: unknown NAS");
		return -1;
	}

	/*
	 *	Fork.
	 */
	handler = signal(SIGCHLD, SIG_DFL);
	if ((pid = fork()) < 0) {
		log(L_ERR, "Accounting: fork: %s", strerror(errno));
		signal(SIGCHLD, handler);
		return -1;
	}

	if (pid > 0) {
		/*
		 *	Parent - Wait for checkrad to terminate.
		 *	We timeout in 10 seconds.
		 */
		got_alrm = 0;
		signal(SIGALRM, alrm_handler);
		alarm(10);
		while((e = waitpid(pid, &st, 0)) != pid)
			if (e < 0 && (errno != EINTR || got_alrm))
				break;
		alarm(0);
		signal(SIGCHLD, handler);
		if (got_alrm) {
			kill(pid, SIGTERM);
			sleep(1);
			kill(pid, SIGKILL);
			log(L_ERR, "Check-TS: timeout waiting for checkrad");
			return 2;
		}
		if (e < 0) {
			log(L_ERR, "Check-TS: unknown error in waitpid()");
			return 2;
		}
		return WEXITSTATUS(st);
	}

	/*
	 *	Child - exec checklogin with the right parameters.
	 */
	for (n = 32; n >= 3; n--)
		close(n);

	ipaddr2str(address, ntohl(ut->nas_address));
	sprintf(port, "%d", ut->nas_port);
	sprintf(session_id, "%.8s", ut->session_id);

	s = CHECKRAD2;
	execl(CHECKRAD2, "checkrad", nas->nastype, address, port,
		ut->login, session_id, NULL);
	if (errno == ENOENT) {
		s = CHECKRAD1;
		execl(CHECKRAD1, "checklogin", nas->nastype, address, port,
			ut->login, session_id, NULL);
	}
	log(L_ERR, "Check-TS: exec %s: %s", s, strerror(errno));

	/*
	 *	Exit - 2 means "some error occured".
	 */
	exit(2);
}

/*
 *	See if a user is already logged in.
 *
 *	Check twice. If on the first pass the user exceeds his
 *	max. number of logins, do a second pass and validate all
 *	logins by querying the terminal server (using eg. telnet).
 *
 *	Returns: 0 == OK, 1 == double logins, 2 == multilink attempt
 */
int rad_check_multi(char *name, VALUE_PAIR *request, int maxsimul)
{
	FILE		*wfp;
	int		fd;
	int		count;
	struct radutmp	u;
	struct utmp	wt;
	VALUE_PAIR	*fra;
	int		mpp = 1;
	UINT4		ipno = 0;

	if ((fd = open(RADUTMP, O_CREAT|O_RDWR, 0644)) < 0)
		return 0;

	/*
	 *	We don't lock in the first pass.
	 */
	count = 0;
	while(read(fd, &u, sizeof(u)) == sizeof(u))
		if (strncmp(name, u.login, RUT_NAMESIZE) == 0
		    && u.type == P_LOGIN)
			count++;

	if (count < maxsimul) {
		close(fd);
		return 0;
	}
	lseek(fd, (off_t)0, SEEK_SET);

	/*
	 *	Setup some stuff, like for MPP detection.
	 */
	if ((fra = pairfind(request, PW_FRAMED_IP_ADDRESS)) != NULL)
		ipno = htonl(fra->lvalue);

	/*
	 *	lockf() the file while reading/writing.
	 */
#if defined(F_LOCK) && !defined(BSD)
		(void)lockf(fd, F_LOCK, LOCK_LEN);
#else
		(void)flock(fd, LOCK_EX);
#endif

	/*
	 *	Allright, there are too many concurrent logins.
	 *	Check all registered logins by querying the
	 *	terminal server directly.
	 */
	count = 0;
	while (read(fd, &u, sizeof(u)) == sizeof(u)) {
		if (strncmp(name, u.login, RUT_NAMESIZE) == 0
		    && u.type == P_LOGIN) {
			if (rad_check_ts(&u) == 1) {
				count++;
				/*
				 *	Does it look like a MPP attempt?
				 */
				if (strchr("SCPA", u.proto) &&
				    ipno && u.framed_address == ipno)
					mpp = 2;
			}
			else {
				/*
				 *	False record - zap it.
				 */

				lseek(fd, -(off_t)sizeof(u), SEEK_CUR);
				u.type = P_IDLE;
				write(fd, &u, sizeof(u));

				if ((wfp = fopen(RADWTMP, "a")) != NULL) {
					make_wtmp(&u, &wt, PW_STATUS_STOP);
					fwrite(&wt, sizeof(wt), 1, wfp);
					fclose(wfp);
				}
			}
		}
	}
	close(fd);

	return (count < maxsimul) ? 0 : mpp;
}

