/*
 * Copyright (c) 1997-1999  The Stanford SRP Authentication Project
 * All Rights Reserved.
 *
 * Permission is hereby granted, free of charge, to any person obtaining
 * a copy of this software and associated documentation files (the
 * "Software"), to deal in the Software without restriction, including
 * without limitation the rights to use, copy, modify, merge, publish,
 * distribute, sublicense, and/or sell copies of the Software, and to
 * permit persons to whom the Software is furnished to do so, subject to
 * the following conditions:
 *
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS-IS" AND WITHOUT WARRANTY OF ANY KIND, 
 * EXPRESS, IMPLIED OR OTHERWISE, INCLUDING WITHOUT LIMITATION, ANY 
 * WARRANTY OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE.  
 *
 * IN NO EVENT SHALL STANFORD BE LIABLE FOR ANY SPECIAL, INCIDENTAL,
 * INDIRECT OR CONSEQUENTIAL DAMAGES OF ANY KIND, OR ANY DAMAGES WHATSOEVER
 * RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER OR NOT ADVISED OF
 * THE POSSIBILITY OF DAMAGE, AND ON ANY THEORY OF LIABILITY, ARISING OUT
 * OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 * In addition, the following conditions apply:
 *
 * 1. Any software that incorporates the SRP authentication technology
 *    must display the following acknowlegment:
 *    "This product uses the 'Secure Remote Password' cryptographic
 *     authentication system developed by Tom Wu (tjw@CS.Stanford.EDU)."
 *
 * 2. Any software that incorporates all or part of the SRP distribution
 *    itself must also display the following acknowledgment:
 *    "This product includes software developed by Tom Wu and Eugene
 *     Jhong for the SRP Distribution (http://srp.stanford.edu/srp/)."
 *
 * 3. Redistributions in source or binary form must retain an intact copy
 *    of this copyright notice and list of conditions.
 */

#include <stdio.h>
#include <sys/types.h>

#include "config.h"

#ifdef OPENSSL
# include "openssl/bn.h"
typedef BIGNUM * BigInteger;
#elif defined(CRYPTOLIB)
# include "libcrypt.h"
typedef BigInt BigInteger;
#elif defined(GNU_MP)
# include "gmp.h"
typedef MP_INT * BigInteger;
#else
no math library specified!
#endif
#define MATH_PRIV

#include "t_defines.h"
#include "t_pwd.h"

/* Math library interface stubs */

BigInteger
BigIntegerFromInt(n)
     unsigned int n;
{
#ifdef OPENSSL
  BIGNUM * a = BN_new();
  BN_set_word(a, n);
  return a;
#elif defined(CRYPTOLIB)
  return bigInit(n);
#elif defined(GNU_MP)
  BigInteger rv;

  rv = (BigInteger) malloc(sizeof(MP_INT));
  mpz_init_set_ui(rv, n);
  return rv;
#endif
}

BigInteger
BigIntegerFromBytes(bytes, length)
     const unsigned char * bytes;
     int length;
{
#ifdef OPENSSL
  BIGNUM * a = BN_new();
  BN_bin2bn(bytes, length, a);
  return a;
#elif defined(CRYPTOLIB)
  BigInteger rv, t;
  int i, n;

  rv = bigInit(0);
  if(length % 4 == 0)
    RSA_bufToBig(bytes, length, rv);
  else {	/* Wouldn't need this if cryptolib behaved better */
    i = length & 0x3;
    if(length > i)
      RSA_bufToBig(bytes + i, length - i, rv);
    for(n = 0; i > 0; --i)
      n = (n << 8) | *bytes++;
    t = bigInit(n);
    bigLeftShift(t, (length & ~0x3) << 3, t);
    bigAdd(rv, t, rv);
    freeBignum(t);
  }
  return rv;
#elif defined(GNU_MP)
  BigInteger rv;
  char hexbuf[MAXHEXPARAMLEN];
  char *p;

  rv = (BigInteger) malloc(sizeof(MP_INT));
  mpz_init_set_str(rv, t_tohex(hexbuf, bytes, length), 16);
  for(p = hexbuf; *p; ++p)
    *p = '\0';
  return rv;
#endif
}

int
BigIntegerToBytes(src, dest)
     BigInteger src;
     unsigned char * dest;
{
#ifdef OPENSSL
  return BN_bn2bin(src, dest);
#elif defined(CRYPTOLIB)
  int i;

  trim(src);
  i = bigBytes(src);
  RSA_bigToBuf(src, i, dest);
  return i;
#elif defined(GNU_MP)
  char hexbuf[MAXHEXPARAMLEN];
  char * p;
  int r;

  mpz_get_str(hexbuf, 16, src);
  r = t_fromhex(dest, hexbuf);
  for(p = hexbuf; *p; ++p)
    *p = '\0';
  return r;
#endif
}

void
BigIntegerToHex(src, dest)
     BigInteger src;
     char * dest;
{
#ifdef OPENSSL
  strcpy(dest, BN_bn2hex(src));
#elif defined(CRYPTOLIB)
  trim(src);
  bigsprint(src, dest);
#elif defined(GNU_MP)
  mpz_get_str(dest, 16, src);
#endif
}

static char b64table[] =
  "0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz./";

void
BigIntegerToString(src, dest, radix)
     BigInteger src;
     char * dest;
     unsigned int radix;
{
  BigInteger t = BigIntegerFromInt(0);
  char * p = dest;
  char c;

  *p++ = b64table[BigIntegerModInt(src, radix)];
  BigIntegerDivInt(t, src, radix);
  while(BigIntegerCmpInt(t, 0) > 0) {
    *p++ = b64table[BigIntegerModInt(t, radix)];
    BigIntegerDivInt(t, t, radix);
  }
  BigIntegerFree(t);
  *p-- = '\0';
  /* reverse the string */
  while(p > dest) {
    c = *p;
    *p-- =  *dest;
    *dest++ = c;
  }
}

int
BigIntegerBitLen(b)
     BigInteger b;
{
#ifdef OPENSSL
  return BN_num_bits(b);
#elif defined(CRYPTOLIB)
  return bigBits(b);
#elif defined(GNU_MP)
  return mpz_sizeinbase(b, 2);
#endif
}

int
BigIntegerCmp(c1, c2)
     BigInteger c1, c2;
{
#ifdef OPENSSL
  return BN_cmp(c1, c2);
#elif defined(CRYPTOLIB)
  return bigCompare(c1, c2);
#elif defined(GNU_MP)
  return mpz_cmp(c1, c2);
#endif
}

int
BigIntegerCmpInt(c1, c2)
     BigInteger c1;
     unsigned int c2;
{
#ifdef OPENSSL
  BIGNUM * a = BN_new();
  int rv;
  BN_set_word(a, c2);
  rv = BN_cmp(c1, a);
  BN_free(a);
  return rv;
#elif defined(CRYPTOLIB)
  BigInteger t;
  int rv;

  t = bigInit(c2);
  rv = bigCompare(c1, t);
  freeBignum(t);
  return rv;
#elif defined(GNU_MP)
  return mpz_cmp_ui(c1, c2);
#endif
}

void
BigIntegerLShift(result, x, bits)
     BigInteger result, x;
     unsigned int bits;
{
#ifdef OPENSSL
  BN_lshift(result, x, bits);
#elif defined(CRYPTOLIB)
  bigLeftShift(x, bits, result);
#elif defined(GNU_MP)
  mpz_mul_2exp(result, x, bits);
#endif
}

void
BigIntegerAdd(result, a1, a2)
     BigInteger result, a1, a2;
{
#ifdef OPENSSL
  BN_add(result, a1, a2);
#elif defined(CRYPTOLIB)
  bigAdd(a1, a2, result);
#elif defined(GNU_MP)
  mpz_add(result, a1, a2);
#endif
}

void
BigIntegerAddInt(result, a1, a2)
     BigInteger result, a1;
     unsigned int a2;
{
#ifdef OPENSSL
  BIGNUM * a = BN_new();
  BN_set_word(a, a2);
  BN_add(result, a1, a);
  BN_free(a);
#elif defined(CRYPTOLIB)
  BigInteger t;

  t = bigInit(a2);
  bigAdd(a1, t, result);
  freeBignum(t);
#elif defined(GNU_MP)
  mpz_add_ui(result, a1, a2);
#endif
}

void
BigIntegerSub(result, s1, s2)
     BigInteger result, s1, s2;
{
#ifdef OPENSSL
  BN_sub(result, s1, s2);
#elif defined(CRYPTOLIB)
  bigSubtract(s1, s2, result);
#elif defined(GNU_MP)
  mpz_sub(result, s1, s2);
#endif
}

void
BigIntegerSubInt(result, s1, s2)
     BigInteger result, s1;
     unsigned int s2;
{
#ifdef OPENSSL
  BIGNUM * s = BN_new();
  BN_set_word(s, s2);
  BN_sub(result, s1, s);
  BN_free(s);
#elif defined(CRYPTOLIB)
  BigInteger t;

  t = bigInit(s2);
  bigSubtract(s1, t, result);
  freeBignum(t);
#elif defined(GNU_MP)
  mpz_sub_ui(result, s1, s2);
#endif
}

void
BigIntegerMul(result, m1, m2)
     BigInteger result, m1, m2;
{
#ifdef OPENSSL
  BN_CTX * ctx = BN_CTX_new();
  BN_mul(result, m1, m2, ctx);
  BN_CTX_free(ctx);
#elif defined(CRYPTOLIB)
  bigMultiply(m1, m2, result);
#elif defined(GNU_MP)
  mpz_mul(result, m1, m2);
#endif
}

void
BigIntegerMulInt(result, m1, m2)
     BigInteger result, m1;
     unsigned int m2;
{
#ifdef OPENSSL
  BN_CTX * ctx = BN_CTX_new();
  BIGNUM * m = BN_new();
  BN_set_word(m, m2);
  BN_mul(result, m1, m, ctx);
  BN_CTX_free(ctx);
#elif defined(CRYPTOLIB)
  BigInteger t;

  t = bigInit(m2);
  bigMultiply(m1, t, result);
  freeBignum(t);
#elif defined(GNU_MP)
  mpz_mul_ui(result, m1, m2);
#endif
}

void
BigIntegerDivInt(result, d, m)
     BigInteger result, d;
     unsigned int m;
{
#ifdef OPENSSL
  BN_CTX * ctx = BN_CTX_new();
  BIGNUM * a = BN_new();
  BN_set_word(a, m);
  BN_div(result, NULL, d, a, ctx);
  BN_CTX_free(ctx);
#elif defined(CRYPTOLIB)
  BigInteger t, u, q;

  t = bigInit(m);
  u = bigInit(0);
  /* We use a separate variable q because cryptolib breaks if result == d */
  q = bigInit(0);
  bigDivide(d, t, q, u);
  freeBignum(t);
  freeBignum(u);
  bigCopy(q, result);
  freeBignum(q);
#elif defined(GNU_MP)
# ifdef GMP2
  mpz_fdiv_q_ui(result, d, m);
# else
  mpz_div_ui(result, d, m);
# endif
#endif
}

void
BigIntegerMod(result, d, m)
     BigInteger result, d, m;
{
#ifdef OPENSSL
  BN_CTX * ctx = BN_CTX_new();
  BN_mod(result, d, m, ctx);
  BN_CTX_free(ctx);
#elif defined(CRYPTOLIB)
  bigMod(d, m, result);
#elif defined(GNU_MP)
  mpz_mod(result, d, m);
#endif
}

unsigned int
BigIntegerModInt(d, m)
     BigInteger d;
     unsigned int m;
{
#ifdef OPENSSL
  return BN_mod_word(d, m);
#elif defined(CRYPTOLIB)
  BigInteger t, u;
  unsigned char r[4];

  t = bigInit(m);
  u = bigInit(0);
  bigMod(d, t, u);
  bigToBuf(u, sizeof(r), r);
  freeBignum(t);
  freeBignum(u);
  return r[0] | (r[1] << 8) | (r[2] << 16) | (r[3] << 24);
#elif defined(GNU_MP)
  MP_INT result;
  unsigned int i;

  mpz_init(&result);

/* Define GMP2 if you're using an old gmp.h but want to link against a
 * newer libgmp.a (e.g. 2.0 or later). */

# ifdef GMP2
  mpz_fdiv_r_ui(&result, d, m);
# else
  mpz_mod_ui(&result, d, m);
# endif
  i = mpz_get_ui(&result);
  mpz_clear(&result);
  return i;
#endif
}

void
BigIntegerModMul(r, m1, m2, modulus)
     BigInteger r, m1, m2, modulus;
{
#ifdef OPENSSL
  BN_CTX * ctx = BN_CTX_new();
  BN_mod_mul(r, m1, m2, modulus, ctx);
  BN_CTX_free(ctx);
#elif defined(CRYPTOLIB)
  bigMultiply(m1, m2, r);
  bigMod(r, modulus, r);
#elif defined(GNU_MP)
  mpz_mul(r, m1, m2);
  mpz_mod(r, r, modulus);
#endif
}

void
BigIntegerModExp(r, b, e, m)
     BigInteger r, b, e, m;
{
#ifdef OPENSSL
  BN_CTX * ctx = BN_CTX_new();
  BN_mod_exp(r, b, e, m, ctx);
  BN_CTX_free(ctx);
#elif defined(CRYPTOLIB)
  bigPow(b, e, m, r);
#elif defined(GNU_MP)
  mpz_powm(r, b, e, m);
#endif
}

void
BigIntegerModExpInt(r, b, e, m)
     BigInteger r, b;
     unsigned int e;
     BigInteger m;
{
#ifdef OPENSSL
  BN_CTX * ctx = BN_CTX_new();
  BIGNUM * p = BN_new();
  BN_set_word(p, e);
  BN_mod_exp(r, b, p, m, ctx);
  BN_free(p);
  BN_CTX_free(ctx);
#elif defined(CRYPTOLIB)
  BigInteger t;

  t = bigInit(e);
  bigPow(b, t, m, r);
  freeBignum(t);
#elif defined(GNU_MP)
  mpz_powm_ui(r, b, e, m);
#endif
}

int
BigIntegerCheckPrime(n)
     BigInteger n;
{
#ifdef OPENSSL
  BN_CTX * ctx = BN_CTX_new();
  int rv = BN_is_prime(n, 25, NULL, ctx, NULL);
  BN_CTX_free(ctx);
  return rv;
#elif defined(CRYPTOLIB)
#if 0
  /*
   * Ugh.  Not only is cryptolib's bigDivide sensitive to inputs
   * and outputs being the same, but now the primeTest needs random
   * numbers, which it gets by calling cryptolib's broken truerand
   * implementation(!)  We have to fake it out by doing our own
   * seeding explicitly.
   */
  static int seeded = 0;
  static unsigned char seedbuf[64];
  if(!seeded) {
    t_random(seedbuf, sizeof(seedbuf));
    seedDesRandom(seedbuf, sizeof(seedbuf));
    memset(seedbuf, 0, sizeof(seedbuf));
    seeded = 1;
  }
#endif /* 0 */
  t_random(NULL, 0);
  return primeTest(n);
#elif defined(GNU_MP)
  return mpz_probab_prime_p(n, 25);
#endif
}

void
BigIntegerFree(b)
     BigInteger b;
{
#ifdef OPENSSL
  BN_free(b);
#elif defined(CRYPTOLIB)
  freeBignum(b);
#elif defined(GNU_MP)
  mpz_clear(b);
  free(b);
#endif
}
