/*
 * Copyright (c) 1997-2002  The Stanford SRP Authentication Project
 * All Rights Reserved.
 *
 * Permission is hereby granted, free of charge, to any person obtaining
 * a copy of this software and associated documentation files (the
 * "Software"), to deal in the Software without restriction, including
 * without limitation the rights to use, copy, modify, merge, publish,
 * distribute, sublicense, and/or sell copies of the Software, and to
 * permit persons to whom the Software is furnished to do so, subject to
 * the following conditions:
 *
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS-IS" AND WITHOUT WARRANTY OF ANY KIND, 
 * EXPRESS, IMPLIED OR OTHERWISE, INCLUDING WITHOUT LIMITATION, ANY 
 * WARRANTY OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE.  
 *
 * IN NO EVENT SHALL STANFORD BE LIABLE FOR ANY SPECIAL, INCIDENTAL,
 * INDIRECT OR CONSEQUENTIAL DAMAGES OF ANY KIND, OR ANY DAMAGES WHATSOEVER
 * RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER OR NOT ADVISED OF
 * THE POSSIBILITY OF DAMAGE, AND ON ANY THEORY OF LIABILITY, ARISING OUT
 * OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 * In addition, the following conditions apply:
 *
 * 1. Any software that incorporates the SRP authentication technology
 *    is requested to display the following acknowlegment:
 *    "This product uses the 'Secure Remote Password' cryptographic
 *     authentication system developed by Tom Wu (tjw@CS.Stanford.EDU)."
 *
 * 2. Any software that incorporates all or part of the SRP distribution
 *    itself must display the following acknowledgment:
 *    "This product includes software developed by Tom Wu and Eugene
 *     Jhong for the SRP Distribution (http://srp.stanford.edu/)."
 *
 * 3. Redistributions in source or binary form must retain an intact copy
 *    of this copyright notice and list of conditions.
 */
#ifndef _SRP_H_
#define _SRP_H_

#include "t_defines.h"
#include "cstr.h"

#if     !defined(P)
#ifdef  __STDC__
#define P(x)    x
#else
#define P(x)    ()
#endif
#endif

/*	For building dynamic link libraries under windows, windows NT 
 *	using MSVC1.5 or MSVC2.0
 */

#ifndef _DLLDECL
#define _DLLDECL

#ifdef MSVC15	/* MSVC1.5 support for 16 bit apps */
#define _MSVC15EXPORT _export
#define _MSVC20EXPORT
#define _DLLAPI _export _pascal
#define _TYPE(a) a _MSVC15EXPORT
#define DLLEXPORT 1

#elif defined(MSVC20) || (defined(_USRDLL) && defined(SRP_EXPORTS))
#define _MSVC15EXPORT
#define _MSVC20EXPORT _declspec(dllexport)
#define _DLLAPI
#define _TYPE(a) _MSVC20EXPORT a
#define DLLEXPORT 1

#else			/* Default, non-dll.  Use this for Unix or DOS */
#define _MSVC15DEXPORT
#define _MSVC20EXPORT
#define _DLLAPI
#define _TYPE(a) a
#endif
#endif /* _DLLDECL */

/* Enforced by params */
#define SRP_MIN_BITS 512

/* The number of bits in the secret parameter, as a function of param size */
#define SRP_SECRET_BITS(mlen) 256
#define SRP_SECRET_MAXBYTES    32	/* max based on secret_bits */

/* Returned codes for SRP API functions */
#define SRP_SUCCESS 0
#define SRP_ERROR -1

typedef struct srp_st SRP;

/* SRP method definitions */
typedef struct srp_meth_st {
  const char * name;

  int (* init)(SRP * srp);
  int (* finish)(SRP * srp);

  int (* params)(SRP * srp, const unsigned char * modulus, int modlen,
		 const unsigned char * generator, int genlen,
		 const unsigned char * salt, int saltlen);
  int (* auth)(SRP * srp, const unsigned char * a, int alen);
  int (* passwd)(SRP * srp, const unsigned char * pass, int passlen);
  int (* genpub)(SRP * srp, cstr ** result);
  int (* key)(SRP * srp, cstr ** result,
	      const unsigned char * pubkey, int pubkeylen);
  int (* verify)(SRP * srp, const unsigned char * proof, int prooflen);
  int (* respond)(SRP * srp, cstr ** proof);

  void * data;
} SRP_METHOD;

/* Magic numbers for the SRP context header */
#define SRP_MAGIC_CLIENT 12
#define SRP_MAGIC_SERVER 28

/*
 * A hybrid structure that represents either client or server state.
 */
struct srp_st {
  int magic;	/* To distinguish client from server (and for sanity) */

  int flags;

  cstr * username;

  BigInteger modulus;
  BigInteger generator;
  cstr * salt;

  BigInteger verifier;
  BigInteger password;

  BigInteger pubkey;
  BigInteger secret;
  BigInteger u;

  BigInteger key;

  cstr * ex_data;

  SRP_METHOD * meth;
  void * meth_data;
};

/*
 * SRP_new() creates a new SRP context object -
 * the method determines which "sense" (client or server)
 * the object operates in.  SRP_free() frees it.
 * (See RFC2945 method definitions below.)
 */
_TYPE( SRP * ) SRP_new P((SRP_METHOD * meth));
_TYPE( int )   SRP_free P((SRP * srp));

/*
 * Both client and server must call both SRP_set_username and
 * SRP_set_parameters, in that order, before calling anything else.
 * SRP_set_user_raw is an alternative to SRP_set_username that
 * accepts an arbitrary length-bounded octet string as input.
 */
_TYPE( int ) SRP_set_username P((SRP * srp, const char * username));
_TYPE( int ) SRP_set_user_raw P((SRP * srp,
				 const unsigned char * user, int userlen));
_TYPE( int ) SRP_set_params P((SRP * srp,
			       const unsigned char * modulus, int modlen,
			       const unsigned char * generator, int genlen,
			       const unsigned char * salt, int saltlen));

/*
 * On the client, SRP_set_authenticator, SRP_gen_exp, and
 * SRP_add_ex_data can be called in any order.
 * On the server, SRP_set_authenticator must come first,
 * followed by the other two in either order.
 */
/*
 * The authenticator is the secret possessed by either side.
 * For the server, this is the bigendian verifier, as an octet string.
 * For the client, this is the bigendian raw secret, as an octet string.
 * The server's authenticator must be the generator raised to the power
 * of the client's raw secret modulo the common modulus.
 *
 * SRP_set_auth_password computes the authenticator from a plaintext
 * password and then calls SRP_set_authenticator automatically.  This is
 * usually used on the client side, while the server usually uses
 * SRP_set_authenticator (since it doesn't know the plaintext password).
 */
_TYPE( int ) SRP_set_authenticator P((SRP * srp,
				      const unsigned char * a, int alen));
_TYPE( int ) SRP_set_auth_password P((SRP * srp, const char * password));
_TYPE( int ) SRP_set_auth_password_raw P((SRP * srp,
					  const unsigned char * password,
					  int passlen));

/*
 * SRP_gen_pub generates the random exponential residue to send
 * to the other side.  The server must withhold its result until
 * it receives the client's number.
 * 
 * If "result" points to a NULL pointer, a new cstr object will be
 * created to hold the result, and "result" will point to it.
 * If "result" points to a non-NULL cstr pointer, the result will be
 * placed there.
 * If "result" itself is NULL, no result will be returned.
 */
_TYPE( int ) SRP_gen_pub P((SRP * srp, cstr ** result));
/*
 * Append the data to the extra data segment.  Authentication will
 * not succeed unless both sides add precisely the same data in
 * the same order.
 */
_TYPE( int ) SRP_add_ex_data P((SRP * srp,
				const unsigned char * data, int datalen));

/*
 * SRP_compute_key must be called after the previous three methods.
 */
_TYPE( int ) SRP_compute_key P((SRP * srp, cstr ** result,
				const unsigned char * pubkey, int pubkeylen));

/*
 * On the client, call SRP_respond first to get the response to send
 * to the server, and call SRP_verify to verify the server's response.
 * On the server, call SRP_verify first to verify the client's response,
 * and call SRP_respond ONLY if verification succeeds.
 *
 * It is an error to call SRP_respond with a NULL pointer.
 */
_TYPE( int ) SRP_verify P((SRP * srp,
			   const unsigned char * proof, int prooflen));
_TYPE( int ) SRP_respond P((SRP * srp, cstr ** response));

/* RFC2945-style SRP authentication */

#define RFC2945_KEY_LEN 40	/* length of session key (bytes) */
#define RFC2945_RESP_LEN 20	/* length of proof hashes (bytes) */

/*
 * RFC2945-style SRP authentication methods.  Use these like:
 * SRP * srp = SRP_new(SRP_RFC2945_client_method());
 */
_TYPE( SRP_METHOD * ) SRP_RFC2945_client_method P((void));
_TYPE( SRP_METHOD * ) SRP_RFC2945_server_method P((void));

/*
 * SRP-6 authentication methods.
 */
_TYPE( SRP_METHOD * ) SRP6_client_method P((void));
_TYPE( SRP_METHOD * ) SRP6_server_method P((void));

/*
 * Convenience function - SRP_server_init_user
 * Looks up the username from the system EPS configuration and calls
 * SRP_set_username, SRP_set_params, and SRP_set_authenticator to
 * initialize server state for that user.
 */
_TYPE( int ) SRP_server_init_user P((SRP * srp, const char * username));

#endif /* _SRP_H_ */
