/*
 * Copyright (c) 1997-2003  The Stanford SRP Authentication Project
 * All Rights Reserved.
 *
 * Permission is hereby granted, free of charge, to any person obtaining
 * a copy of this software and associated documentation files (the
 * "Software"), to deal in the Software without restriction, including
 * without limitation the rights to use, copy, modify, merge, publish,
 * distribute, sublicense, and/or sell copies of the Software, and to
 * permit persons to whom the Software is furnished to do so, subject to
 * the following conditions:
 *
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS-IS" AND WITHOUT WARRANTY OF ANY KIND, 
 * EXPRESS, IMPLIED OR OTHERWISE, INCLUDING WITHOUT LIMITATION, ANY 
 * WARRANTY OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE.  
 *
 * IN NO EVENT SHALL STANFORD BE LIABLE FOR ANY SPECIAL, INCIDENTAL,
 * INDIRECT OR CONSEQUENTIAL DAMAGES OF ANY KIND, OR ANY DAMAGES WHATSOEVER
 * RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER OR NOT ADVISED OF
 * THE POSSIBILITY OF DAMAGE, AND ON ANY THEORY OF LIABILITY, ARISING OUT
 * OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 * In addition, the following conditions apply:
 *
 * 1. Any software that incorporates the SRP authentication technology
 *    is requested to display the following acknowlegment:
 *    "This product uses the 'Secure Remote Password' cryptographic
 *     authentication system developed by Tom Wu (tjw@CS.Stanford.EDU)."
 *
 * 2. Any software that incorporates all or part of the SRP distribution
 *    itself must display the following acknowledgment:
 *    "This product includes software developed by Tom Wu and Eugene
 *     Jhong for the SRP Distribution (http://srp.stanford.edu/)."
 *
 * 3. Redistributions in source or binary form must retain an intact copy
 *    of this copyright notice and list of conditions.
 */

#include <stdlib.h>
#include "t_pwd.h"
#include "srp.h"

static int library_initialized = 0;

_TYPE( int )
SRP_initialize_library()
{
  if(library_initialized == 0) {
    BigIntegerInitialize();
    t_stronginitrand();
    library_initialized = 1;
  }
  return SRP_SUCCESS;
}

_TYPE( int )
SRP_finalize_library()
{
  if(library_initialized > 0) {
    library_initialized = 0;
    BigIntegerFinalize();
  }
  return SRP_SUCCESS;
}

_TYPE( SRP_SERVER_LOOKUP * )
SRP_SERVER_LOOKUP_new(SRP_SERVER_LOOKUP_METHOD * meth)
{
  SRP_SERVER_LOOKUP * slu = (SRP_SERVER_LOOKUP *) malloc(sizeof(SRP_SERVER_LOOKUP));
  if(slu == NULL)
    return NULL;

  slu->meth = meth;
  slu->data = NULL;
  if(slu->meth->init == NULL || (*slu->meth->init)(slu) == SRP_SUCCESS)
    return slu;
  free(slu);
  return NULL;
}

_TYPE( int )
SRP_SERVER_LOOKUP_free(SRP_SERVER_LOOKUP * slu)
{
  if(slu->meth->finish)
    (*slu->meth->finish)(slu);
  free(slu);
  return SRP_SUCCESS;
}

_TYPE( int )
SRP_SERVER_do_lookup(SRP_SERVER_LOOKUP * slu, SRP * srp, cstr * username)
{
  return (*slu->meth->lookup)(slu, srp, username);
}

_TYPE( SRP * )
SRP_new(SRP_METHOD * meth)
{
  SRP * srp = (SRP *) malloc(sizeof(SRP));

  if(srp == NULL)
    return NULL;

  srp->flags = 0;
  srp->username = cstr_new();
  srp->modulus = NULL;
  srp->accel = NULL;
  srp->generator = NULL;
  srp->salt = NULL;
  srp->verifier = NULL;
  srp->password = NULL;
  srp->pubkey = NULL;
  srp->secret = NULL;
  srp->u = NULL;
  srp->key = NULL;
  srp->ex_data = cstr_new();
  srp->meth = meth;
  srp->meth_data = NULL;
  srp->slu = NULL;
  if(srp->meth->init == NULL || (*srp->meth->init)(srp) == SRP_SUCCESS)
    return srp;
  free(srp);
  return NULL;
}

_TYPE( int )
SRP_free(SRP * srp)
{
  if(srp->meth->finish)
    (*srp->meth->finish)(srp);

  if(srp->username)
    cstr_clear_free(srp->username);
  if(srp->modulus)
    BigIntegerFree(srp->modulus);
  if(srp->accel)
    BigIntegerModAccelFree(srp->accel);
  if(srp->generator)
    BigIntegerFree(srp->generator);
  if(srp->salt)
    cstr_clear_free(srp->salt);
  if(srp->verifier)
    BigIntegerClearFree(srp->verifier);
  if(srp->password)
    BigIntegerClearFree(srp->password);
  if(srp->pubkey)
    BigIntegerFree(srp->pubkey);
  if(srp->secret)
    BigIntegerClearFree(srp->secret);
  if(srp->u)
    BigIntegerFree(srp->u);
  if(srp->key)
    BigIntegerClearFree(srp->key);
  if(srp->ex_data)
    cstr_clear_free(srp->ex_data);
  return SRP_SUCCESS;
}

_TYPE( int )
SRP_set_server_lookup(SRP * srp, SRP_SERVER_LOOKUP * lookup)
{
  srp->slu = lookup;
  return SRP_SUCCESS;
}

_TYPE( int )
SRP_set_username(SRP * srp, const char * username)
{
  cstr_set(srp->username, username);
  if(srp->slu)
    return SRP_SERVER_do_lookup(srp->slu, srp, srp->username);
  else
    return SRP_SUCCESS;
}

_TYPE( int )
SRP_set_user_raw(SRP * srp, const unsigned char * user, int userlen)
{
  cstr_setn(srp->username, user, userlen);
  if(srp->slu)
    return SRP_SERVER_do_lookup(srp->slu, srp, srp->username);
  else
    return SRP_SUCCESS;
}

_TYPE( int )
SRP_set_params(SRP * srp, const unsigned char * modulus, int modlen,
	       const unsigned char * generator, int genlen,
	       const unsigned char * salt, int saltlen)
{
  return (*srp->meth->params)(srp, modulus, modlen, generator, genlen,
			      salt, saltlen);
}

_TYPE( int )
SRP_set_authenticator(SRP * srp, const unsigned char * a, int alen)
{
  return (*srp->meth->auth)(srp, a, alen);
}

_TYPE( int )
SRP_set_auth_password(SRP * srp, const char * password)
{
  return (*srp->meth->passwd)(srp, (const unsigned char *)password,
			      strlen(password));
}

_TYPE( int )
SRP_set_auth_password_raw(SRP * srp,
			  const unsigned char * password, int passlen)
{
  return (*srp->meth->passwd)(srp, password, passlen);
}

_TYPE( int )
SRP_gen_pub(SRP * srp, cstr ** result)
{
  return (*srp->meth->genpub)(srp, result);
}

_TYPE( int )
SRP_add_ex_data(SRP * srp, const unsigned char * data, int datalen)
{
  cstr_appendn(srp->ex_data, data, datalen);
  return SRP_SUCCESS;
}

_TYPE( int )
SRP_compute_key(SRP * srp, cstr ** result,
		const unsigned char * pubkey, int pubkeylen)
{
  return (*srp->meth->key)(srp, result, pubkey, pubkeylen);
}

_TYPE( int )
SRP_verify(SRP * srp, const unsigned char * proof, int prooflen)
{
  return (*srp->meth->verify)(srp, proof, prooflen);
}

_TYPE( int )
SRP_respond(SRP * srp, cstr ** proof)
{
  return (*srp->meth->respond)(srp, proof);
}

#include "t_pwd.h"

/* Convenience functions - currently depend on classic EPS API */

_TYPE( int )
SRP_server_init_user(SRP * srp, const char * username)
{
  SRP_set_server_lookup(srp, SRP_SERVER_system_lookup());
  if(SRP_set_username(srp, username) != SRP_SUCCESS)
    return SRP_ERROR;
  return SRP_SUCCESS;
}

static int
slu_nop(SRP_SERVER_LOOKUP * slu)
{
  return SRP_SUCCESS;
}

static int
slu_sys_lookup(SRP_SERVER_LOOKUP * slu, SRP * srp, cstr * username)
{
  struct t_passwd * p;

  p = gettpnam(username->data);
  if(p == NULL)
    return SRP_ERROR;
  if(SRP_set_params(srp, p->tc.modulus.data, p->tc.modulus.len,
		    p->tc.generator.data, p->tc.generator.len,
		    p->tp.salt.data, p->tp.salt.len) != SRP_SUCCESS)
    return SRP_ERROR;
  if(SRP_set_authenticator(srp, p->tp.password.data, p->tp.password.len) !=
     SRP_SUCCESS)
    return SRP_ERROR;

  return SRP_SUCCESS;
}

static struct srp_s_lu_meth_st srp_s_lu_sys_meth = {
  "SRP/EPS system lookup",
  slu_nop,
  slu_nop,
  slu_sys_lookup,
  NULL
};

static struct srp_server_lu_st srp_s_sys_lookup = {
  &srp_s_lu_sys_meth,
  NULL
};

_TYPE( SRP_SERVER_LOOKUP * )
SRP_SERVER_system_lookup()
{
  return &srp_s_sys_lookup;
}

_TYPE( int )
SRP_use_engine(const char * engine)
{
  if(BigIntegerUseEngine(engine) > 0)
    return SRP_SUCCESS;
  else
    return SRP_ERROR;
}
