#include <stdio.h>
#include <errno.h>
#include <ctype.h>
#include "error.h"

char *program_name = "mc68mu";

#define PORTA  0
#define DDRA   4

#define RAMSIZE 0x1A0
#define RAMSTART 0x50
#define ROMEND 0x4000
#define VECSTART (ROMEND-16)

typedef unsigned char byte;
typedef unsigned int word;
typedef unsigned char flag;

byte memory[ROMEND];

word ra, rx, sp, pc;
enum {
  ccr_c = 2,
  ccr_z = 3,
  ccr_h = 4,
  ccr_n = 5,
  ccr_i = 6,
  ccr_irq = 7
};
flag ccr[8];
unsigned long cycle;

#define fatal(x, args...) error(4,0,x , ## args)

static byte 
mem(word addr)
{
  addr &= 0xFFFF;
  if (addr < ROMEND)
    return memory[addr];
  else
    {
      fatal("cycle %d memory read %04x out of bounds, pc=%04x", 
	    cycle, addr, pc);
      return 0;
    }
}

static void
setmem(word addr, byte x)
{
  addr &= 0xFFFF;
  x &= 0xFF;
  if (addr >= RAMSTART && addr < RAMSTART+RAMSIZE)
    memory[addr] = x;
  else if (addr == PORTA)
    printf("cycle %6ld: %02x\n", cycle, x);
  else if (addr == DDRA)
    {
      memory[addr] = x;
      if (x == 0)
	putchar('\n');
    }
  else
    fatal("cycle %d memory write %04x out of bounds, pc=%04x", 
	  cycle, addr, pc);
}

static void
setnz(word r)
{
  ccr[ccr_z] = r == 0;
  ccr[ccr_n] = (r & 0x80) != 0;
}

/* #define setpc(x) (pc = (x) & 0x07ff) */
#define setpc(x) (pc = (x))
#define incpc() (setpc(pc+1))
#define push(x) (setmem(sp,x), sp = ((sp - 1) & 0xFF) | 0xC0)
#define pop() mem(sp = ((sp + 1) & 0xFF) | 0xC0)

word
memaddr(word amode)
{
  word result;
  switch (amode >> 4 & 0xF)
    {
    case 0xA:
      abort();
    case 0xB:
      result = mem(pc);
      incpc();
      break;
    case 0xC:
      result = mem(pc) << 8 | mem(pc+1);
      setpc(pc+2);
      break;
    case 0xD:
      result = (mem(pc) << 8 | mem(pc+1)) + rx;
      setpc(pc+2);
      break;
    case 0xE:
      result = mem(pc) + rx;
      incpc();
      break;
    case 0xF:
      result = rx;
      break;
    default:
      abort();
    }
  return result;
}

byte
decodemem(word amode)
{
  if ((amode >> 4 & 0xF) == 0xA)
    {
      byte result;
      result = mem(pc);
      incpc();
      return result;
    }
  else
    return mem(memaddr(amode));
}

byte
rmw_r(word amode)
{
  switch (amode >> 4 & 0xF)
    {
    case 0x3:
      return mem(mem(pc));
    case 0x4:
      return ra;
    case 0x5:
      return rx;
    case 0x6:
      return mem(mem(pc)+rx);
    case 0x7:
      return mem(rx);
    default:
      abort();
    }
}

void
rmw_w(word amode, byte x)
{
  x = x & 0xFF;
  setnz(x);
  switch (amode >> 4 & 0xF)
    {
    case 0x3:
      setmem(mem(pc), x);
      incpc();
      return;
    case 0x4:
      ra = x;
      return;
    case 0x5:
      rx = x;
      return;
    case 0x6:
      setmem(mem(pc)+rx, x);
      incpc();
      return;
    case 0x7:
      setmem(rx, x);
      return;
    default:
      abort();
    }
}

void
do_debug(byte opcode)
{
  printf("cycle %8lu: pc=%04x ", cycle, pc);
  switch (opcode >> 4 & 0xF)
    {
    case 0x3:
      printf("mem[%02x] = %02x\n", mem(pc), mem(mem(pc)));
      incpc();
      return;
    case 0x4:
      printf("A=%02x\n", ra);
      return;
    case 0x5:
      printf("X=%02x\n", rx);
      return;
    case 0x6:
      printf("mem[%02x+%02x] = %02x\n", mem(pc), rx, mem(mem(pc)+rx));
      incpc();
      return;
    case 0x7:
      printf("mem[0+%02x] = %02x\n", rx, mem(rx));
      return;
    default:
      abort();
    }
}

void
do_add(byte inp, flag c)
{
  word tmp;
  tmp = (ra & 0xFF) + inp + c;
  ccr[ccr_h] = ((ra & 0xf) + (inp & 0xf) + c) > 0xf;
  ccr[ccr_c] = tmp > 0xff;
  ra = tmp & 0xFF;
  setnz(ra);
}

word
do_sub(byte inp, flag c)
{
  word tmp;
  tmp = (ra & 0xFF) - inp - c;
  ccr[ccr_c] = (tmp & 0x100) != 0;
  tmp &= 0xFF;
  setnz(tmp);
  return tmp;
}

static const int cycles[256] = 
{
  5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 
  5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 
  3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 
  5, 0, 0, 5, 5, 0, 5, 5, 5, 5, 5, 0, 5, 4, 0, 5, 
  3, 0,11, 3, 3, 0, 3, 3, 3, 3, 3, 0, 3, 3, 0, 3, 
  3, 0, 0, 3, 3, 0, 3, 3, 3, 3, 3, 0, 3, 3, 0, 3, 
  6, 0, 0, 6, 6, 0, 6, 6, 6, 6, 6, 0, 6, 5, 0, 6, 
  5, 0, 0, 5, 5, 0, 5, 5, 5, 5, 5, 0, 5, 4, 0, 5, 
  
  9, 6, 0,10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 2, 2,
  0, 0, 0, 0, 0, 0, 0, 2, 2, 2, 2, 2, 2, 2, 0, 2,
  2, 2, 2, 2, 2, 2, 2, 0, 2, 2, 2, 2, 0, 6, 2, 0,
  3, 3, 3, 3, 3, 3, 3, 4, 3, 3, 3, 3, 2, 5, 3, 4,
  4, 4, 4, 4, 4, 4, 4, 5, 4, 4, 4, 4, 3, 6, 4, 5,
  5, 5, 5, 5, 5, 5, 5, 6, 5, 5, 5, 5, 4, 7, 5, 6,
  4, 4, 4, 4, 4, 4, 4, 5, 4, 4, 4, 4, 3, 6, 4, 5,
  3, 3, 3, 3, 3, 3, 3, 4, 3, 3, 3, 3, 2, 5, 3, 4
};

void
oneinst(void)
{
  byte opcode;
  word tmp;
  opcode = mem(pc);
  incpc();
  cycle += cycles[opcode];
  if ((opcode >> 4 & 0xF) >= 0xA)
    switch (opcode & 0xF)
      {
      case 0x0:
	ra = do_sub(decodemem(opcode), 0);
	return;
      case 0x1:
	(void) do_sub(decodemem(opcode), 0);
	return;
      case 0x2:
	ra = do_sub(decodemem(opcode), ccr[ccr_c]);
	return;
      case 0x3:
	tmp = ra; ra = rx;
	(void) do_sub(decodemem(opcode), 0);
	ra = tmp;
	return;
      case 0x4:
	ra = ra & decodemem(opcode);
	setnz(ra);
	return;
      case 0x5:
	setnz(ra & decodemem(opcode));
	return;
      case 0x6:
	ra = decodemem(opcode);
	setnz(ra);
	return;
      case 0x7:
	setmem(memaddr(opcode), ra);
	setnz(ra);
	return;
      case 0x8:
	ra = ra ^ decodemem(opcode);
	setnz(ra);
	return;
      case 0x9:
	do_add(decodemem(opcode), ccr[ccr_c]);
	return;
      case 0xA:
	ra = ra | decodemem(opcode);
	setnz(ra);
	return;
      case 0xB:
	do_add(decodemem(opcode), 0);
	return;
      case 0xC:
	pc = memaddr(opcode);
	return;
      case 0xD:
	if (opcode == 0xAD)
	  {
	    tmp = pc + 1 + (mem(pc) ^ 0x80) - 0x80;
	    incpc();
	  }
	else
	  tmp = memaddr(opcode);
	push(pc & 0xFF);
	push(pc >> 8 & 0xFF);
	pc = tmp;
	return;
      case 0xE:
	rx = decodemem(opcode);
	setnz(rx);
	return;
      case 0xF:
	setmem(memaddr(opcode), rx);
	setnz(rx);
	return;
      }
  else if ((opcode >> 4 & 0xF) >= 0x3 && (opcode >> 4 & 0xF) < 0x8)
    switch (opcode & 0xF)
      {
      case 0x0:
	rmw_w(opcode, -rmw_r(opcode));
	return;
      case 0x3:
	rmw_w(opcode, 0xFF - rmw_r(opcode));
	ccr[ccr_c] = 1;
	return;
      case 0x4:
	tmp = rmw_r(opcode); 
	ccr[ccr_c] = tmp & 1;
	rmw_w(opcode, tmp >> 1);
	return;
      case 0x6:
	tmp = rmw_r(opcode);
	rmw_w(opcode, tmp >> 1 | ccr[ccr_c] << 7);
	ccr[ccr_c] = tmp & 1;
	return;
      case 0x7:
	tmp = rmw_r(opcode); 
	ccr[ccr_c] = tmp & 1;
	rmw_w(opcode, tmp >> 1 | (tmp & 0x80));
	return;
      case 0x8:
	tmp = rmw_r(opcode) << 1; 
	ccr[ccr_c] = (tmp & 0x100) != 0;
	rmw_w(opcode, tmp);
	return;
      case 0x9:
	tmp = rmw_r(opcode) << 1; 
	rmw_w(opcode, tmp | ccr[ccr_c]);
	ccr[ccr_c] = (tmp & 0x100) != 0;
	return;
      case 0xA:
	rmw_w(opcode, rmw_r(opcode) - 1);
	return;
      case 0xB:
	do_debug(opcode);
	return;
      case 0xC:
	rmw_w(opcode, rmw_r(opcode) + 1);
	return;
      case 0xD:
	rmw_w(opcode, rmw_r(opcode));
	return;
      case 0xF:
	rmw_w(opcode, 0);
	return;
      default:
	break;
      }
  else if ((opcode >> 4 & 0xF) == 0x2)
    {
      ccr[0] = 0;
      ccr[1] = ccr[ccr_c] | ccr[ccr_z];
      if (ccr[opcode >> 1 & 0x7] == (opcode & 1))
	setpc(pc + 1 + (mem(pc) ^ 0x80) - 0x80);
      else
	incpc();
      return;
    }
  else if ((opcode >> 4 & 0xF) <= 0x1)
    {
      byte tmp2 = 1 << (opcode >> 1 & 7);
      byte tmp3 = ~opcode << (opcode >> 1 & 7) & tmp2;
      tmp = mem(pc);
      incpc();
      if ((opcode >> 4 & 0xF) == 0x1)
	setmem(tmp, (mem(tmp) & ~tmp2) | tmp3);
      else 
	{
	  ccr[ccr_c] = mem(tmp) >> (opcode >> 1 & 7) & 1;
	  if (ccr[ccr_c] != (opcode & 1))
	    setpc(pc + 1 + (mem(pc) ^ 0x80) - 0x80);
	  else
	    incpc();
	}
      return;
    }
  switch (opcode)
    {
    case 0x42:
      tmp = rx * ra;
      ccr[ccr_c] = 0;
      ccr[ccr_h] = 0;
      rx = tmp >> 8;
      ra = tmp & 0xFF;
      return;
    case 0x80:
      tmp = pop();
      ccr[ccr_c] = tmp      & 1;
      ccr[ccr_z] = tmp >> 1 & 1;
      ccr[ccr_n] = tmp >> 2 & 1;
      ccr[ccr_i] = tmp >> 3 & 1;
      ccr[ccr_h] = tmp >> 4 & 1;
      ra = pop();
      rx = pop();
      /* Fall through... */
    case 0x81:
      tmp = pop();
      setpc(pop() | tmp << 8);
      return;
    case 0x83:
      push(pc & 0xFF);
      push(pc >> 8);
      push(rx);
      push(ra);
      tmp = (ccr[ccr_c] | ccr[ccr_z] << 1 | ccr[ccr_n] << 2
	     | ccr[ccr_i] << 3 | ccr[ccr_h] << 4 | 0xE0);
      push(tmp);
      setpc(mem(VECSTART+0xC) << 8 | mem(VECSTART+0xD));
      return;
    case 0x8E:
      printf("cycle %8lu: pc=%04x STOP\n", cycle, pc);
      exit(0);
      return;
    case 0x8F:
      fatal("WAIT unimplemented");
      return;
    case 0x97:
      rx = ra;
      return;
    case 0x98:  case 0x99:
      ccr[ccr_c] = opcode & 1;
      return;
    case 0x9A:  case 0x9B:
      ccr[ccr_i] = opcode & 1;
      return;
    case 0x9C:
      sp = 0xFF;
      return;
    case 0x9D:
      return;
    case 0x9F:
      ra = rx;
      return;
    default:
      fatal("invalid instruction at %04x", pc);
      return;
    }
}

void
softreset(void)
{
  setpc(mem(VECSTART+0xE) << 8 | mem(VECSTART+0xF));
  sp = 0xFF;
  ccr[ccr_i] = 1;
  /* also fix I/O ports here.  */
}

static byte
x2d(const char *x, const char *file, int line)
{
  int r;
  char c;
  
  if (!isxdigit(x[0]) || !isxdigit(x[1]))
    error_at_line(1, 0, file, line, "invalid character in S-record");
  c = toupper(x[0]);
  r = (c - '0' + ('0'+10-'A')*(c > '9')) << 4;
  c = toupper(x[1]);
  r |= c - '0' + ('0'+10-'A')*(c > '9');
  return r;
}

void
loadup(const char *file)
{
  FILE *f = fopen(file, "r");
  char line[520];
  byte buf[256];
  int ln;
  
  if (f == NULL)
    error(1, errno, "couldn't open %s", file);
  
  for (ln = 1;; ln++)
    {
      unsigned llen, cksum;
      int i;
      unsigned long addr, ca;
      int asize;

      if (fgets(line, sizeof(line), f) == NULL)
	error_at_line(1, errno, file, ln, "unexpected end-of-file");
      if (line[0] != 'S')
	continue;
      if (line[1] == '0')
	continue;
      if (line[1] >= '7' && line[1] <= '9')
	break;
      if (line[1] < '1' || line[1] > '3')
	error_at_line(1, 0, file, ln, "unknown S-record type `%c'", line[1]);
      asize = line[1] - '1' + 2;
      
      llen = x2d(line+2, file, ln);
      for (i = 0; i < llen; i++)
	buf[i] = x2d(line+4+2*i, file, ln);
      
      if (llen < asize+1)
	error_at_line(1, 0, file, ln, "S-record too short for type");
      cksum = llen;
      for (i = 0; i < llen; i++)
	cksum += buf[i];
      if (cksum % 256 != 255)
	error_at_line(0, 0, file, ln, 
		      "warning: S-record checksum invalid, %x", cksum);
	
      addr = 0;
      for (i = 0; i < asize; i++)
	addr = addr << 8 | buf[i];
      
      for (ca = 0; ca < llen - asize - 1; ca++)
	{
	  unsigned long a = addr + ca;
	  
	  if (a > ROMEND
	      || (a >= RAMSTART && a < RAMSTART+RAMSIZE 
		  && buf[ca+asize] != 0))
	    error_at_line(1, 0, file, ln, 
			  "S-record trying to write non-ROM address %04x", a);
	  memory[a] = buf[ca+asize];
	}
    }
  fclose(f);
}

int
main(int argc, char **argv)
{
  if (argc != 2)
    error(3, 0, "usage: %s srec-file", argv[0]);
  loadup(argv[1]);
  softreset();
  for (;;)
    {
      /* printf("cyc %4ld: pc=%04x a=%02x x=%02x sp=%02x\n",
	 cycle, pc, ra, rx, sp); */
      oneinst();
    }
}
