-- *************************************************************************
-- DISCLAIMER. THIS SOFTWARE WAS WRITTEN BY EMPLOYEES OF THE U.S.
-- GOVERNMENT AS A PART OF THEIR OFFICIAL DUTIES AND, THEREFORE, IS NOT
-- PROTECTED BY COPYRIGHT. HOWEVER, THIS SOFTWARE CODIFIES THE FINALIST
-- CANDIDATE ALGORITHMS (i.e., MARS, RC6tm, RIJNDAEL, SERPENT, AND
-- TWOFISH) IN THE ADVANCED ENCRYPTION STANDARD (AES) DEVELOPMENT EFFORT
-- SPONSORED BY THE NATIONAL INSTITUTE OF STANDARDS AND TECHNOLOGY (NIST)
-- AND MAY BE PROTECTED BY ONE OR MORE FORMS OF INTELLECTUAL PROPERTY. THE
-- U.S. GOVERNMENT MAKES NO WARRANTY, EITHER EXPRESSED OR IMPLIED,
-- INCLUDING BUT NO LIMITED TO ANY IMPLIED WARRANTIES OF MERCHANTABILITY
-- OR FITNESS FOR A PARTICULAR PURPOSE, REGARDING THIS SOFTWARE. THE U.S.
-- GOVERNMENT FURTHER MAKES NO WARRANTY THAT THIS SOFTWARE WILL NOT
-- INFRINGE ANY OTHER UNITED STATES OR FOREIGN PATENT OR OTHER
-- INTELLECTUAL PROPERTY RIGHT. IN NO EVENT SHALL THE U.S. GOVERNMENT BE
-- LIABLE TO ANYONE FOR COMPENSATORY, PUNITIVE, EXEMPLARY, SPECIAL,
-- COLLATERAL, INCIDENTAL, CONSEQUENTIAL, OR ANY OTHER TYPE OF DAMAGES IN
-- CONNECTION WITH OR ARISING OUT OF COPY OR USE OF THIS SOFTWARE.
-- *************************************************************************
--
-- File Name : RC6_Top_Pipelined.vhdl
-- Author    : NSA
-- Date      : 07 October 99
-- Project   : AES Candidate Evaluation --RC6
-- Purpose   : This model is the top level structural model for a
--             pipelined implementation of RC6, an Advanced Encryption
--             Standard Candidate. It consists of port mappings among the
--             lower level components.
-- Notes     :
-- ===========================================================================

library ieee;
use ieee.std_logic_1164.all;
use ieee.numeric_std.all;
use WORK.rc6_pack.all;

-- ===========================================================================
-- =========================== Interface Description =========================
-- ===========================================================================

entity RC6_TOP_PIPE is

  port(clock     : in std_logic;
        reset     : in std_logic;

        ENC_DEC_B : in std_logic;  -- '1' = encrypt, '0' = decrypt
        DATA_IN   : in SLV_128;    -- 128-bit input data word (plaintext)
        DATA_LOAD : in std_logic;  -- data valid; load new input data word
        CV_IN     : in CV_TYPE;    -- 128, 192, 256-bit cv (user supplied key)
        CV_LOAD   : in std_logic;  -- cv_in is valid; load new cryptovariable

        DATA_OUT  : out SLV_128;   -- 128-bit output data word (ciphertext)
        DONE      : out std_logic  -- indicates 'data_out' is valid

  );

end RC6_TOP_PIPE;

architecture STRUCTURAL of RC6_TOP_PIPE is


-- ===========================================================================
-- =========================== Component Definition ==========================
-- ===========================================================================

component INTERFACE 

  port (clock          :  in std_logic;
        reset          :  in std_logic;

        DATA_LOAD      :  in std_logic;
        DATAIN         :  in SLV_128;
        CV_LOAD        :  in std_logic;
        CVIN           :  in CV_TYPE;
        ENC_DEC_B      :  in std_logic;

        CTRL_DATA_LOAD :  out std_logic;
        CTRL_ENC_DEC_B :  out std_logic;
        ALG_DATA       :  out SLV_128;
        KS_CVLOAD      :  out std_logic;
        KS_CV          :  out L_ARRAY_TYPE
  );

end component;


component ALG_PIPE

  port (clock        :  in STD_LOGIC;      -- clock signal
        reset        :  in STD_LOGIC;      -- active high reset (asynchronous)

        ALG_DATAIN   :  in SLV_128;        -- input data
        ALG_KEY_ENC  :  in S_ARRAY_TYPE;   -- array of encrypt sub-keys (Si)
        ALG_KEY_DEC  :  in S_ARRAY_TYPE;   -- array of decrypt sub-keys (Si)
        ALG_ENC      :  in std_logic;      -- (1 = encrypt, 0 = decrypt)

        ALG_DATAOUT  :  out SLV_128        -- Processed output data
  );

end component;


component KEY_SCHEDULE_PIPE

  port (clock            :  in std_logic;     -- clock signal
        reset            :  in std_logic;     -- active high reset (asynch)

        KS_LOADCV        :  in std_logic;     -- load a new cryptovariable
        KS_CV            :  in L_ARRAY_TYPE;  -- cryptovariable input bus
        CR_ROUND_OUT     :  in S_ARRAY_TYPE;  -- S array from CR_ROUND

        KS_ROUND_KEY_ENC :  out S_ARRAY_TYPE; -- encrypt round key output (Si)
        KS_ROUND_KEY_DEC :  out S_ARRAY_TYPE  -- decrypt round key output (Si)

  );

end component;


component CONTROL_PIPE

  port (clock           :  in std_logic;
        reset           :  in std_logic;

        CTRL_DATA_LOAD  :  in std_logic;   -- data load signal from interface
        ENC_DEC_B       :  in std_logic;   -- encrypt/decrypt control

        OUT_DONE        :  out std_logic   -- done processing
       
  );

end component;

component CREATE_ROUNDKEY_BLOCK 
  
  port (cv_in    : in  L_ARRAY_TYPE;
        roundkey : out S_ARRAY_TYPE
  );

end component;


-- ===========================================================================
-- =========================== Signal Definition =============================
-- ===========================================================================

signal top_datain        : SLV_128;       -- top level data interconnection
signal top_dataload      : std_logic;     -- start new data connection
signal top_loadcv        : std_logic;     -- start new cv
signal top_cv            : L_ARRAY_TYPE;  -- cryptovariable bus interconnect
signal top_enc_decb      : std_logic;     -- encrypt select interconnect
signal top_round_key_enc : S_ARRAY_TYPE;  -- round key interconnects (array)
signal top_round_key_dec : S_ARRAY_TYPE;  -- round key interconnects (array)
signal top_ks_ready      : std_logic;     -- cv runup complete signal to ctrl
signal top_cr_round_out  : S_ARRAY_TYPE;

begin                                                                                                                                                                                                                             
INTER : INTERFACE port map (clock,             -- rising edge clock
                            reset,             -- active high reset
                            data_load,         -- ext. load new data
                            data_in,           -- ext. data input 
                            cv_load,           -- ext. load new cv
                            cv_in,             -- ext. cv input bus
                            enc_dec_b,         -- ext. encrypt select
                            top_dataload,      -- start new data
                            top_enc_decb,      -- encrypt select intercon.
                            top_datain,        -- data interconnect
                            top_loadcv,        -- load new cv intercon.
                            top_cv );          -- cv bus connection


CTRL : CONTROL_PIPE port map (clock,           -- rising edge clock
                              reset,           -- active high reset
                              top_dataload,    -- process new data
                              top_enc_decb,    -- encrypt select from interface
                              done);           -- start key expansion


ALG : ALG_PIPE port map (clock,                -- rising edge clock
                         reset,                -- active high reset
                         top_datain,           -- input data
                         top_round_key_enc,    -- round key input array
                         top_round_key_dec,    -- round key input array
                         top_enc_decb,         -- encrypt select
                         data_out );           -- data output connect

KEYSCH : KEY_SCHEDULE_PIPE port map (clock,               -- rising edge
                                     reset,               -- active high
                                     top_loadcv,          -- load new cv
                                     top_cv,              -- cv input bus
                                     top_cr_round_out, 
                                     top_round_key_enc,   -- round key outputs
                                     top_round_key_dec ); -- round key outputs


CROUT : CREATE_ROUNDKEY_BLOCK port map (top_cv,           -- cv input bus
                                        top_cr_round_out );

end STRUCTURAL;


-- ===========================================================================
-- =========================== Configuration =================================
-- ===========================================================================


configuration CFG_RC6_TOP_PIPE of RC6_TOP_PIPE is

   for STRUCTURAL

      for CTRL: CONTROL_PIPE use
         entity work.CONTROL_PIPE(CONTROL_PIPE_RTL);
      end for;

      for all: KEY_SCHEDULE_PIPE use
         entity work.KEY_SCHEDULE_PIPE(KEY_SCHEDULE_PIPE_RTL);
      end for;

      for all: ALG_PIPE use
         entity work.ALG_PIPE(ALG_PIPE_RTL);
      end for;

      for INTER: INTERFACE USE
         entity work.INTERFACE(INTERFACE_RTL);
      end for;

   end for;

end CFG_RC6_TOP_PIPE;





