-- *************************************************************************
-- DISCLAIMER. THIS SOFTWARE WAS WRITTEN BY EMPLOYEES OF THE U.S.
-- GOVERNMENT AS A PART OF THEIR OFFICIAL DUTIES AND, THEREFORE, IS NOT
-- PROTECTED BY COPYRIGHT. HOWEVER, THIS SOFTWARE CODIFIES THE FINALIST
-- CANDIDATE ALGORITHMS (i.e., MARS, RC6tm, RIJNDAEL, SERPENT, AND
-- TWOFISH) IN THE ADVANCED ENCRYPTION STANDARD (AES) DEVELOPMENT EFFORT
-- SPONSORED BY THE NATIONAL INSTITUTE OF STANDARDS AND TECHNOLOGY (NIST)
-- AND MAY BE PROTECTED BY ONE OR MORE FORMS OF INTELLECTUAL PROPERTY. THE
-- U.S. GOVERNMENT MAKES NO WARRANTY, EITHER EXPRESSED OR IMPLIED,
-- INCLUDING BUT NO LIMITED TO ANY IMPLIED WARRANTIES OF MERCHANTABILITY
-- OR FITNESS FOR A PARTICULAR PURPOSE, REGARDING THIS SOFTWARE. THE U.S.
-- GOVERNMENT FURTHER MAKES NO WARRANTY THAT THIS SOFTWARE WILL NOT
-- INFRINGE ANY OTHER UNITED STATES OR FOREIGN PATENT OR OTHER
-- INTELLECTUAL PROPERTY RIGHT. IN NO EVENT SHALL THE U.S. GOVERNMENT BE
-- LIABLE TO ANYONE FOR COMPENSATORY, PUNITIVE, EXEMPLARY, SPECIAL,
-- COLLATERAL, INCIDENTAL, CONSEQUENTIAL, OR ANY OTHER TYPE OF DAMAGES IN
-- CONNECTION WITH OR ARISING OUT OF COPY OR USE OF THIS SOFTWARE.
-- *************************************************************************
--
-- File Name : key_sched_pipe.vhdl
-- Author    : NSA
-- Date      : 04 Oct 1999
-- Project   : AES Candidate Evaluation --RC6 
-- Purpose   : build key schedule for pipelined implementation
-- Notes     :
-- ===========================================================================

library ieee;
use ieee.std_logic_1164.all;
use ieee.numeric_std.all;
use WORK.rc6_pack.all;

-- ===========================================================================
-- =========================== Interface Description =========================
-- ===========================================================================

entity KEY_SCHEDULE_PIPE is

  port (clock            :  in std_logic;     -- clock signal
        reset            :  in std_logic;     -- active high reset (asynch)

        KS_LOADCV        :  in std_logic;     -- load a new cryptovariable
        KS_CV            :  in L_ARRAY_TYPE;  -- cryptovariable input bus
        CR_ROUND_OUT     :  in S_ARRAY_TYPE;  -- S array from CR_ROUND

        KS_ROUND_KEY_ENC :  out S_ARRAY_TYPE; -- encrypt round key output (Si)
        KS_ROUND_KEY_DEC :  out S_ARRAY_TYPE  -- decrypt round key output (Si)

  );

end KEY_SCHEDULE_PIPE;

architecture KEY_SCHEDULE_PIPE_RTL of KEY_SCHEDULE_PIPE is


-- ===========================================================================
-- =========================== Component Definition ==========================
-- ===========================================================================

component REG128B 

  port (clock      :  in std_logic;   -- clock signal
        reset      :  in std_logic;   -- active high reset (asynchronous)

        DATA_IN    :  in SLV_128;     -- input data bus
        DATA_OUT   :  out SLV_128     -- output data bus

  );

end component;


component REGISTER_DELAY_S

  port (clock    :  in std_logic;   -- clock signal
        reset    :  in std_logic;   -- active high reset (asynchronous)
    
        LATCH    :  in std_logic;
        DATA_IN  :  in SLV_32;      -- input data bus

        DATA_OUT :  out SLV_32      -- output data bus

  );

end component;

-- ===========================================================================
-- =========================== Signal Definition =============================
-- ===========================================================================

signal S_REG           : S_MESH_TYPE;    -- S register interconnections
signal S_INT           : S_MESH_TYPE;    -- procedure call to reg interconnect
signal S_ORIG          : S_ARRAY_TYPE;
signal LATCH           : LATCH_TYPE;

begin


-- ===========================================================================
-- ====================== Generate Pipe Structure ============================
-- ===========================================================================
--
-- PURPOSE:  
--
-- The following generate statements create a pipelined architecture for 
-- the key expansion.  For each round of expansion, a new set of S values
-- are created, followed by the creation of Si (Sbox lookup on the new S
-- values that were generated).


S_ORIG <= CR_ROUND_OUT; -- added on 6 Mar 00 by cficke

G1: for PIPE_STEP in 0 to ((((22/RUNUP_CYCLES)+1)*44)-1) generate

   G1a: if PIPE_STEP < 44 generate
      S_INT(PIPE_STEP) <= S_ORIG(PIPE_STEP);
   end generate;

   G1b: if PIPE_STEP >= 44 generate
      S_INT(PIPE_STEP) <= S_REG(PIPE_STEP-44);
   end generate;

   G1c: if PIPE_STEP < 44 generate

   SREG : REGISTER_DELAY_S port map (clock,
                                     reset,
                                     KS_LOADCV,
                                     S_INT(PIPE_STEP),
                                     S_REG(PIPE_STEP) );  -- Register S values
   end generate;
 
   G1d: if PIPE_STEP >= 44 generate

   SREG : REGISTER_DELAY_S port map (clock,
                                     reset,
                                     LATCH(((PIPE_STEP/44))*RUNUP_CYCLES-1),
                                     S_INT(PIPE_STEP),
                                     S_REG(PIPE_STEP) );  -- Register S values
   end generate;
end generate;  -- G1
  

G2: for roundkey_step in 0 to 43 generate

   -- Generate interconnections for the encryption round keys
   KS_ROUND_KEY_ENC(roundkey_step) <= 
   S_REG( ((roundkey_step/2)/RUNUP_CYCLES)*44+roundkey_step );

   -- Generate interconnections for the decryption round keys
   KS_ROUND_KEY_DEC( roundkey_step) <=
   S_REG( ((roundkey_step/2)/RUNUP_CYCLES)*44+(43-roundkey_step) );

end generate;    -- G2

-- ===========================================================================
-- =========================== Data Movement =================================
-- ===========================================================================


DATA_FLOW : process( clock, reset )

begin

if reset = '1' then
                          
   for PIPE_STEP in 0 to 22+RUNUP_CYCLES loop
      LATCH(PIPE_STEP) <= '0';
   end loop;

elsif clock'event and clock = '1' then

   if KS_LOADCV = '1' then

      LATCH(0)     <= '1';

   else

      LATCH(0)     <= '0';

   end if; -- KS_LOADCV = '1'

   LATCH(1 to 22+RUNUP_CYCLES) <= LATCH(0 to 21+RUNUP_CYCLES);

end if; -- reset = '1'

end process; -- DATA_FLOW

end KEY_SCHEDULE_PIPE_RTL;

-- ===========================================================================
-- ========================= Configuration ===================================
-- ===========================================================================

configuration CFG_KEY_SCHEDULE_PIPE of KEY_SCHEDULE_PIPE is

   for KEY_SCHEDULE_PIPE_RTL

   end for;

end;


