-- *************************************************************************
-- DISCLAIMER. THIS SOFTWARE WAS WRITTEN BY EMPLOYEES OF THE U.S.
-- GOVERNMENT AS A PART OF THEIR OFFICIAL DUTIES AND, THEREFORE, IS NOT
-- PROTECTED BY COPYRIGHT. HOWEVER, THIS SOFTWARE CODIFIES THE FINALIST
-- CANDIDATE ALGORITHMS (i.e., MARS, RC6tm, RIJNDAEL, SERPENT, AND
-- TWOFISH) IN THE ADVANCED ENCRYPTION STANDARD (AES) DEVELOPMENT EFFORT
-- SPONSORED BY THE NATIONAL INSTITUTE OF STANDARDS AND TECHNOLOGY (NIST)
-- AND MAY BE PROTECTED BY ONE OR MORE FORMS OF INTELLECTUAL PROPERTY. THE
-- U.S. GOVERNMENT MAKES NO WARRANTY, EITHER EXPRESSED OR IMPLIED,
-- INCLUDING BUT NO LIMITED TO ANY IMPLIED WARRANTIES OF MERCHANTABILITY
-- OR FITNESS FOR A PARTICULAR PURPOSE, REGARDING THIS SOFTWARE. THE U.S.
-- GOVERNMENT FURTHER MAKES NO WARRANTY THAT THIS SOFTWARE WILL NOT
-- INFRINGE ANY OTHER UNITED STATES OR FOREIGN PATENT OR OTHER
-- INTELLECTUAL PROPERTY RIGHT. IN NO EVENT SHALL THE U.S. GOVERNMENT BE
-- LIABLE TO ANYONE FOR COMPENSATORY, PUNITIVE, EXEMPLARY, SPECIAL,
-- COLLATERAL, INCIDENTAL, CONSEQUENTIAL, OR ANY OTHER TYPE OF DAMAGES IN
-- CONNECTION WITH OR ARISING OUT OF COPY OR USE OF THIS SOFTWARE.
-- *************************************************************************

-- ===========================================================================
-- File Name : TWOFISH_Top_Pipelined.vhdl
-- Author    : NSA
-- Date      : January 2000
-- Project   : AES Candidate Evaluation - TWOFISH
-- Purpose   : This model is the top level structural model for a
--             pipelined implementation of TWOFISH, an Advanced Encryption
--             Standard Candidate. It consists of port mappings among the
--             lower level components.
-- Notes     :
-- ===========================================================================

library ieee;
use ieee.std_logic_1164.all;
use WORK.twofish_pack.all;

-- ===========================================================================
-- =========================== Interface Description =========================
-- ===========================================================================

entity TWOFISH_TOP_PIPE is

  port(clock     : in std_logic;
        reset     : in std_logic;

        ENC_DEC_B : in std_logic;  -- '1' = encrypt, '0' = decrypt
        DATA_IN   : in SLV_128;    -- 128-bit input data word (plaintext)
        DATA_LOAD : in std_logic;  -- data valid; load new input data word
        CV_IN     : in SLV_256;    -- 128, 192, 256-bit cv (user supplied key)
        CV_LOAD   : in std_logic;  -- cv_in is valid; load new cryptovariable
        CV_SIZE   : in SLV_2;      -- '00'= 128, '01'= 192, '10'= 256

        DATA_OUT  : out SLV_128;   -- 128-bit output data word (ciphertext)
        DONE      : out std_logic  -- indicates 'data_out' is valid

  );

end TWOFISH_TOP_PIPE;

architecture STRUCTURAL of TWOFISH_TOP_PIPE is


-- ===========================================================================
-- =========================== Component Definition ==========================
-- ===========================================================================

component INTERFACE_PIPE 

  port (clock          :  in std_logic;
        reset          :  in std_logic;

        DATA_LOAD      :  in std_logic;
        DATAIN         :  in SLV_128;
        CV_LOAD        :  in std_logic;
        CV_SIZE        :  in SLV_2;
        CVIN           :  in SLV_256;
        ENC_DEC_B      :  in std_logic;

        CTRL_DATA_LOAD :  out std_logic;
        CTRL_ENC_DEC_B :  out std_logic;
        ALG_DATA       :  out SLV_128;
        KS_CVLOAD      :  out std_logic;
        KS_CV          :  out SLV_256;
        CVSIZE_ARRAY   :  out CVSIZE_PIPE_TYPE
  );

end component;


component ALG_PIPE

  port (clock        :  in STD_LOGIC;      -- clock signal
        reset        :  in STD_LOGIC;      -- active high reset (asynchronous)

        ALG_DATAIN   :  in SLV_128;        -- input data
        ALG_KEY_ENC  :  in KEY_ARRAY_TYPE;   -- array of encrypt sub-keys (Si)
        ALG_KEY_DEC  :  in KEY_ARRAY_TYPE;   -- array of decrypt sub-keys (Si)
        ALG_CVSIZE   :  in CVSIZE_PIPE_TYPE; -- pipelined CV size
        ALG_S        :  in S_PIPE_TYPE;      -- pipelined S values
        ALG_ENC      :  in std_logic;      -- (1 = encrypt, 0 = decrypt)

        ALG_DATAOUT  :  out SLV_128        -- Processed output data
  );

end component;


component KEY_SCHEDULE_PIPE

  port (clock            :  in std_logic;       -- clock signal
        reset            :  in std_logic;       -- active high reset (asynch)

        KS_LOADCV        :  in std_logic;       -- load a new cryptovariable
        KS_CV            :  in SLV_256;         -- cryptovariable input bus
        KS_CVSIZE        :  in CVSIZE_PIPE_TYPE;-- cv length
        KS_ENC           :  in std_logic;       -- encrypt/decrypt
        KS_S             :  out S_PIPE_TYPE;       -- S-box keys (pipelined)
        KS_ROUND_KEY_ENC :  out KEY_ARRAY_TYPE; -- encrypt round key output
        KS_ROUND_KEY_DEC :  out KEY_ARRAY_TYPE  -- decrypt round key output

  );

end component;


component CONTROL_PIPE

  port (clock           :  in std_logic;
        reset           :  in std_logic;

        CTRL_DATA_LOAD  :  in std_logic;   -- data load signal from interface

        OUT_DONE        :  out std_logic   -- done processing
       
  );

end component;



-- ===========================================================================
-- =========================== Signal Definition =============================
-- ===========================================================================

signal top_datain        : SLV_128;         -- top level data interconnection
signal top_dataload      : std_logic;       -- start new data connection
signal top_loadcv        : std_logic;       -- start new cv
signal top_cv            : SLV_256;         -- cryptovariable bus interconnect
signal top_cvsize        : CVSIZE_PIPE_TYPE;-- cryptovariable size interconnect
signal top_enc_decb      : std_logic;       -- encrypt select interconnect
signal top_round_key_enc : KEY_ARRAY_TYPE;  -- round key interconnects (array)
signal top_round_key_dec : KEY_ARRAY_TYPE;  -- round key interconnects (array)
signal top_s             : S_PIPE_TYPE;

begin                                                                                                                                                                                                                             
INTER : INTERFACE_PIPE port map (clock,             -- rising edge clock
                            reset,             -- active high reset
                            data_load,         -- ext. load new data
                            data_in,           -- ext. data input 
                            cv_load,           -- ext. load new cv
                            cv_size,           -- ext. cv size select
                            cv_in,             -- ext. cv input bus
                            enc_dec_b,         -- ext. encrypt select
                            top_dataload,      -- start new data
                            top_enc_decb,      -- encrypt select intercon.
                            top_datain,        -- data interconnect
                            top_loadcv,        -- load new cv intercon.
                            top_cv,            -- cv bus connection
                            top_cvsize);       -- cv size connection


CTRL : CONTROL_PIPE port map (clock,           -- rising edge clock
                              reset,           -- active high reset
                              top_dataload,    -- process new data
                              done);           -- start key expansion


ALG : ALG_PIPE port map (clock,                -- rising edge clock
                         reset,                -- active high reset
                         top_datain,           -- input data
                         top_round_key_enc,    -- round key input array
                         top_round_key_dec,    -- round key input array
                         top_cvsize,        -- cv size
                         top_s,                -- S-box keys
                         top_enc_decb,         -- encrypt select
                         data_out );           -- data output connect

KEYSCH : KEY_SCHEDULE_PIPE port map (clock,               -- rising edge
                                     reset,               -- active high
                                     top_loadcv,          -- load new cv
                                     top_cv,              -- cv input bus
                                     top_cvsize,          -- cv size
                                     top_enc_decb,        -- encrypt/decrypt
                                     top_s,               -- S-box keys
                                     top_round_key_enc,   -- round key outputs
                                     top_round_key_dec ); -- round key outputs


end STRUCTURAL;


-- ===========================================================================
-- =========================== Configuration =================================
-- ===========================================================================


configuration CFG_TWOFISH_TOP_PIPE of TWOFISH_TOP_PIPE is

   for STRUCTURAL

      for CTRL: CONTROL_PIPE use
         entity work.CONTROL_PIPE(CONTROL_PIPE_RTL);
      end for;

      for all: KEY_SCHEDULE_PIPE use
         entity work.KEY_SCHEDULE_PIPE(KEY_SCHEDULE_PIPE_RTL);
      end for;

      for all: ALG_PIPE use
         entity work.ALG_PIPE(ALG_PIPE_RTL);
      end for;

      for INTER: INTERFACE_PIPE USE
         entity work.INTERFACE_PIPE(INTERFACE_PIPE_RTL);
      end for;

   end for;

end CFG_TWOFISH_TOP_PIPE;

