-- *************************************************************************
-- DISCLAIMER. THIS SOFTWARE WAS WRITTEN BY EMPLOYEES OF THE U.S.
-- GOVERNMENT AS A PART OF THEIR OFFICIAL DUTIES AND, THEREFORE, IS NOT
-- PROTECTED BY COPYRIGHT. HOWEVER, THIS SOFTWARE CODIFIES THE FINALIST
-- CANDIDATE ALGORITHMS (i.e., MARS, RC6tm, RIJNDAEL, SERPENT, AND
-- TWOFISH) IN THE ADVANCED ENCRYPTION STANDARD (AES) DEVELOPMENT EFFORT
-- SPONSORED BY THE NATIONAL INSTITUTE OF STANDARDS AND TECHNOLOGY (NIST)
-- AND MAY BE PROTECTED BY ONE OR MORE FORMS OF INTELLECTUAL PROPERTY. THE
-- U.S. GOVERNMENT MAKES NO WARRANTY, EITHER EXPRESSED OR IMPLIED,
-- INCLUDING BUT NO LIMITED TO ANY IMPLIED WARRANTIES OF MERCHANTABILITY
-- OR FITNESS FOR A PARTICULAR PURPOSE, REGARDING THIS SOFTWARE. THE U.S.
-- GOVERNMENT FURTHER MAKES NO WARRANTY THAT THIS SOFTWARE WILL NOT
-- INFRINGE ANY OTHER UNITED STATES OR FOREIGN PATENT OR OTHER
-- INTELLECTUAL PROPERTY RIGHT. IN NO EVENT SHALL THE U.S. GOVERNMENT BE
-- LIABLE TO ANYONE FOR COMPENSATORY, PUNITIVE, EXEMPLARY, SPECIAL,
-- COLLATERAL, INCIDENTAL, CONSEQUENTIAL, OR ANY OTHER TYPE OF DAMAGES IN
-- CONNECTION WITH OR ARISING OUT OF COPY OR USE OF THIS SOFTWARE.
-- *************************************************************************

-- ===========================================================================
-- File Name : interface_pipe.vhdl
-- Author    : NSA
-- Date      : January 2000
-- Project   : AES Candidate Evaluation - TWOFISH
-- Purpose   : This block buffers all inputs and routes them to the 
--             necessary entity.
-- Notes     :
-- ===========================================================================

library ieee;
use ieee.std_logic_1164.all;
use WORK.twofish_pack.all;

-- ===========================================================================
-- =========================== Interface Description =========================
-- ===========================================================================

entity INTERFACE_PIPE is

  port (clock          :  in std_logic;
        reset          :  in std_logic;

        DATA_LOAD      :  in std_logic;     -- data load pulse
        DATAIN         :  in SLV_128;       -- 128 bit block
        CV_LOAD        :  in std_logic;     -- crypto variable load pulse
        CV_SIZE        :  in SLV_2;         -- '00'= 128, '01'= 192, '10'= 256
        CVIN           :  in SLV_256;
        ENC_DEC_B      :  in std_logic;     -- '1' = encrypt, '0' = decrypt

        CTRL_DATA_LOAD :  out std_logic;    -- data load signal to controller
        CTRL_ENC_DEC_B :  out std_logic;
        ALG_DATA       :  out SLV_128;      -- 128 bit data block to algorithm
        KS_CVLOAD      :  out std_logic;
        KS_CV          :  out SLV_256;
        CVSIZE_ARRAY   :  out CVSIZE_PIPE_TYPE
  );

end INTERFACE_PIPE;
 
architecture INTERFACE_PIPE_RTL of INTERFACE_PIPE is

-- ===========================================================================
-- =========================== Signal Definition =============================
-- ===========================================================================

signal ALG_DATA_INT      : SLV_128;
signal CVSIZE_ARRAY_INT  : CVSIZE_PIPE_TYPE;
signal CV_INT            : SLV_256;

begin

-- ===========================================================================
-- =========================== Data Movement =================================
-- ===========================================================================

ALG_DATA      <= ALG_DATA_INT;
CVSIZE_ARRAY  <= CVSIZE_ARRAY_INT;
KS_CV         <= CV_INT;

main: process( clock, reset )

begin

   if reset = '1' then

      CTRL_DATA_LOAD <= '0';
      CTRL_ENC_DEC_B <= '0';
      KS_CVLOAD      <= '0';
      CV_INT         <= (others => '0');
      ALG_DATA_INT   <= (others => '0');
      for i in FIRST_ROUND to LAST_ROUND loop
        CVSIZE_ARRAY_INT(i) <= (others => '0');
      end loop;

   elsif clock'event and clock = '1' then

      CTRL_DATA_LOAD     <= DATA_LOAD;  -- pass control signals through
      CTRL_ENC_DEC_B     <= ENC_DEC_B;
      KS_CVLOAD          <= CV_LOAD;

      CVSIZE_ARRAY_INT(0) <= CV_SIZE;
      for i in FIRST_ROUND+1 to LAST_ROUND loop
        CVSIZE_ARRAY_INT(i) <= CVSIZE_ARRAY_INT(i-1);
      end loop;

      if DATA_LOAD = '1' then
        ALG_DATA_INT <= DATAIN;
      else
        ALG_DATA_INT   <= ALG_DATA_INT;
      end if;

      if CV_LOAD = '1' then         -- latch CV
        CV_INT <= CVIN;
      else                          -- hold previous CV data
         CV_INT <= CV_INT;
      end if;  -- CV_LOAD = '1'


   end if;  -- reset = '1'

end process;

end INTERFACE_PIPE_RTL;
