-- *************************************************************************
-- DISCLAIMER. THIS SOFTWARE WAS WRITTEN BY EMPLOYEES OF THE U.S.
-- GOVERNMENT AS A PART OF THEIR OFFICIAL DUTIES AND, THEREFORE, IS NOT
-- PROTECTED BY COPYRIGHT. HOWEVER, THIS SOFTWARE CODIFIES THE FINALIST
-- CANDIDATE ALGORITHMS (i.e., MARS, RC6tm, RIJNDAEL, SERPENT, AND
-- TWOFISH) IN THE ADVANCED ENCRYPTION STANDARD (AES) DEVELOPMENT EFFORT
-- SPONSORED BY THE NATIONAL INSTITUTE OF STANDARDS AND TECHNOLOGY (NIST)
-- AND MAY BE PROTECTED BY ONE OR MORE FORMS OF INTELLECTUAL PROPERTY. THE
-- U.S. GOVERNMENT MAKES NO WARRANTY, EITHER EXPRESSED OR IMPLIED,
-- INCLUDING BUT NO LIMITED TO ANY IMPLIED WARRANTIES OF MERCHANTABILITY
-- OR FITNESS FOR A PARTICULAR PURPOSE, REGARDING THIS SOFTWARE. THE U.S.
-- GOVERNMENT FURTHER MAKES NO WARRANTY THAT THIS SOFTWARE WILL NOT
-- INFRINGE ANY OTHER UNITED STATES OR FOREIGN PATENT OR OTHER
-- INTELLECTUAL PROPERTY RIGHT. IN NO EVENT SHALL THE U.S. GOVERNMENT BE
-- LIABLE TO ANYONE FOR COMPENSATORY, PUNITIVE, EXEMPLARY, SPECIAL,
-- COLLATERAL, INCIDENTAL, CONSEQUENTIAL, OR ANY OTHER TYPE OF DAMAGES IN
-- CONNECTION WITH OR ARISING OUT OF COPY OR USE OF THIS SOFTWARE.
-- *************************************************************************

-- ===========================================================================
-- File Name : SERPENT_Top_Iterative.vhdl
-- Author    : NSA
-- Date      : Aug 99
-- Project   : AES Candidate Evaluation
-- Purpose   : This model is the top level structural model for an
--             iterative implementation of SERPENT, an Advanced Encryption
--             Standard Candidate. It consists of port mappings among the
--             lower level components.
-- Notes     :
-- ===========================================================================

library ieee;
use ieee.std_logic_1164.all;
use ieee.numeric_std.all;
use WORK.serpent_pack.all;

-- ===========================================================================
-- =========================== Interface Description =========================
-- ===========================================================================

entity SERPENT_TOP_ITER is

  port (clock     : in std_logic;
        reset     : in std_logic;

        ENC_DEC_B : in std_logic;  -- '1' = encrypt, '0' = decrypt
        DATA_IN   : in SLV_128;    -- 128-bit input data word (plaintext)
        DATA_LOAD : in std_logic;  -- data valid; load new input data word
        CV_IN     : in CV_TYPE;    -- 128, 192, 256-bit cv (user supplied key)
        CV_LOAD   : in std_logic;  -- cv_in is valid; load new cryptovariable
        CV_SIZE   : in SLV_2;      -- '00'= 128, '01'= 192, '10'= 256

        DATA_OUT  : out SLV_128;   -- 128-bit output data word (ciphertext)
        DONE      : out std_logic  -- indicates 'data_out' is valid

  );

end SERPENT_TOP_ITER;

architecture STRUCTURAL of SERPENT_TOP_ITER is


-- ===========================================================================
-- =========================== Component Definition ==========================
-- ===========================================================================

component CONTROL_ITER is

  port (clock          : in std_logic;
        reset          : in std_logic;

        DATA_LOAD      : in std_logic;   -- data start signal from interface
        CV_LOAD        : in std_logic;   -- cv start signal from interface
        CTRL_ALG_DONE  : in std_logic;   -- done signal from algorithm unit
        CTRL_KS_READY  : in std_logic;   -- runup done, schedule ready
        CTRL_ENC       : in std_logic;   -- encrypt/decrypt signal

        CTRL_ALG_START : out std_logic;  -- start encryption
        CTRL_KS_START  : out std_logic   -- start key schedule
       
  );

end component;

component ALG_ITERATIVE

  port (clock       : in STD_LOGIC;
        reset       : in STD_LOGIC;

        ALG_DATAIN  : in SLV_128;
        ALG_KEY1    : in SLV_128;
        ALG_START   : in STD_LOGIC;
        ALG_ENC     : in STD_LOGIC;

        ALG_DATAOUT : out SLV_128;
        ALG_DONE    : out STD_LOGIC
  );

end component;

component INTERFACE is

  port (clock          : in std_logic;
        reset          : in std_logic;

        DATA_LOAD      : in std_logic;
        DATAIN         : in SLV_128;
        CV_LOAD        : in std_logic;
        CV_SIZE        : in SLV_2;
        CVIN           : in SLV_256;
        ENC_DEC_B      : in std_logic;

        CTRL_DATA_LOAD : out std_logic;
        CTRL_ENC_DEC_B : out std_logic;
        ALG_DATA       : out SLV_128;
        KS_CVLOAD      : out std_logic;
        KS_CV          : out SLV_256
  );

end component;

component Key_Schedule_Iterative

  port (clock        : in std_logic;    -- clock signal
        reset        : in std_logic;    -- active high reset (asynch)

        KS_LOADCV    : in std_logic;    -- load a new cryptovariable
        KS_START     : in std_logic;    -- start a new expansion sequence
        KS_CV        : in SLV_256;      -- cryptovariable input bus
        KS_ENC       : in std_logic;    -- '1' = encrypt, '0' = decrypt

        KS_READY     : out std_logic;   -- '1'=cv expanded, '0'=inprocess
        KS_ROUND_KEY : out SLV_128      -- output round key (Ki-1)

  );

end component;


-- ===========================================================================
-- =========================== Signal Definition =============================
-- ===========================================================================

signal top_datain     : SLV_128;        -- top level data interconnection
signal top_dataout    : SLV_128;        -- top level data output connection
signal top_dataload   : std_logic;      -- start new data connection
signal top_loadcv     : std_logic;      -- start new cv
signal top_cv_size    : SLV_2;          -- cryptovariable size interconnect
signal top_cv         : SLV_256;        -- cryptovariable bus interconnect
signal top_enc_decb   : std_logic;      -- encrypt select interconnect
signal top_round_key1 : SLV_128;        -- round key interconnects
signal top_alg_start  : std_logic;      -- algorithm start connection
signal top_done       : std_logic;      -- done signal connection
signal top_ks_start   : std_logic;      -- key expansion start signal
signal top_ks_ready   : std_logic;      -- key runup complete signal

begin

CTRL : CONTROL_ITER port map (clock,           -- rising edge clock
                              reset,           -- active high reset
                              top_dataload,    -- process new data
                              top_loadcv,      -- process new cv
                              top_done,        -- alg complete
                              top_ks_ready,    -- key runup complete 
                              top_enc_decb,    -- enc/dec select 
                              top_alg_start,   -- start alg processing
                              top_ks_start );  -- start key expansion

ALG : ALG_ITERATIVE port map (clock,           -- rising edge clock  
                              reset,           -- active high reset
                              top_datain,      -- input data
                              top_round_key1,  -- round key inputs
                              top_alg_start,   -- start processing 
                              top_enc_decb,    -- encrypt select
                              data_out,        -- processed data output
                              top_done );      -- processing complete

KEYSCH : Key_Schedule_Iterative port map (clock,            -- rising edge
                                          reset,            -- active high
                                          top_loadcv,       -- load new cv
                                          top_ks_start,     -- start key expan
                                          top_cv,           -- cv input bus
                                          top_enc_decb,     -- encrypt select
                                          top_ks_ready,     -- runup complete
                                          top_round_key1);  -- key output

INTER : INTERFACE port map (clock,             -- rising edge clock
                            reset,             -- active high reset
                            data_load,         -- ext. load new data
                            data_in,           -- ext. data input 
                            cv_load,           -- ext. load new cv
                            cv_size,           -- ext. cv size select
                            cv_in,             -- ext. cv input bus
                            enc_dec_b,         -- ext. encrypt select
                            top_dataload,      -- start new data
                            top_enc_decb,      -- encrypt select intercon.
                            top_datain,        -- data interconnect
                            top_loadcv,        -- load new cv intercon.
                            top_cv );          -- cv bus connection

DONE <= top_done;

end STRUCTURAL;


-- ===========================================================================
-- =========================== Configuration =================================
-- ===========================================================================


configuration conf_SERPENT_TOP_ITER of SERPENT_TOP_ITER is

   for STRUCTURAL

      for CTRL: CONTROL_ITER use
         entity work.CONTROL_ITER(CONTROL_ITER_RTL);
      end for;

      for all: Key_Schedule_Iterative use
         entity work.Key_Schedule_Iterative(Key_Schedule_Iterative_RTL);
      end for;

      for all: ALG_ITERATIVE use
         entity work.ALG_ITERATIVE(ALG_ITERATIVE_RTL);
      end for;

      for INTER: INTERFACE USE
         entity work.INTERFACE(INTERFACE_RTL);
      end for;

   end for;

end conf_SERPENT_TOP_ITER;
