-- *************************************************************************
-- DISCLAIMER. THIS SOFTWARE WAS WRITTEN BY EMPLOYEES OF THE U.S.
-- GOVERNMENT AS A PART OF THEIR OFFICIAL DUTIES AND, THEREFORE, IS NOT
-- PROTECTED BY COPYRIGHT. HOWEVER, THIS SOFTWARE CODIFIES THE FINALIST
-- CANDIDATE ALGORITHMS (i.e., MARS, RC6tm, RIJNDAEL, SERPENT, AND
-- TWOFISH) IN THE ADVANCED ENCRYPTION STANDARD (AES) DEVELOPMENT EFFORT
-- SPONSORED BY THE NATIONAL INSTITUTE OF STANDARDS AND TECHNOLOGY (NIST)
-- AND MAY BE PROTECTED BY ONE OR MORE FORMS OF INTELLECTUAL PROPERTY. THE
-- U.S. GOVERNMENT MAKES NO WARRANTY, EITHER EXPRESSED OR IMPLIED,
-- INCLUDING BUT NO LIMITED TO ANY IMPLIED WARRANTIES OF MERCHANTABILITY
-- OR FITNESS FOR A PARTICULAR PURPOSE, REGARDING THIS SOFTWARE. THE U.S.
-- GOVERNMENT FURTHER MAKES NO WARRANTY THAT THIS SOFTWARE WILL NOT
-- INFRINGE ANY OTHER UNITED STATES OR FOREIGN PATENT OR OTHER
-- INTELLECTUAL PROPERTY RIGHT. IN NO EVENT SHALL THE U.S. GOVERNMENT BE
-- LIABLE TO ANYONE FOR COMPENSATORY, PUNITIVE, EXEMPLARY, SPECIAL,
-- COLLATERAL, INCIDENTAL, CONSEQUENTIAL, OR ANY OTHER TYPE OF DAMAGES IN
-- CONNECTION WITH OR ARISING OUT OF COPY OR USE OF THIS SOFTWARE.
-- *************************************************************************

-- ===========================================================================
-- File Name : alg_pipe.vhdl
-- Author    : NSA
-- Date      : 07 September 99
-- Project   : AES Candidate Evaluation
-- Purpose   : This block implements the Serpent algorithm
--             for a pipelined case
-- Notes     :
-- ===========================================================================

library ieee;
use ieee.std_logic_1164.all;
use ieee.numeric_std.all;
use WORK.serpent_pack.all;

-- ===========================================================================
-- =========================== Interface Description =========================
-- ===========================================================================

entity ALG_PIPE is

  port (clock       :  in STD_LOGIC;      -- clock signal
        reset       :  in STD_LOGIC;      -- active high reset (asynchronous)

        ALG_DATAIN  :  in SLV_128;        -- input data
        ALG_KEY     :  in PIPE_KEY_TYPE;  -- array of sub-keys (Ki)
        ALG_ENC     :  in std_logic;      -- '1' = encrypt, '0' = decrypt
        ALG_DATAOUT :  out SLV_128        -- processed output data

  );

end ALG_PIPE;

architecture ALG_PIPE_RTL of ALG_PIPE is


-- ===========================================================================
-- ========================== Component Definition ===========================
-- ===========================================================================

component REG128B 

  port (clock    :  in std_logic;   -- clock signal
        reset    :  in std_logic;   -- active high reset (asynchronous)

        DATA_IN  :  in SLV_128;     -- input data bus
        DATA_OUT :  out SLV_128     -- output data bus

  );

end component;


-- ===========================================================================
-- =========================== Signal Definition =============================
-- ===========================================================================

signal ALG_DATA       : PIPE_DATA_TYPE;  -- register output
signal ALG_DATA_INT   : PIPE_DATA_TYPE;  -- procedure call to reg interconnect
signal ALG_DATAIN_INT : SLV_128;         -- data to init. perm. interconnect
signal ROUND_STEP     : ROUND_TYPE;      -- indicates current algorithm step

begin


-- ===========================================================================
-- ======================== Generate Pipe Structure ==========================
-- ===========================================================================
--
-- PURPOSE:  
--
-- The following generate statements create a pipelined architecture for 
-- the encryption/decryption.

G1: for ROUND_STEP in FIRST_ROUND to LAST_ROUND+1 generate

-- =========================== First Round ===================================
-- Generate first round and special case connections

   G1a: if (ROUND_STEP = FIRST_ROUND) generate

      INIT : IP( ALG_DATAIN, ALG_DATAIN_INT );          -- initial permutation

      STEP : SERPENT_ROUND( ALG_DATAIN_INT,             -- input data 
                            ALG_KEY(ROUND_STEP),        -- round sub-key
                            ALG_ENC,                    -- enc/dec direction
                            STD_LOGIC_VECTOR(TO_UNSIGNED(ROUND_STEP, 6)),
                            ALG_DATA_INT(ROUND_STEP) ); -- data interconnect

      DATAREG : REG128B port map ( clock,
                                   reset,
                                   ALG_DATA_INT(ROUND_STEP),   -- interconnect
                                   ALG_DATA(ROUND_STEP) );     -- output data




   end generate; -- G1a

-- ========================== Round i ========================================
-- Generate middle rounds for creating encryption/decryption.

   G1b: if (ROUND_STEP < 32 and ROUND_STEP > 0) generate

      STEP : SERPENT_ROUND( ALG_DATA(ROUND_STEP-1),      -- input data
                            ALG_KEY(ROUND_STEP),         -- round sub-key
                            ALG_ENC,                     -- enc/dec direction
                            STD_LOGIC_VECTOR(TO_UNSIGNED(ROUND_STEP, 6)),
                            ALG_DATA_INT(ROUND_STEP) );  -- data interconnect

      DATAREG : REG128B port map ( clock,
                                   reset,
                                   ALG_DATA_INT(ROUND_STEP),  -- interconnect 
                                   ALG_DATA(ROUND_STEP) );    -- output data


   end generate; -- G1b

-- ========================= Last Round ======================================
-- Generate last round and special case connections

   G1c: if (ROUND_STEP = LAST_ROUND+1) generate

      STEP : SERPENT_ROUND( ALG_DATA(ROUND_STEP-1),      -- input data
                            ALG_KEY(ROUND_STEP),         -- sub-key 31
                            ALG_ENC,                     -- enc/dec direction
                            STD_LOGIC_VECTOR(TO_UNSIGNED(ROUND_STEP, 6)),
                            ALG_DATA_INT(ROUND_STEP) );  -- data interconnect

      UNMAP : FP( ALG_DATA_INT(ROUND_STEP), ALG_DATA(ROUND_STEP) );

      DATAREG : REG128B port map ( clock,
                                   reset,
                                   ALG_DATA(ROUND_STEP), -- interconnect
                                   ALG_DATAOUT );        -- output data

   end generate; -- G1c

end generate;  -- G1

end ALG_PIPE_RTL;

-- ===========================================================================
-- ============================ Configuration ================================
-- ===========================================================================

configuration CFG_ALG_PIPE of ALG_PIPE is

   for ALG_PIPE_RTL

   end for;

end CFG_ALG_PIPE;


