-- *************************************************************************
-- DISCLAIMER. THIS SOFTWARE WAS WRITTEN BY EMPLOYEES OF THE U.S.
-- GOVERNMENT AS A PART OF THEIR OFFICIAL DUTIES AND, THEREFORE, IS NOT
-- PROTECTED BY COPYRIGHT. HOWEVER, THIS SOFTWARE CODIFIES THE FINALIST
-- CANDIDATE ALGORITHMS (i.e., MARS, RC6tm, RIJNDAEL, SERPENT, AND
-- TWOFISH) IN THE ADVANCED ENCRYPTION STANDARD (AES) DEVELOPMENT EFFORT
-- SPONSORED BY THE NATIONAL INSTITUTE OF STANDARDS AND TECHNOLOGY (NIST)
-- AND MAY BE PROTECTED BY ONE OR MORE FORMS OF INTELLECTUAL PROPERTY. THE
-- U.S. GOVERNMENT MAKES NO WARRANTY, EITHER EXPRESSED OR IMPLIED,
-- INCLUDING BUT NO LIMITED TO ANY IMPLIED WARRANTIES OF MERCHANTABILITY
-- OR FITNESS FOR A PARTICULAR PURPOSE, REGARDING THIS SOFTWARE. THE U.S.
-- GOVERNMENT FURTHER MAKES NO WARRANTY THAT THIS SOFTWARE WILL NOT
-- INFRINGE ANY OTHER UNITED STATES OR FOREIGN PATENT OR OTHER
-- INTELLECTUAL PROPERTY RIGHT. IN NO EVENT SHALL THE U.S. GOVERNMENT BE
-- LIABLE TO ANYONE FOR COMPENSATORY, PUNITIVE, EXEMPLARY, SPECIAL,
-- COLLATERAL, INCIDENTAL, CONSEQUENTIAL, OR ANY OTHER TYPE OF DAMAGES IN
-- CONNECTION WITH OR ARISING OUT OF COPY OR USE OF THIS SOFTWARE.
-- *************************************************************************

-- ===========================================================================
-- File Name : key_sched_iterative.vhdl
-- Author    : NSA
-- Date      : 04 September 1999
-- Project   : Serpent iterative key schedule block 
-- Purpose   : build key schedule for iterative implementation
-- Notes     :
-- ===========================================================================

library ieee;
use ieee.std_logic_1164.all;
use ieee.numeric_std.all;
use WORK.serpent_pack.all;

-- ===========================================================================
-- =========================== Interface Description =========================
-- ===========================================================================

entity KEY_SCHEDULE_ITERATIVE is

  port (clock        : in std_logic;    -- clock signal
        reset        : in std_logic;    -- active high reset (asynch)

        KS_LOADCV    : in std_logic;    -- load a new cryptovariable
        KS_START     : in std_logic;    -- start a new expansion sequence
        KS_CV        : in SLV_256;      -- cryptovariable input bus
        KS_ENC       : in std_logic;    -- encrypt select (1=enc, 0=dec)

        KS_READY     : out std_logic;   -- '1'=cv expanded, '0'=inprocess
        KS_ROUND_KEY : out SLV_128      -- output round key (Ki)

  );

end KEY_SCHEDULE_ITERATIVE;

architecture KEY_SCHEDULE_ITERATIVE_RTL of KEY_SCHEDULE_ITERATIVE is


component RUNUP_ROUNDKEYS

  port (clock         : in std_logic;    -- clock signal
        reset         : in std_logic;    -- active high reset (asynchronous)
    
        KS_LOADCV     : in std_logic;    -- load a new cryptovariable
        KS_CV         : in SLV_256;      -- cryptovariable input bus
        KS_ENC        : in std_logic;    -- encrypt/decrypt select

        KS_READY      : out std_logic;   -- '1'=cv expanded, '0'=inprocess
        KS_W_INITIAL  : out W_TYPE       -- output key bus

  );

end component;

-- ===========================================================================
-- =========================== Type Definition ===============================
-- ===========================================================================

type EXPAND_STATE_TYPE is ( HOLD, CV_EXPAND );

-- ===========================================================================
-- =========================== Signal Definition =============================
-- ===========================================================================

signal W              : W_TYPE;              -- current w register expansion
signal KS_W_INITIAL   : W_TYPE;              -- newly generated W array
signal CV_EXPAND_STEP : INTEGER;             -- counter for cv expansion
signal EXPAND_STATE   : EXPAND_STATE_TYPE;

begin

-- ===========================================================================
-- =========================== Data Movement =================================
-- ===========================================================================


GEN_ROUNDKEY : process( clock, reset )

begin

   if reset = '1' then

      for INDEX in -8 to -1 loop            -- cycle through entire W array
         W(INDEX) <= ( others => '0' );     -- clear the W array of registers
      end loop;

      KS_ROUND_KEY <= ( others => '0' );    -- clear round key outputs

   elsif clock'event and clock = '1' then

      if KS_START = '1' then                -- check for a new expansion start
                                            -- Generate next round key, w
         KS_ROUND( KS_ENC,                  -- select encrypt(=1) or decrypt(=0)
                   KS_W_INITIAL,            -- input w register (from new cv)
                   STD_LOGIC_VECTOR(TO_UNSIGNED(CV_EXPAND_STEP, 6)),
                   STD_LOGIC_VECTOR(TO_UNSIGNED(CV_EXPAND_STEP mod 8, 6)),
                   W,                       -- updated w register
                   KS_ROUND_KEY );          -- output round key


      elsif EXPAND_STATE = CV_EXPAND then   -- check for expansion activity
                                            -- generate next round key, w
         KS_ROUND( KS_ENC,                  -- select encrypt(=1) or decrypt(=0)
                   W,                       -- input w register
                   STD_LOGIC_VECTOR(TO_UNSIGNED(CV_EXPAND_STEP, 6)),
                   STD_LOGIC_VECTOR(TO_UNSIGNED(CV_EXPAND_STEP mod 8, 6)),
                   W,                       -- updated w register
                   KS_ROUND_KEY );          -- output round key

      else

         KS_ROUND_KEY <= ( others => '0' ); -- not expanding, so zero roundkey

      end if; -- KS_START = '1'

   end if; -- reset = '1'

end process; -- GEN_ROUNDKEY


-- ===========================================================================
-- =========================== State Machine / Controller ====================
-- ===========================================================================


EXPAND_FLOW : process( clock, reset )

begin

   if reset = '1' then                      -- Active high reset (asynchronous)

      CV_EXPAND_STEP <= 0;                  -- put controller in hold state
      EXPAND_STATE   <= HOLD;               -- no activity for expansion

   elsif clock'event and clock = '1' then

      case EXPAND_STATE is

         when HOLD =>

            if KS_START = '1' then

               EXPAND_STATE   <= CV_EXPAND;
               CV_EXPAND_STEP <= 1;
      
            else

               EXPAND_STATE   <= HOLD;
               CV_EXPAND_STEP <= 0;
       
            end if; -- KS_START = '1'

         when CV_EXPAND =>

            if CV_EXPAND_STEP = LAST_ROUND+1 then    -- check for complete
 
               EXPAND_STATE   <= HOLD;    -- set controller back to hold state
               CV_EXPAND_STEP <= 0;

            else

               EXPAND_STATE   <= EXPAND_STATE;
               CV_EXPAND_STEP <= CV_EXPAND_STEP + 1;

            end if;

      end case;
                                             
   end if;  -- reset = '1'

end process; -- EXPAND_FLOW

GEN_NEXT_KEYS: RUNUP_ROUNDKEYS
                        port map( clock,
                                  reset,
                                  KS_LOADCV,
                                  KS_CV,
                                  KS_ENC,
                                  KS_READY,
                                  KS_W_INITIAL );

end KEY_SCHEDULE_ITERATIVE_RTL;


-- configuration conf_KEY_SCHEDULE_ITERATIVE of KEY_SCHEDULE_ITERATIVE is

--   for KEY_SCHEDULE_ITERATIVE_RTL

--      for GEN_NEXT_KEYS: RUNUP_ROUNDKEYS use
--         entity work.RUNUP_ROUNDKEYS(RUNUP_ROUNDKEYS_RTL);
--      end for;

--   end for;

-- end conf_KEY_SCHEDULE_ITERATIVE;
