-- *************************************************************************
-- DISCLAIMER. THIS SOFTWARE WAS WRITTEN BY EMPLOYEES OF THE U.S.
-- GOVERNMENT AS A PART OF THEIR OFFICIAL DUTIES AND, THEREFORE, IS NOT
-- PROTECTED BY COPYRIGHT. HOWEVER, THIS SOFTWARE CODIFIES THE FINALIST
-- CANDIDATE ALGORITHMS (i.e., MARS, RC6tm, RIJNDAEL, SERPENT, AND
-- TWOFISH) IN THE ADVANCED ENCRYPTION STANDARD (AES) DEVELOPMENT EFFORT
-- SPONSORED BY THE NATIONAL INSTITUTE OF STANDARDS AND TECHNOLOGY (NIST)
-- AND MAY BE PROTECTED BY ONE OR MORE FORMS OF INTELLECTUAL PROPERTY. THE
-- U.S. GOVERNMENT MAKES NO WARRANTY, EITHER EXPRESSED OR IMPLIED,
-- INCLUDING BUT NO LIMITED TO ANY IMPLIED WARRANTIES OF MERCHANTABILITY
-- OR FITNESS FOR A PARTICULAR PURPOSE, REGARDING THIS SOFTWARE. THE U.S.
-- GOVERNMENT FURTHER MAKES NO WARRANTY THAT THIS SOFTWARE WILL NOT
-- INFRINGE ANY OTHER UNITED STATES OR FOREIGN PATENT OR OTHER
-- INTELLECTUAL PROPERTY RIGHT. IN NO EVENT SHALL THE U.S. GOVERNMENT BE
-- LIABLE TO ANYONE FOR COMPENSATORY, PUNITIVE, EXEMPLARY, SPECIAL,
-- COLLATERAL, INCIDENTAL, CONSEQUENTIAL, OR ANY OTHER TYPE OF DAMAGES IN
-- CONNECTION WITH OR ARISING OUT OF COPY OR USE OF THIS SOFTWARE.
-- *************************************************************************

-- ===========================================================================
-- File Name : runup_roundkeys.vhdl
-- Author    : NSA
-- Date      : October 1999
-- Project   : AES Candidate Evaluation
-- Purpose   : This block performs the initial key runup required prior
--             to decryption for the SERPENT algorithm. The key schedule
--             is expanded without extracting subkeys. Then, the
--             schedule is run in reverse such that the subkeys are
--             made available to the algorithm in the correct order.
-- Notes     : 
-- ===========================================================================

library ieee;
use ieee.std_logic_1164.all;
use ieee.numeric_std.all;
use WORK.serpent_pack.all;

-- ===========================================================================
-- =========================== Interface Description =========================
-- ===========================================================================

entity RUNUP_ROUNDKEYS is

  port (clock         :  in std_logic;    -- clock signal
        reset         :  in std_logic;    -- active high reset (asynchronous)
    
        KS_LOADCV     :  in std_logic;    -- load a new cryptovariable
        KS_CV         :  in SLV_256;      -- cryptovariable input bus
        KS_ENC        :  in std_logic;    -- encrypt/decrypt select

        KS_READY      :  out std_logic;   -- '1'=cv expanded, '0'=inprocess
        KS_W_INITIAL  :  out W_TYPE       -- output key bus

  );

end RUNUP_ROUNDKEYS;

architecture RUNUP_ROUNDKEYS_RTL of RUNUP_ROUNDKEYS is


type RUNUP_STATE_TYPE is  ( HOLD, CV_RUNUP, DONE );


-- ===========================================================================
-- =========================== Signal Definition =============================
-- ===========================================================================

signal CV_RUNUP_STEP    : integer range 0 to 255; -- counter for cv runup
signal RUNUP_STATE      : RUNUP_STATE_TYPE;
signal w                : W_TYPE;
signal KS_W_INITIAL_INT : W_TYPE;

begin

-- ===========================================================================
-- =========================== Data Movement =================================
-- ===========================================================================

KS_READY     <= '1' when RUNUP_STATE = DONE else '0';
KS_W_INITIAL <= KS_W_INITIAL_INT;

DATA_FLOW : process( clock, reset )

variable w_tmp : W_TYPE;

begin

  if reset = '1' then

    for w_index in 0 to 7 loop
      w(-1-w_index) <= KS_CV( (w_index)*32 to (((w_index+1)*32)-1) );
    end loop;

    for w_index in -8 to -1 loop
      KS_W_INITIAL_INT(w_index) <= ( others => '0' );
    end loop;

  elsif clock'event and clock = '1' then

    case RUNUP_STATE is

      when HOLD => 

        if KS_ENC = '1' then

          for w_index in 0 to 7 loop
            KS_W_INITIAL_INT(-1-w_index) <=
               KS_CV((w_index)*32 to (((w_index+1)*32)-1) );
          end loop;

        else

          for w_index in -8 to -1 loop
            KS_W_INITIAL_INT(w_index) <= ( others => '0' );
          end loop;

        end if; -- KS_ENC = '1'

        if KS_LOADCV = '1' then

          for w_index in 0 to 7 loop
            w_tmp(-1-w_index) :=
                 KS_CV( (w_index)*32 to (((w_index+1)*32)-1) );
          end loop;

          w <= UPDATE_W_ENCRYPT("000000", w_tmp);

        else

          w <= w;

        end if; -- KS_LOADCV = '1'


      when CV_RUNUP => 

        if CV_RUNUP_STEP = LAST_DCVRUNUP_STEP_I then

          KS_W_INITIAL_INT <= UPDATE_W_ENCRYPT(STD_LOGIC_VECTOR(TO_UNSIGNED(
                              CV_RUNUP_STEP, 6)), w );

          for w_index in -8 to -1 loop
            w(w_index) <= w(w_index);
          end loop;

        else

          for w_index in -8 to -1 loop
            KS_W_INITIAL_INT(w_index) <= ( others => '0' );
          end loop;

          w <= UPDATE_W_ENCRYPT(STD_LOGIC_VECTOR(TO_UNSIGNED(
               CV_RUNUP_STEP, 6)), w ); 

        end if;


      when DONE => 

        if KS_ENC = '1' then

          for w_index in 0 to 7 loop
            KS_W_INITIAL_INT(-1-w_index) <= 
               KS_CV( (w_index)*32 to (((w_index+1)*32)-1) );
          end loop;

        else

          for w_index in -8 to -1 loop
            KS_W_INITIAL_INT(w_index) <= KS_W_INITIAL_INT(w_index);
          end loop;

        end if; -- KS_ENC = '1'

        if KS_LOADCV = '1' then

          for w_index in 0 to 7 loop
            w_tmp(-1-w_index) :=
                 KS_CV( (w_index)*32 to (((w_index+1)*32)-1) );
          end loop;

          w <= UPDATE_W_ENCRYPT("000000", w_tmp);

        else

          w <= w;

        end if; -- KS_LOADCV = '1'


      when others => 

          w <= w;

    end case;

  end if;

end process; -- DATA_FLOW

-- ===========================================================================
-- =========================== State Machine / Controller ====================
-- ===========================================================================

RUNUP_FLOW: process( clock, reset )

begin

   if reset = '1' then                     -- Active high reset (asynchronous)

      CV_RUNUP_STEP <= 1;                  -- put runup controller in hold
      RUNUP_STATE   <= HOLD;

   elsif clock'event and clock = '1' then

      case RUNUP_STATE is

         when HOLD =>
 
            if KS_LOADCV = '1' then

               if KS_ENC = '0' then
                  RUNUP_STATE <= CV_RUNUP;
               else
                  RUNUP_STATE <= DONE;
               end if;

               CV_RUNUP_STEP <= 1;
            else

               RUNUP_STATE <= HOLD;
               CV_RUNUP_STEP <= CV_RUNUP_STEP;

            end if;

         when CV_RUNUP =>

            if ( CV_RUNUP_STEP /= LAST_ECVRUNUP_STEP and KS_ENC = '1') or
               ( CV_RUNUP_STEP /= LAST_DCVRUNUP_STEP_I and KS_ENC = '0' ) then

               CV_RUNUP_STEP <= CV_RUNUP_STEP + 1;     -- increment counter
               RUNUP_STATE   <= RUNUP_STATE;

            else                              -- disable cvrunup controller

               CV_RUNUP_STEP <= 1;
               RUNUP_STATE   <= DONE;

            end if;

         when DONE =>

            if KS_LOADCV = '1' then

              if KS_ENC = '0' then
                RUNUP_STATE <= CV_RUNUP;
              else
                RUNUP_STATE <= DONE;
              end if;

            else

               RUNUP_STATE   <= DONE;

            end if;

            CV_RUNUP_STEP <= 1;  --mob

      end case;
                                             
   end if;  -- reset = '1'

end process; -- RUNUP_FLOW


end RUNUP_ROUNDKEYS_RTL;
