-- *************************************************************************
-- DISCLAIMER. THIS SOFTWARE WAS WRITTEN BY EMPLOYEES OF THE U.S.
-- GOVERNMENT AS A PART OF THEIR OFFICIAL DUTIES AND, THEREFORE, IS NOT
-- PROTECTED BY COPYRIGHT. HOWEVER, THIS SOFTWARE CODIFIES THE FINALIST
-- CANDIDATE ALGORITHMS (i.e., MARS, RC6tm, RIJNDAEL, SERPENT, AND
-- TWOFISH) IN THE ADVANCED ENCRYPTION STANDARD (AES) DEVELOPMENT EFFORT
-- SPONSORED BY THE NATIONAL INSTITUTE OF STANDARDS AND TECHNOLOGY (NIST)
-- AND MAY BE PROTECTED BY ONE OR MORE FORMS OF INTELLECTUAL PROPERTY. THE
-- U.S. GOVERNMENT MAKES NO WARRANTY, EITHER EXPRESSED OR IMPLIED,
-- INCLUDING BUT NO LIMITED TO ANY IMPLIED WARRANTIES OF MERCHANTABILITY
-- OR FITNESS FOR A PARTICULAR PURPOSE, REGARDING THIS SOFTWARE. THE U.S.
-- GOVERNMENT FURTHER MAKES NO WARRANTY THAT THIS SOFTWARE WILL NOT
-- INFRINGE ANY OTHER UNITED STATES OR FOREIGN PATENT OR OTHER
-- INTELLECTUAL PROPERTY RIGHT. IN NO EVENT SHALL THE U.S. GOVERNMENT BE
-- LIABLE TO ANYONE FOR COMPENSATORY, PUNITIVE, EXEMPLARY, SPECIAL,
-- COLLATERAL, INCIDENTAL, CONSEQUENTIAL, OR ANY OTHER TYPE OF DAMAGES IN
-- CONNECTION WITH OR ARISING OUT OF COPY OR USE OF THIS SOFTWARE.
-- *************************************************************************
-- ===========================================================================
-- ===========================================================================
-- File Name : key_sched_pipe.vhdl
-- Author    : NSA
-- Date      : Dec 1999
-- Project   : AES Candidate Evaluation - MARS
-- Purpose   : build key schedule for pipelined implementation
-- Notes     :
-- ===========================================================================

library ieee;
use ieee.std_logic_1164.all;
use ieee.numeric_std.all;
use WORK.mars_pack.all;

-- ===========================================================================
-- =========================== Interface Description =========================
-- ===========================================================================

entity KEY_SCHEDULE_PIPE is

  port (clock            :  in std_logic;     -- clock signal
        reset            :  in std_logic;     -- active high reset (asynch)

        KS_LOADCV        :  in std_logic;     -- load a new cryptovariable
        KS_CV            :  in CV_ARRAY_TYPE;  -- cryptovariable input bus
        KS_CVSIZE        :  in SLV_2; 
        KS_ENC           :  in std_logic;

        KS_ROUND_KEY_ENC :  out K_ARRAY_TYPE; -- encrypt round key output
        KS_ROUND_KEY_DEC :  out K_ARRAY_TYPE  -- decrypt round key output

  );

end KEY_SCHEDULE_PIPE;

architecture KEY_SCHEDULE_PIPE_RTL of KEY_SCHEDULE_PIPE is


-- ===========================================================================
-- =========================== Component Definition ==========================
-- ===========================================================================

component PARTIAL_KREG 
  port (clock    :  in std_logic;   -- clock signal
        reset    :  in std_logic;   -- active high reset (asynchronous)
    
        LATCH    :  in std_logic;
        DATA_IN  :  in K_PARTIAL_TYPE;      -- input data bus

        DATA_OUT :  out K_PARTIAL_TYPE      -- output data bus
  );
end component;


component PARTIAL_TREG is
  port (clock    :  in std_logic;   -- clock signal
        reset    :  in std_logic;   -- active high reset (asynchronous)
    
        LATCH    :  in std_logic;
        DATA_IN  :  in T_ARRAY_TYPE;      -- input data bus

        DATA_OUT :  out T_ARRAY_TYPE      -- output data bus
  );
end component;
 
component CREATE_T_BLOCK is
  port (T         : T_ARRAY_TYPE;
        j            : SLV_2;
        signal T_OUT : out T_ARRAY_TYPE 
    );
end component;



-- ===========================================================================
-- =========================== Signal Definition =============================
-- ===========================================================================

type T_TYPE4   is array(0 to 3) of T_ARRAY_TYPE;
type T_TYPE3   is array(0 to 3) of T_ARRAY_TYPE;
type KEY_TYPE  is array ( 0 to 3 ) of K_PARTIAL_TYPE;
type KEY_ROW_TYPE is array ( 0 to ((LAST_ROUND+1)/RUNUP_CYCLES)+1) of K_PARTIAL_TYPE;
type KEY_MATRIX_TYPE is array (0 to 3) of KEY_ROW_TYPE;

type LATCH_ARRAY_TYPE is array (0 to (LAST_ROUND/RUNUP_CYCLES)+1) of SLV_4;

type ROUND_VEC_TYPE is array (0 to 3) of SLV_2;
constant ROUND_VEC : ROUND_VEC_TYPE :=
( "00" ,
 "01" ,
 "10" ,
 "11" );

constant ROUND_VEC_ADD : ROUND_VEC_TYPE :=
( "01" ,
  "10" ,
  "11",
  "00" );

signal ROUND_VEC_SIG   : ROUND_VEC_TYPE := ROUND_VEC;
signal ROUND_VEC_SIG_ADD   : ROUND_VEC_TYPE := ROUND_VEC_ADD;

signal T_START         : T_ARRAY_TYPE;
signal DEC_T           : T_TYPE4;
signal ENC_T           : T_TYPE4;
signal MAKE_SUBKEYS_IN : T_TYPE4;
signal KEY_UNFIXED     : KEY_TYPE;
signal KEY_FIXED       : KEY_TYPE;
signal T_REG           : T_TYPE3;
signal SUBKEY          : KEY_MATRIX_TYPE;
signal LATCH_REG       : LATCH_TYPE;
signal LATCH           : std_logic;
signal CV_RUNUP_STEP   : ROUND_TYPE;     -- cryptovariable runup counter
signal CV_RUNUP_ACTIVE : std_logic;      -- active = '1', hold = '0'
signal LATCH_ARRAY     : LATCH_ARRAY_TYPE; 

type WEAKKEY_CTRL_TYPE is array(0 to 3) of SLV_2;
signal WEAKKEY_CTRL  : WEAKKEY_CTRL_TYPE;

-- ===========================================================================
-- ====================== Generate Pipe Structure ============================
-- ===========================================================================
--

begin

  -- Create starting point from CV
INIT_T( KS_CV,  T_START );

  -- Create full T expansion (used for decrypt mode)
 inst1 : CREATE_T_BLOCK  port map (T_START ,
                  ROUND_VEC_SIG(0),
                 DEC_T(0) );


  ENC_T(0) <= DEC_T(0);
 inst2 : CREATE_T_BLOCK  port map (DEC_T(0) ,  
               ROUND_VEC_SIG(1),
              DEC_T(1) );



 inst3 : CREATE_T_BLOCK  port map (DEC_T(1) ,  
     ROUND_VEC_SIG(2),
 DEC_T(2) );


 inst4 : CREATE_T_BLOCK  port map (DEC_T(2),
     ROUND_VEC_SIG(3),
    DEC_T(3));



  -- Create first 3 rows (including T storage for next row)
  G1: for row in 0 to 2 generate
    MAKE_SUBKEYS_IN(row) <= ENC_T(row) when KS_ENC = '1' else
                            DEC_T(3 - row);
    MAKE_SUBKEYS( MAKE_SUBKEYS_IN(row),
                  KS_ENC,
                  KEY_UNFIXED(row) );

  WEAKKEY_CTRL(row) <= std_logic_vector(TO_UNSIGNED(row,2)) when KS_ENC = '1' else
                  std_logic_vector(TO_UNSIGNED(3-row,2));

    FIX_WEAKKEYS( KEY_UNFIXED(row),
                  WEAKKEY_CTRL(row),
                  KEY_FIXED(row) );

    TR: PARTIAL_TREG port map ( clock,
                                 reset,
                                 LATCH_ARRAY(0)(row),
                                 MAKE_SUBKEYS_IN(row),
                                 T_REG(row) );
    KR: PARTIAL_KREG port map ( clock,
                                 reset,
                                 LATCH_ARRAY(0)(row),
                                 KEY_FIXED(row),
                                 SUBKEY(row)(0) );



 inst5 : CREATE_T_BLOCK  port map ( T_REG(row), 
    ROUND_VEC_SIG_ADD(row),
   ENC_T(row+1) );



  end generate;  -- G1

  -- Create last row (no T storage for next row)

  MAKE_SUBKEYS_IN(3) <= ENC_T(3) when KS_ENC = '1' else
                        DEC_T(0);
  MAKE_SUBKEYS( MAKE_SUBKEYS_IN(3),
                KS_ENC,
                KEY_UNFIXED(3) );
  WEAKKEY_CTRL(3) <= std_logic_vector(TO_UNSIGNED(3,2)) when KS_ENC = '1' else
                     std_logic_vector(TO_UNSIGNED(0,2));

  FIX_WEAKKEYS( KEY_UNFIXED(3),
                WEAKKEY_CTRL(3),
                KEY_FIXED(3) );

  KR3: PARTIAL_KREG port map ( clock,
                               reset,
                               LATCH_ARRAY(0)(3),
                               KEY_FIXED(3),
                               SUBKEY(3)(0) );
----------------------------------------------------------------------

  -- Build out first 10 subkeys (row 1)
  G2: for col in 0 to 11/RUNUP_CYCLES generate
    KR_G2: PARTIAL_KREG port map ( clock,
                               reset,
                               LATCH_ARRAY(col+1)(0),
                               SUBKEY(0)(col),
                               SUBKEY(0)(col+1) );
  end generate;  -- G2

  -- Build out next 10 subkeys (row 2)
  G3: for col in 0 to 16/RUNUP_CYCLES generate
    KR_G3: PARTIAL_KREG port map ( clock,
                               reset,
                               LATCH_ARRAY(col+1)(1),
                               SUBKEY(1)(col),
                               SUBKEY(1)(col+1) );
  end generate;  -- G3

  -- Build out next 10 subkeys (row 3)
  G4: for col in 0 to 21/RUNUP_CYCLES generate
    KR_G4: PARTIAL_KREG port map ( clock,
                               reset,
                               LATCH_ARRAY(col+1)(2),
                               SUBKEY(2)(col),
                               SUBKEY(2)(col+1) );
  end generate;  -- G4

  -- Build out next 10 subkeys (row 4)
  G5: for col in 0 to (LAST_ROUND/RUNUP_CYCLES) generate
    KR_G5: PARTIAL_KREG port map ( clock,
                               reset,
                               LATCH_ARRAY(col+1)(3),
                               SUBKEY(3)(col),
                               SUBKEY(3)(col+1) );
  end generate;  -- G5



  -- Create latch signal for all registers
  G12: for row in 0 to 3 generate
    G12a: for col in 0 to (LAST_ROUND/RUNUP_CYCLES) generate
      LATCH_ARRAY(col)(row) <= LATCH_REG((row+col+1) * RUNUP_CYCLES - 2)
                               when KS_ENC = '1' else
                               LATCH_REG(4*(col+1) * RUNUP_CYCLES - 2)
                               when col <= LAST_ROUND/(4*RUNUP_CYCLES) else
                               LATCH_REG(LATCH_REG'high);
    end generate;
  end generate;


  -- Assign final outputs - Row 1a
  G11: for j in 0 to 3 generate
      KS_ROUND_KEY_ENC(j)  <= SUBKEY(0)(0)(j);
      KS_ROUND_KEY_DEC(j)  <= SUBKEY(0)(0)(9-j);
  end generate;  -- G11

  -- Assign final outputs - Row 1b
  G6: for round in 9 to 11 generate
      KS_ROUND_KEY_ENC(2*round-14)  <= SUBKEY(0)(round/RUNUP_CYCLES)(2*round-14);
      KS_ROUND_KEY_ENC(2*round-13)  <= SUBKEY(0)(round/RUNUP_CYCLES)(2*round-13);
      KS_ROUND_KEY_DEC(2*round-14)  <= SUBKEY(0)(round/(4*RUNUP_CYCLES))(9-(2*round-14));
      KS_ROUND_KEY_DEC(2*round-13)  <= SUBKEY(0)(round/(4*RUNUP_CYCLES))(9-(2*round-13));
  end generate;  -- G6

  -- Assign final outputs - Row 2
  G7: for round in 12 to 16 generate
      KS_ROUND_KEY_ENC(2*round-14)  <= SUBKEY(1)(round/RUNUP_CYCLES-1)(2*round-24);
      KS_ROUND_KEY_ENC(2*round-13)  <= SUBKEY(1)(round/RUNUP_CYCLES-1)(2*round-23);
      KS_ROUND_KEY_DEC(2*round-14)  <= SUBKEY(1)(round/(4*RUNUP_CYCLES))(9-(2*round-24));
      KS_ROUND_KEY_DEC(2*round-13)  <= SUBKEY(1)(round/(4*RUNUP_CYCLES))(9-(2*round-23));
  end generate;  -- G7

  -- Assign final outputs - Row 3
  G8: for round in 17 to 21 generate
      KS_ROUND_KEY_ENC(2*round-14)  <= SUBKEY(2)(round/RUNUP_CYCLES-2)(2*round-34);
      KS_ROUND_KEY_ENC(2*round-13)  <= SUBKEY(2)(round/RUNUP_CYCLES-2)(2*round-33);
      KS_ROUND_KEY_DEC(2*round-14)  <= SUBKEY(2)(round/(4*RUNUP_CYCLES))(9-(2*round-34));
      KS_ROUND_KEY_DEC(2*round-13)  <= SUBKEY(2)(round/(4*RUNUP_CYCLES))(9-(2*round-33));
  end generate;  -- G8

  -- Assign final outputs - Row 4a
  G9: for round in 22 to 24 generate
      KS_ROUND_KEY_ENC(2*round-14)  <= SUBKEY(3)((round/RUNUP_CYCLES)-3)(2*round-44);
      KS_ROUND_KEY_ENC(2*round-13)  <= SUBKEY(3)((round/RUNUP_CYCLES)-3)(2*round-43);
      KS_ROUND_KEY_DEC(2*round-14)  <= SUBKEY(3)((round/(4*RUNUP_CYCLES)))(9-(2*round-44));
      KS_ROUND_KEY_DEC(2*round-13)  <= SUBKEY(3)((round/(4*RUNUP_CYCLES)))(9-(2*round-43));
  end generate;  -- G9

  -- Assign final outputs - Row 4b
  G10: for j in 36 to 39 generate
      KS_ROUND_KEY_ENC(j)  <= SUBKEY(3)(33/RUNUP_CYCLES-3)(j-30);
      KS_ROUND_KEY_DEC(j)  <= SUBKEY(3)(33/(4*RUNUP_CYCLES))(9-(j-30));
  end generate;  -- G10






-- ===========================================================================
-- =========================== Data Movement =================================
-- ===========================================================================


DATA_FLOW : process( clock, reset )

begin

if reset = '1' then

   for PIPE_STEP in 0 to (LAST_ROUND + 5*RUNUP_CYCLES) loop
      LATCH_REG(PIPE_STEP) <= '0';
   end loop;

elsif clock'event and clock = '1' then

   if KS_LOADCV = '1' then
      LATCH_REG(0)     <= '1';
   else
      LATCH_REG(0)     <= '0';
   end if; -- KS_LOADCV = '1'

   LATCH_REG(1 to LATCH_REG'high) <= 
             LATCH_REG(0 to LATCH_REG'high - 1);

end if; -- reset = '1'

end process; -- DATA_FLOW


-- ===========================================================================
-- =========================== State Machine / Controller ====================
-- ===========================================================================

end KEY_SCHEDULE_PIPE_RTL;

-- ===========================================================================
-- ========================= Configuration ===================================
-- ===========================================================================

configuration CFG_KEY_SCHEDULE_PIPE of KEY_SCHEDULE_PIPE is

   for KEY_SCHEDULE_PIPE_RTL

   end for;

end;
