-- *************************************************************************
-- DISCLAIMER. THIS SOFTWARE WAS WRITTEN BY EMPLOYEES OF THE U.S.
-- GOVERNMENT AS A PART OF THEIR OFFICIAL DUTIES AND, THEREFORE, IS NOT
-- PROTECTED BY COPYRIGHT. HOWEVER, THIS SOFTWARE CODIFIES THE FINALIST
-- CANDIDATE ALGORITHMS (i.e., MARS, RC6tm, RIJNDAEL, SERPENT, AND
-- TWOFISH) IN THE ADVANCED ENCRYPTION STANDARD (AES) DEVELOPMENT EFFORT
-- SPONSORED BY THE NATIONAL INSTITUTE OF STANDARDS AND TECHNOLOGY (NIST)
-- AND MAY BE PROTECTED BY ONE OR MORE FORMS OF INTELLECTUAL PROPERTY. THE
-- U.S. GOVERNMENT MAKES NO WARRANTY, EITHER EXPRESSED OR IMPLIED,
-- INCLUDING BUT NO LIMITED TO ANY IMPLIED WARRANTIES OF MERCHANTABILITY
-- OR FITNESS FOR A PARTICULAR PURPOSE, REGARDING THIS SOFTWARE. THE U.S.
-- GOVERNMENT FURTHER MAKES NO WARRANTY THAT THIS SOFTWARE WILL NOT
-- INFRINGE ANY OTHER UNITED STATES OR FOREIGN PATENT OR OTHER
-- INTELLECTUAL PROPERTY RIGHT. IN NO EVENT SHALL THE U.S. GOVERNMENT BE
-- LIABLE TO ANYONE FOR COMPENSATORY, PUNITIVE, EXEMPLARY, SPECIAL,
-- COLLATERAL, INCIDENTAL, CONSEQUENTIAL, OR ANY OTHER TYPE OF DAMAGES IN
-- CONNECTION WITH OR ARISING OUT OF COPY OR USE OF THIS SOFTWARE.
-- *************************************************************************
-- ===========================================================================
-- ===========================================================================
-- File Name : mars_pkg.vhdl
-- Author    : NSA
-- Date      : October 99
-- Project   : AES Candidate Evaluation - MARS
-- Purpose   : This package defines common types, subtypes, constants,
--             and functions required to implement various VHDL models
--             for the creation of ASIC simulation of MARS, an Advanced
--             Encryption Standard (AES) candidate algorithm.
-- Notes     :
-- ===========================================================================

library ieee;
use ieee.std_logic_1164.all;
use ieee.numeric_std.all;

package mars_pack is

-- ==========================================================================
-- ======= Type, sub-type and function declarations for general use =========
-- ==========================================================================
type STATE_TYPE         is (nop, wait4ks, start, busy, queue);

subtype ROUND_TYPE      is integer range 0 to 63;


subtype SLV_2           is std_logic_vector(1 downto 0);
subtype SLV_4           is std_logic_vector(3 downto 0);
subtype SLV_5           is std_logic_vector(4 downto 0);
subtype SLV_6           is std_logic_vector(5 downto 0);
subtype SLV_8           is std_logic_vector(7 downto 0);
subtype SLV_9           is std_logic_vector(8 downto 0);
subtype SLV_10          is std_logic_vector(9 downto 0);
subtype SLV_12          is std_logic_vector(11 downto 0);
subtype SLV_32          is std_logic_vector(31 downto 0);
subtype SLV_34          is std_logic_vector(33 downto 0);
subtype SLV_64          is std_logic_vector(63 downto 0);
subtype SLV_96          is std_logic_vector(95 downto 0);
subtype SLV_128         is std_logic_vector(127 downto 0);
subtype SLV_256         is std_logic_vector(255 downto 0);
subtype U_32            is unsigned(31 downto 0);

type DATA_TYPE          is ARRAY(3 downto 0) of SLV_32;

constant FIRST_ROUND : ROUND_TYPE := 0;
constant LAST_ROUND  : ROUND_TYPE := 33;


procedure BYTESWAP32( datain  : in SLV_32;
               signal dataout : out SLV_32 );

function BYTESWAP32_FUNCT( datain  : in SLV_32)
                return SLV_32;

function BYTESWAPU32_FUNCT( datain  : in U_32)
                return U_32;



type SBOX_TYPE       is array (0 to 511) of SLV_32;
type ARRAY_TYPE_128  is array (3 downto 0) of U_32;
type ARRAY_TYPE_32   is array (3 downto 0) of SLV_8;

constant S : SBOX_TYPE := (
  X"09d0c479", X"28c8ffe0", X"84aa6c39", X"9dad7287", 
  X"7dff9be3", X"d4268361", X"c96da1d4", X"7974cc93", 
  X"85d0582e", X"2a4b5705", X"1ca16a62", X"c3bd279d", 
  X"0f1f25e5", X"5160372f", X"c695c1fb", X"4d7ff1e4", 
  X"ae5f6bf4", X"0d72ee46", X"ff23de8a", X"b1cf8e83", 
  X"f14902e2", X"3e981e42", X"8bf53eb6", X"7f4bf8ac", 
  X"83631f83", X"25970205", X"76afe784", X"3a7931d4", 
  X"4f846450", X"5c64c3f6", X"210a5f18", X"c6986a26", 
  X"28f4e826", X"3a60a81c", X"d340a664", X"7ea820c4", 
  X"526687c5", X"7eddd12b", X"32a11d1d", X"9c9ef086", 
  X"80f6e831", X"ab6f04ad", X"56fb9b53", X"8b2e095c", 
  X"b68556ae", X"d2250b0d", X"294a7721", X"e21fb253", 
  X"ae136749", X"e82aae86", X"93365104", X"99404a66", 
  X"78a784dc", X"b69ba84b", X"04046793", X"23db5c1e", 
  X"46cae1d6", X"2fe28134", X"5a223942", X"1863cd5b", 
  X"c190c6e3", X"07dfb846", X"6eb88816", X"2d0dcc4a", 
  X"a4ccae59", X"3798670d", X"cbfa9493", X"4f481d45", 
  X"eafc8ca8", X"db1129d6", X"b0449e20", X"0f5407fb", 
  X"6167d9a8", X"d1f45763", X"4daa96c3", X"3bec5958", 
  X"ababa014", X"b6ccd201", X"38d6279f", X"02682215", 
  X"8f376cd5", X"092c237e", X"bfc56593", X"32889d2c", 
  X"854b3e95", X"05bb9b43", X"7dcd5dcd", X"a02e926c", 
  X"fae527e5", X"36a1c330", X"3412e1ae", X"f257f462", 
  X"3c4f1d71", X"30a2e809", X"68e5f551", X"9c61ba44", 
  X"5ded0ab8", X"75ce09c8", X"9654f93e", X"698c0cca", 
  X"243cb3e4", X"2b062b97", X"0f3b8d9e", X"00e050df", 
  X"fc5d6166", X"e35f9288", X"c079550d", X"0591aee8", 
  X"8e531e74", X"75fe3578", X"2f6d829a", X"f60b21ae", 
  X"95e8eb8d", X"6699486b", X"901d7d9b", X"fd6d6e31", 
  X"1090acef", X"e0670dd8", X"dab2e692", X"cd6d4365", 
  X"e5393514", X"3af345f0", X"6241fc4d", X"460da3a3", 
  X"7bcf3729", X"8bf1d1e0", X"14aac070", X"1587ed55", 
  X"3afd7d3e", X"d2f29e01", X"29a9d1f6", X"efb10c53", 
  X"cf3b870f", X"b414935c", X"664465ed", X"024acac7", 
  X"59a744c1", X"1d2936a7", X"dc580aa6", X"cf574ca8", 
  X"040a7a10", X"6cd81807", X"8a98be4c", X"accea063", 
  X"c33e92b5", X"d1e0e03d", X"b322517e", X"2092bd13", 
  X"386b2c4a", X"52e8dd58", X"58656dfb", X"50820371", 
  X"41811896", X"e337ef7e", X"d39fb119", X"c97f0df6", 
  X"68fea01b", X"a150a6e5", X"55258962", X"eb6ff41b", 
  X"d7c9cd7a", X"a619cd9e", X"bcf09576", X"2672c073", 
  X"f003fb3c", X"4ab7a50b", X"1484126a", X"487ba9b1", 
  X"a64fc9c6", X"f6957d49", X"38b06a75", X"dd805fcd", 
  X"63d094cf", X"f51c999e", X"1aa4d343", X"b8495294", 
  X"ce9f8e99", X"bffcd770", X"c7c275cc", X"378453a7", 
  X"7b21be33", X"397f41bd", X"4e94d131", X"92cc1f98", 
  X"5915ea51", X"99f861b7", X"c9980a88", X"1d74fd5f", 
  X"b0a495f8", X"614deed0", X"b5778eea", X"5941792d", 
  X"fa90c1f8", X"33f824b4", X"c4965372", X"3ff6d550", 
  X"4ca5fec0", X"8630e964", X"5b3fbbd6", X"7da26a48", 
  X"b203231a", X"04297514", X"2d639306", X"2eb13149", 
  X"16a45272", X"532459a0", X"8e5f4872", X"f966c7d9", 
  X"07128dc0", X"0d44db62", X"afc8d52d", X"06316131", 
  X"d838e7ce", X"1bc41d00", X"3a2e8c0f", X"ea83837e", 
  X"b984737d", X"13ba4891", X"c4f8b949", X"a6d6acb3", 
  X"a215cdce", X"8359838b", X"6bd1aa31", X"f579dd52", 
  X"21b93f93", X"f5176781", X"187dfdde", X"e94aeb76", 
  X"2b38fd54", X"431de1da", X"ab394825", X"9ad3048f", 
  X"dfea32aa", X"659473e3", X"623f7863", X"f3346c59", 
  X"ab3ab685", X"3346a90b", X"6b56443e", X"c6de01f8", 
  X"8d421fc0", X"9b0ed10c", X"88f1a1e9", X"54c1f029", 
  X"7dead57b", X"8d7ba426", X"4cf5178a", X"551a7cca", 
  X"1a9a5f08", X"fcd651b9", X"25605182", X"e11fc6c3", 
  X"b6fd9676", X"337b3027", X"b7c8eb14", X"9e5fd030", 
  X"6b57e354", X"ad913cf7", X"7e16688d", X"58872a69", 
  X"2c2fc7df", X"e389ccc6", X"30738df1", X"0824a734", 
  X"e1797a8b", X"a4a8d57b", X"5b5d193b", X"c8a8309b", 
  X"73f9a978", X"73398d32", X"0f59573e", X"e9df2b03", 
  X"e8a5b6c8", X"848d0704", X"98df93c2", X"720a1dc3", 
  X"684f259a", X"943ba848", X"a6370152", X"863b5ea3", 
  X"d17b978b", X"6d9b58ef", X"0a700dd4", X"a73d36bf", 
  X"8e6a0829", X"8695bc14", X"e35b3447", X"933ac568", 
  X"8894b022", X"2f511c27", X"ddfbcc3c", X"006662b6", 
  X"117c83fe", X"4e12b414", X"c2bca766", X"3a2fec10", 
  X"f4562420", X"55792e2a", X"46f5d857", X"ceda25ce", 
  X"c3601d3b", X"6c00ab46", X"efac9c28", X"b3c35047", 
  X"611dfee3", X"257c3207", X"fdd58482", X"3b14d84f", 
  X"23becb64", X"a075f3a3", X"088f8ead", X"07adf158", 
  X"7796943c", X"facabf3d", X"c09730cd", X"f7679969", 
  X"da44e9ed", X"2c854c12", X"35935fa3", X"2f057d9f", 
  X"690624f8", X"1cb0bafd", X"7b0dbdc6", X"810f23bb", 
  X"fa929a1a", X"6d969a17", X"6742979b", X"74ac7d05", 
  X"010e65c4", X"86a3d963", X"f907b5a0", X"d0042bd3", 
  X"158d7d03", X"287a8255", X"bba8366f", X"096edc33", 
  X"21916a7b", X"77b56b86", X"951622f9", X"a6c5e650", 
  X"8cea17d1", X"cd8c62bc", X"a3d63433", X"358a68fd", 
  X"0f9b9d3c", X"d6aa295b", X"fe33384a", X"c000738e", 
  X"cd67eb2f", X"e2eb6dc2", X"97338b02", X"06c9f246", 
  X"419cf1ad", X"2b83c045", X"3723f18a", X"cb5b3089", 
  X"160bead7", X"5d494656", X"35f8a74b", X"1e4e6c9e", 
  X"000399bd", X"67466880", X"b4174831", X"acf423b2", 
  X"ca815ab3", X"5a6395e7", X"302a67c5", X"8bdb446b", 
  X"108f8fa4", X"10223eda", X"92b8b48b", X"7f38d0ee", 
  X"ab2701d4", X"0262d415", X"af224a30", X"b3d88aba", 
  X"f8b2c3af", X"daf7ef70", X"cc97d3b7", X"e9614b6c", 
  X"2baebff4", X"70f687cf", X"386c9156", X"ce092ee5", 
  X"01e87da6", X"6ce91e6a", X"bb7bcc84", X"c7922c20", 
  X"9d3b71fd", X"060e41c6", X"d7590f15", X"4e03bb47", 
  X"183c198e", X"63eeb240", X"2ddbf49a", X"6d5cba54", 
  X"923750af", X"f9e14236", X"7838162b", X"59726c72", 
  X"81b66760", X"bb2926c1", X"48a0ce0d", X"a6c0496d", 
  X"ad43507b", X"718d496a", X"9df057af", X"44b1bde6", 
  X"054356dc", X"de7ced35", X"d51a138b", X"62088cc9", 
  X"35830311", X"c96efca2", X"686f86ec", X"8e77cb68", 
  X"63e1d6b8", X"c80f9778", X"79c491fd", X"1b4c67f2", 
  X"72698d7d", X"5e368c31", X"f7d95e2e", X"a1d3493f", 
  X"dcd9433e", X"896f1552", X"4bc4ca7a", X"a6d1baf4", 
  X"a5a96dcc", X"0bef8b46", X"a169fda7", X"74df40b7", 
  X"4e208804", X"9a756607", X"038e87c8", X"20211e44", 
  X"8b7ad4bf", X"c6403f35", X"1848e36d", X"80bdb038", 
  X"1e62891c", X"643d2107", X"bf04d6f8", X"21092c8c", 
  X"f644f389", X"0778404e", X"7b78adb8", X"a2c52d53", 
  X"42157abe", X"a2253e2e", X"7bf3f4ae", X"80f594f9", 
  X"953194e7", X"77eb92ed", X"b3816930", X"da8d9336", 
  X"bf447469", X"f26d9483", X"ee6faed5", X"71371235", 
  X"de425f73", X"b4e59f43", X"7dbe2d4e", X"2d37b185", 
  X"49dc9a63", X"98c39d98", X"1301c9a2", X"389b1bbf", 
  X"0c18588d", X"a421c1ba", X"7aa3865c", X"71e08558", 
  X"3c5cfcaa", X"7d239ca4", X"0297d9dd", X"d7dc2830", 
  X"4b37802b", X"7428ab54", X"aeee0347", X"4b3fbb85", 
  X"692f2f08", X"134e578e", X"36d9e0bf", X"ae8b5fcf", 
  X"edb93ecf", X"2b27248e", X"170eb1ef", X"7dc57fd6", 
  X"1e760f16", X"b1136601", X"864e1b9b", X"d7ea7319", 
  X"3ab871bd", X"cfa4d76f", X"e31bd782", X"0dbeb469", 
  X"abb96061", X"5370f85d", X"ffb07e37", X"da30d0fb", 
  X"ebc977b6", X"0b98b40f", X"3a4d0fe6", X"df4fc26b", 
  X"159cf22a", X"c298d6e2", X"2b78ef6a", X"61a94ac0", 
  X"ab561187", X"14eea0f0", X"df0d4164", X"19af70ee"
);





-- ==========================================================================
-- ============ Declarations for the Encrypt/Decrypt section ================
-- ==========================================================================


type PIPE_DATA_TYPE  is array(FIRST_ROUND to LAST_ROUND) of SLV_128;
type PIPE_DATA_TYPE2 is array(FIRST_ROUND to LAST_ROUND - 1) of SLV_128;

function  E_FUNCTION  (  enc_key1  : in SLV_32; 
                         enc_key2  : in SLV_32;
                         in_value  : in SLV_32)
                         return SLV_96;

function FORWARD_UNKEYED_ROUND_FUNCT (D           : in SLV_128;
                                      enc_dec_b   : in std_logic;
                                      round       : in SLV_6 ) 
                                      RETURN       SLV_128;

procedure FORWARD_UNKEYED_ROUND( ALG_DATA   : in SLV_128;    
                                 ALG_ENC    : in std_logic;
                                 ROUND_STEP : in SLV_6;
                          signal dataout    : out SLV_128 );

function FORWARD_KEYED_ROUND_FUNCT (datain   : in SLV_128;
                          dec_key1 : in SLV_32;
                          dec_key2 : in SLV_32;
                          enc_decb : in std_logic;
                          enc_key1 : in SLV_32;
                          enc_key2 : in SLV_32;
                          round    : in SLV_6 )
                          return SLV_128;

procedure FORWARD_KEYED_ROUND (datain   : in SLV_128;
                          enc_key1 : in SLV_32;
                          enc_key2 : in SLV_32;
                          dec_key1 : in SLV_32;
                          dec_key2 : in SLV_32;
                          enc_decb : in std_logic;
                          round    : in SLV_6;
                   signal DATAOUT  : OUT SLV_128);

function BACKWARD_KEYED_ROUND_FUNCT(datain   : in SLV_128;
                          dec_key1 : in SLV_32;
                          dec_key2 : in SLV_32;
                          enc_decb : in std_logic;
                          enc_key1 : in SLV_32;
                          enc_key2 : in SLV_32
                         
                          ) RETURN SLV_128;

procedure BACKWARD_KEYED_ROUND(ALG_DATA:in SLV_128;
                              enc_key1 : in SLV_32;
                              enc_key2 : in SLV_32;
                              dec_key1 : in SLV_32;
                              dec_key2 : in SLV_32;
                              ALG_ENC  : in std_logic;
                              signal dataout : out SLV_128  );

function BACKWARD_UNKEYED_ROUND_FUNCT (datain    : in SLV_128;
                                       enc_dec_b : in std_logic;
                                       round     : in SLV_6 )
                                       return SLV_128;

procedure BACKWARD_UNKEYED_ROUND (datain     : in SLV_128;
                                  enc_decb   : in std_logic;
                                  round_step : in SLV_6;
                           signal DATAOUT    : out SLV_128 );

function KEY_ADD_FUNCT    (DATAIN      : in SLV_128; 
                           DEC_KEY0    : in SLV_32;
                           DEC_KEY1    : in SLV_32;
                           DEC_KEY2    : in SLV_32;
                           DEC_KEY3    : in SLV_32;
                           ENC_DECB    : in std_logic;
                           ENC_KEY0    : in SLV_32;
                           ENC_KEY1    : in SLV_32;
                           ENC_KEY2    : in SLV_32;
                           ENC_KEY3    : in SLV_32 )
                           return SLV_128;

procedure  KEY_ADD( DATAIN      : in SLV_128;
                    ENC_KEY0    : in SLV_32;
                    ENC_KEY1    : in SLV_32;
                    ENC_KEY2    : in SLV_32;
                    ENC_KEY3    : in SLV_32;
                    DEC_KEY0    : in SLV_32;
                    DEC_KEY1    : in SLV_32;
                    DEC_KEY2    : in SLV_32;
                    DEC_KEY3    : in SLV_32;
                    ENC_DECB    : in std_logic;
             signal DATAOUT     : out SLV_128 );

function KEY_SUB_FUNCT(DATAIN      : in SLV_128;
                       DEC_KEY0    : in SLV_32;
                       DEC_KEY1    : in SLV_32;
                       DEC_KEY2    : in SLV_32;
                       DEC_KEY3    : in SLV_32;
                       ENC_DECB    : in std_logic; 
                       ENC_KEY0    : in SLV_32;
                       ENC_KEY1    : in SLV_32;
                       ENC_KEY2    : in SLV_32;
                       ENC_KEY3    : in SLV_32 )
                       return SLV_128;

procedure KEY_SUB(  DATAIN      : in SLV_128;
                    ENC_KEY0    : in SLV_32;
                    ENC_KEY1    : in SLV_32;
                    ENC_KEY2    : in SLV_32;
                    ENC_KEY3    : in SLV_32;
                    DEC_KEY0    : in SLV_32;
                    DEC_KEY1    : in SLV_32;
                    DEC_KEY2    : in SLV_32;
                    DEC_KEY3    : in SLV_32;
                    ENC_DECB    : in std_logic;
                    signal DATAOUT : out SLV_128 );

function SBOX0_LOOKUP (S_BOX_INDEX : SLV_8)
                       return SLV_32;

function SBOX1_LOOKUP (S_BOX_INDEX : SLV_8)
                       return SLV_32;

function ROTATE_WORD_RIGHT ( datain  : ARRAY_TYPE_128 )
                             return ARRAY_TYPE_128;

function ROTATE_WORD_LEFT  ( datain  : ARRAY_TYPE_128 )
                             return ARRAY_TYPE_128;

function WORD_SWAP_FUNCT (datain         : in SLV_128;
                          enc_decb       : in std_logic;
                          round_step     : in SLV_6;
                          swap_step      : in SLV_6 )
                          return ARRAY_TYPE_128;


procedure WORD_SWAP (datain         : in SLV_128;
                     enc_decb       : in std_logic;
                     round_step     : in SLV_6;
                     swap_step      : in SLV_6;
                     signal DATAOUT : out ARRAY_TYPE_128 );



-- ==========================================================================
-- ============== Declarations for the Key Schedule section =================
-- ==========================================================================

constant HOLD              : integer := 0;

type B_ARRAY_TYPE is array (0 to 3)  of SLV_32;
type K_ARRAY_TYPE is array (0 to 39) of SLV_32;
type T_ARRAY_TYPE is array (0 to 14) of SLV_32;
type CV_ARRAY_TYPE is array (0 to 7) of SLV_32;
subtype OFFSET_TYPE   is integer range 0 to 3;
subtype INDEX_TYPE   is integer range 0 to 15;
subtype KEY_INDEX_TYPE   is integer range 0 to 39;
subtype RUNUP_INDEX_TYPE is integer range 0 to 19;

type K_PARTIAL_TYPE is array( 0 to 9 ) of SLV_32;

type T_PARTIAL_TYPE is array(0 to 4) of SLV_32;

type INDEX_TABLE_TYPE is array(0 to 14) of INDEX_TYPE;
constant LT_INDEX_TABLE_7 : INDEX_TABLE_TYPE := (
  8, 9, 10, 11, 12, 13, 14, 0, 1, 2, 3, 4, 5, 6, 7 );
constant LT_INDEX_TABLE_2 : INDEX_TABLE_TYPE := (
  13, 14, 0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12 );
constant STIR_INDEX_TABLE_1 : INDEX_TABLE_TYPE := (
  14, 0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13 );




constant B                 : B_ARRAY_TYPE := ( X"A4A8D57B", X"5B5D193B",
                                               X"C8A8309B", X"73F9A978" );
constant RUNUP_CYCLES      : integer := 6;
constant PIPE_STAGES       : integer := (LAST_ROUND + 1) / RUNUP_CYCLES;
constant KS_PIPE_STAGES    : integer := 4; -- depth of Key Sched pipe
constant NUM_RUNUP_ROUNDS  : integer := 41;   -- Total number of steps
constant NUM_ROUND_PER_CLK : integer := 1;
-- Number of clocks for cv runup
constant LAST_CVRUNUP_STEP : integer := NUM_RUNUP_ROUNDS/NUM_ROUND_PER_CLK;

constant KEY_OVERRUN_STEP  : integer := 13;

subtype LATCH_TYPE is std_logic_vector(0 to
                      (4*((LAST_ROUND/RUNUP_CYCLES)+1) * RUNUP_CYCLES - 1));

type S_ARRAY_TYPE  is array (0 to 39) of SLV_32;
type L_ARRAY_TYPE  is array (0 to 7) of SLV_32;
type S_PIPE_TYPE   is array (0 to PIPE_STAGES) of S_ARRAY_TYPE;
type S_MESH_TYPE   is array (0 to ((((34/RUNUP_CYCLES)+1)*40)-1)) of SLV_32;



function INIT_T_FUNCT (CV      : CV_ARRAY_TYPE  )
                       return T_ARRAY_TYPE;

procedure INIT_T (CV       : CV_ARRAY_TYPE;
                  signal T : out T_ARRAY_TYPE );

function SBOX_LOOKUP (S_BOX_INDEX : SLV_9)
                      return SLV_32;

function LINEAR_TRANSFORM (OFFSET : SLV_2;
                           T      : T_ARRAY_TYPE )
                           return T_ARRAY_TYPE;

function STIR (T : T_ARRAY_TYPE)
               return T_ARRAY_TYPE;

function GENERATE_MASK (W : SLV_32)
                        return SLV_32;

function FIX_WEAKKEYS_FUNCT (K : K_PARTIAL_TYPE;
                             j : SLV_2 )
                             return K_PARTIAL_TYPE;

procedure FIX_WEAKKEYS (K_IN         : K_PARTIAL_TYPE;
                        j            : SLV_2;
                        signal K_OUT : out K_PARTIAL_TYPE );


function CREATE_ROUNDKEY_FUNCT (CV      : CV_ARRAY_TYPE   )
                                return K_ARRAY_TYPE;

procedure CREATE_ROUNDKEY (CV       : CV_ARRAY_TYPE;
                           signal K : out K_ARRAY_TYPE );

function CREATE_T_FUNCT (T    : T_ARRAY_TYPE;
                         j    : SLV_2)
                         return T_ARRAY_TYPE;

procedure CREATE_T (T_IN         : T_ARRAY_TYPE;
                    j            : SLV_2;
                    signal T_OUT : out T_ARRAY_TYPE );

function MAKE_SUBKEYS_FUNCT (T      : T_ARRAY_TYPE )
                             return K_PARTIAL_TYPE;

procedure MAKE_SUBKEYS ( T_IN      : T_ARRAY_TYPE;
                         enc_decb : std_logic;
                         signal K_OUT : out K_PARTIAL_TYPE );


end mars_pack;


-- ==========================================================================
-- ==========================================================================
-- ==========================================================================


package body mars_pack is


-- ==========================================================================
-- ===================== Definitions for general use ========================
-- ==========================================================================


-- ==========================================================================
--
--  procedure BYTESWAP32
--
--  This function swaps bytes in a 32-bit word where the input word is
--  a single 32-bit word. The byte swapping is performed as follows:
--
--     ABCD  -->  DCBA 
--
-- ==========================================================================

procedure BYTESWAP32( datain  : in SLV_32;
               signal dataout : out SLV_32 ) is

begin

  dataout <= datain(7 downto 0)   & datain(15 downto 8) &
             datain(23 downto 16) & datain(31 downto 24);

end BYTESWAP32;


-- ==========================================================================
--
--  function BYTESWAP32
--
--  This function swaps bytes in a 32-bit word where the input word is
--  a single 32-bit word. The byte swapping is performed as follows:
--
--     ABCD  -->  DCBA 
--
-- ==========================================================================

function BYTESWAP32_FUNCT( datain  : in SLV_32)
                           return SLV_32 is
-- pragma map_to_operator BYTESWAP32_FUNCT_dw_op
-- pragma return_port_name BYTESWAP32_FUNCT_out

begin

  return (datain(7 downto 0)   & datain(15 downto 8) &
             datain(23 downto 16) & datain(31 downto 24));
end BYTESWAP32_FUNCT;



-- ==========================================================================
--
--  function BYTESWAPU32_FUNCT
--
--  This function swaps bytes in a 32-bit word where the input word is
--  an unsigned 32-bit word. The byte swapping is performed as follows:
--
--     ABCD  -->  DCBA 
--
-- ==========================================================================

function BYTESWAPU32_FUNCT( datain  : in U_32)
                            return U_32 is
-- pragma map_to_operator BYTESWAPU32_FUNCT_dw_op
-- pragma return_port_name BYTESWAPU32_FUNCT_out

begin

  return (datain(7 downto 0)   & datain(15 downto 8) &
             datain(23 downto 16) & datain(31 downto 24));
end BYTESWAPU32_FUNCT;




-- ==========================================================================
-- ============= Definitions for the Encrypt/Decrypt section ================
 ==========================================================================


-- ==========================================================================
--
-- function pre add
--
-- ==========================================================================
function KEY_ADD_FUNCT    (DATAIN      : in SLV_128;
                           DEC_KEY0    : in SLV_32;
                           DEC_KEY1    : in SLV_32;
                           DEC_KEY2    : in SLV_32;
                           DEC_KEY3    : in SLV_32;
                           ENC_DECB    : in std_logic; 
                           ENC_KEY0    : in SLV_32;
                           ENC_KEY1    : in SLV_32;
                           ENC_KEY2    : in SLV_32;
                           ENC_KEY3    : in SLV_32 )
                           return SLV_128 is
-- pragma map_to_operator KEY_ADD_FUNCT_dw_op
-- pragma return_port_name KEY_ADD_FUNCT_out

variable dataout  : ARRAY_TYPE_128;
variable key      : ARRAY_TYPE_128;

begin

  for i in 3 downto 0 loop
    dataout(i) := unsigned(BYTESWAP32_FUNCT(DATAIN((32*i+31) downto (32*i))));
  end loop;


  if ENC_DECB = '1' then      -- encrypt

    key(3) := unsigned(ENC_KEY3);
    key(2) := unsigned(ENC_KEY2);
    key(1) := unsigned(ENC_KEY1);
    key(0) := unsigned(ENC_KEY0);

  else

    key(3) := unsigned(DEC_KEY3);
    key(2) := unsigned(DEC_KEY2);
    key(1) := unsigned(DEC_KEY1);
    key(0) := unsigned(DEC_KEY0);

  end if;

  for i in 3 downto 0 loop
    dataout(i) := dataout(i) + key(i);
  end loop;

  return std_logic_vector(BYTESWAPU32_FUNCT(dataout(3))) &
         std_logic_vector(BYTESWAPU32_FUNCT(dataout(2))) &
         std_logic_vector(BYTESWAPU32_FUNCT(dataout(1))) &
         std_logic_vector(BYTESWAPU32_FUNCT(dataout(0)));

end KEY_ADD_FUNCT;



-- ==========================================================================
--
--  procedure pre add
--
-- ==========================================================================

procedure  KEY_ADD( DATAIN      : in SLV_128;
                    ENC_KEY0    : in SLV_32;
                    ENC_KEY1    : in SLV_32;
                    ENC_KEY2    : in SLV_32;
                    ENC_KEY3    : in SLV_32;
                    DEC_KEY0    : in SLV_32;
                    DEC_KEY1    : in SLV_32;
                    DEC_KEY2    : in SLV_32;
                    DEC_KEY3    : in SLV_32;
                    ENC_DECB    : in std_logic;
                    signal DATAOUT : out SLV_128 ) is


begin
    DATAOUT <= KEY_ADD_FUNCT( DATAIN,
                              DEC_KEY0,
                              DEC_KEY1,
                              DEC_KEY2,
                              DEC_KEY3,
                              ENC_DECB, 
                              ENC_KEY0,
                              ENC_KEY1,
                              ENC_KEY2,
                              ENC_KEY3 );

end KEY_ADD;


-- ==========================================================================
--
-- function post add
--
-- ==========================================================================
function KEY_SUB_FUNCT( DATAIN      : in SLV_128;
                        DEC_KEY0    : in SLV_32;
                        DEC_KEY1    : in SLV_32;
                        DEC_KEY2    : in SLV_32;
                        DEC_KEY3    : in SLV_32;
                        ENC_DECB    : in std_logic;
                        ENC_KEY0    : in SLV_32;
                        ENC_KEY1    : in SLV_32;
                        ENC_KEY2    : in SLV_32;
                        ENC_KEY3    : in SLV_32 )
                        return SLV_128 is
-- pragma map_to_operator KEY_SUB_FUNCT_dw_op
-- pragma return_port_name KEY_SUB_FUNCT_out

variable dataout: ARRAY_TYPE_128;

begin

  for i in 3 downto 0 loop
    dataout(i) := unsigned( BYTESWAP32_FUNCT(DATAIN(32*i+31 downto 32*i)) );
  end loop;

  if ENC_DECB = '1' then

    dataout(3) := dataout(3) - unsigned(ENC_KEY3);
    dataout(2) := dataout(2) - unsigned(ENC_KEY2);
    dataout(1) := dataout(1) - unsigned(ENC_KEY1);
    dataout(0) := dataout(0) - unsigned(ENC_KEY0);

  else

    dataout(3) := dataout(3) - unsigned(DEC_KEY3);
    dataout(2) := dataout(2) - unsigned(DEC_KEY2);
    dataout(1) := dataout(1) - unsigned(DEC_KEY1);
    dataout(0) := dataout(0) - unsigned(DEC_KEY0);

  end if;

  return std_logic_vector(BYTESWAPU32_FUNCT(dataout(3))) &
         std_logic_vector(BYTESWAPU32_FUNCT(dataout(2))) &
         std_logic_vector(BYTESWAPU32_FUNCT(dataout(1))) &
         std_logic_vector(BYTESWAPU32_FUNCT(dataout(0)));

end KEY_SUB_FUNCT;



-- ==========================================================================
--
-- procedure post add
--
-- ==========================================================================
procedure KEY_SUB(  DATAIN          : in SLV_128;
                    ENC_KEY0    : in SLV_32;
                    ENC_KEY1    : in SLV_32;
                    ENC_KEY2    : in SLV_32;
                    ENC_KEY3    : in SLV_32;
                    DEC_KEY0    : in SLV_32;
                    DEC_KEY1    : in SLV_32;
                    DEC_KEY2    : in SLV_32;
                    DEC_KEY3    : in SLV_32;
                    ENC_DECB    : in std_logic;
                    signal DATAOUT : out SLV_128 ) is

begin

    DATAOUT <= KEY_SUB_FUNCT( DATAIN,
                              DEC_KEY0,
                              DEC_KEY1,
                              DEC_KEY2,
                              DEC_KEY3,
                              ENC_DECB, 
                              ENC_KEY0,
                              ENC_KEY1,
                              ENC_KEY2,
                              ENC_KEY3 );

end KEY_SUB;




-- ==========================================================================
--
--  function FORWARD_UNKEYED_ROUND_FUNCT
--
-- ==========================================================================

function FORWARD_UNKEYED_ROUND_FUNCT (
         D         : in SLV_128;
         enc_dec_b : in std_logic;
         round     : in SLV_6)
         RETURN       SLV_128 is
-- pragma map_to_operator FORWARD_UNKEYED_ROUND_FUNCT_dw_op
-- pragma return_port_name FORWARD_UNKEYED_ROUND_FUNCT_out

  variable D_temp: ARRAY_TYPE_128;
  variable b     : ARRAY_TYPE_32;
  variable output : SLV_128;

 begin

  D_temp := WORD_SWAP_FUNCT (D, enc_dec_b, round, "000001");
  
  b(3) := std_logic_vector( D_temp(0)(31 downto 24) );
  b(2) := std_logic_vector( D_temp(0)(23 downto 16));
  b(1) := std_logic_vector( D_temp(0)(15 downto 8));
  b(0) := std_logic_vector( D_temp(0)(7 downto 0) );

    -- Mixing/S-box lookups
  D_temp(1) := unsigned( std_logic_vector(D_temp(1)) xor SBOX0_LOOKUP(b(0)) );
  D_temp(1) := D_temp(1) + unsigned( SBOX1_LOOKUP(b(1)) );
  D_temp(2) := D_temp(2) + unsigned( SBOX0_LOOKUP(b(2)) );
  D_temp(3) := unsigned( std_logic_vector(D_temp(3)) xor SBOX1_LOOKUP(b(3)) );

    -- Rotate source word
  D_temp(0) := D_temp(0) ror 24;

    -- Conditional addition
  if ( TO_INTEGER(unsigned(round)) = 1 or TO_INTEGER(unsigned(round)) = 5 ) then
    D_temp(0) := D_temp(0) + D_temp(3);
  elsif ( TO_INTEGER(unsigned(round)) = 2 or TO_INTEGER(unsigned(round)) = 6 ) then
    D_temp(0) := D_temp(0) + D_temp(1);
  else
    D_temp(0) := D_temp(0);
  end if;

    -- Rotate D by one 32-bit word to the right
  D_temp := ROTATE_WORD_RIGHT(D_temp);

  output := std_logic_vector(BYTESWAPU32_FUNCT(D_temp(3))) &
            std_logic_vector(BYTESWAPU32_FUNCT(D_temp(2))) &
            std_logic_vector(BYTESWAPU32_FUNCT(D_temp(1))) &
            std_logic_vector(BYTESWAPU32_FUNCT(D_temp(0)));

  return ( output );

end FORWARD_UNKEYED_ROUND_FUNCT;


-- ==========================================================================
--  Procedure FORWARD_UNKEYED_ROUND
--===========================================================================
procedure FORWARD_UNKEYED_ROUND( ALG_DATA       :in SLV_128;    
                                 ALG_ENC        : in std_logic;
                                 ROUND_STEP     : in SLV_6;
                                 signal dataout : out SLV_128 ) is

begin
  dataout <= FORWARD_UNKEYED_ROUND_FUNCT( ALG_DATA,
                                          ALG_ENC,
                                          ROUND_STEP);

end FORWARD_UNKEYED_ROUND;


-- ==========================================================================
--
--  function FORWARD_KEYED_ROUND_FUNCT
--
--  Performs FORWARD MARS encrypt block cipher. Encryption or decryption
--  is performed based on the 'encrypt' signal.
--
-- ==========================================================================

function FORWARD_KEYED_ROUND_FUNCT (datain   : in SLV_128; 
                                    dec_key1 : in SLV_32;
                                    dec_key2 : in SLV_32;
                                    enc_decb : in std_logic;
                                    enc_key1 : in SLV_32;
                                    enc_key2 : in SLV_32;
                                                                                                                    round    : in SLV_6 ) 
                                    return SLV_128 is

-- pragma map_to_operator FORWARD_KEYED_ROUND_FUNCT_dw_op
-- pragma return_port_name FORWARD_KEYED_ROUND_FUNCT_out

variable D: ARRAY_TYPE_128;
variable E_out : SLV_96;

begin

  D := WORD_SWAP_FUNCT (datain, enc_decb, round,  std_logic_vector(TO_UNSIGNED(9,6)));

-- Encrypt
  if enc_decb = '1' then

    -- E-Function
    E_out := E_FUNCTION( enc_key1, enc_key2,std_logic_vector(D(0)) );

    -- Rotate source word by 13 bits
    D(0)  := D(0) rol 13;

    -- Mix E-Function output with data
    D(1)  := D(1) + unsigned(E_out(95 downto 64));
    D(2)  := unsigned(D(2)) + unsigned(E_out(63 downto 32));
    D(3)  := unsigned(std_logic_vector(D(3)) xor E_out(31 downto 0));

    -- Rotate data 32 bits to the right
    D := ROTATE_WORD_RIGHT( D );

--Decrypt
  else

    -- Rotate data 32 bits to the left
    D := ROTATE_WORD_LEFT( D );

    -- Rotate source word by 13 bits
    D(0)  := D(0) ror 13;

    -- E-Function
    E_out := E_FUNCTION( dec_key1, dec_key2,std_logic_vector(D(0)));

    -- Mix E-Function output with data
    D(2)  := unsigned(D(2)) - unsigned(E_out(63 downto 32));

    D(3)  := unsigned(D(3)) - unsigned(E_out(95 downto 64));
    D(1)  := unsigned(std_logic_vector(D(1)) xor E_out(31 downto 0));

  end if;

  return std_logic_vector(BYTESWAPU32_FUNCT(D(3))) &
         std_logic_vector(BYTESWAPU32_FUNCT(D(2))) &
         std_logic_vector(BYTESWAPU32_FUNCT(D(1))) &
         std_logic_vector(BYTESWAPU32_FUNCT(D(0)));

end FORWARD_KEYED_ROUND_FUNCT;



-- ==========================================================================
--
--  Procedure FORWARD_KEYED_ROUND
--
--  Performs FORWARD MARS encrypt block cipher. Encryption or decryption
--  is performed based on the 'encrypt' signal.
--
-- ==========================================================================
procedure FORWARD_KEYED_ROUND (datain   : in SLV_128;
                               enc_key1 : in SLV_32;
                               enc_key2 : in SLV_32;
                               dec_key1 : in SLV_32;
                               dec_key2 : in SLV_32;
                               enc_decb : in std_logic;
                               round    : in SLV_6;
                               signal DATAOUT  : OUT SLV_128) is 
begin

   DATAOUT <= FORWARD_KEYED_ROUND_FUNCT( datain,
                                         dec_key1,
                                         dec_key2,
                                         enc_decb,
                                         enc_key1,
                                         enc_key2,
                                         round);

end FORWARD_KEYED_ROUND;


-- ==========================================================================
--
--  function BACKWARD_KEYED_ROUND_FUNCT
--
-- ==========================================================================

function BACKWARD_KEYED_ROUND_FUNCT(
                          datain   : in SLV_128;
                          dec_key1 : in SLV_32;
                          dec_key2 : in SLV_32;
                          enc_decb : in std_logic;
                          enc_key1 : in SLV_32;
                          enc_key2 : in SLV_32 ) RETURN SLV_128 is
-- pragma map_to_operator BACKWARD_KEYED_ROUND_FUNCT_dw_op
-- pragma return_port_name BACKWARD_KEYED_ROUND_FUNCT_out

variable D: ARRAY_TYPE_128;
variable E_out : SLV_96;

begin

  for i in 3 downto 0 loop
    D(i) := unsigned(BYTESWAP32_FUNCT(datain(32*i+31 downto 32*i)));
  end loop;

-- Encrypt
  if enc_decb = '1' then

    -- E-Function
    E_out := E_FUNCTION( enc_key1, enc_key2,std_logic_vector(D(0)));

    -- Rotate source word by 13 bits
    D(0)  := D(0) rol 13;

    -- Mix E-Function output with data
    D(3)  := unsigned(D(3)) + unsigned(E_out(95 downto 64));
    D(2)  := unsigned(D(2)) + unsigned(E_out(63 downto 32));
    D(1)  := unsigned(std_logic_vector(D(1)) xor E_out(31 downto 0));

    -- Rotate data 32 bits to the right
    D := ROTATE_WORD_RIGHT( D );

--Decrypt
  else

    -- Rotate data 32 bits to the left
    D := ROTATE_WORD_LEFT( D );

    -- Rotate source word by 13 bits
    D(0)  := D(0) ror 13;

    -- E-Function
    E_out := E_FUNCTION(dec_key1, dec_key2,std_logic_vector(D(0)));

    -- Mix E-Function output with data
    D(2)  := unsigned(D(2)) - unsigned(E_out(63 downto 32));

    D(1)  := unsigned(D(1)) - unsigned(E_out(95 downto 64));
    D(3)  := unsigned(std_logic_vector(D(3)) xor E_out(31 downto 0));

  end if;

 
 return std_logic_vector(BYTESWAPU32_FUNCT(D(3))) &
         std_logic_vector(BYTESWAPU32_FUNCT(D(2))) &
         std_logic_vector(BYTESWAPU32_FUNCT(D(1))) &
         std_logic_vector(BYTESWAPU32_FUNCT(D(0)));

end BACKWARD_KEYED_ROUND_FUNCT;


-- ==========================================================================
--
--  procedure BACKWARD_KEYED_ROUND
--
-- ==========================================================================

procedure BACKWARD_KEYED_ROUND(ALG_DATA:in SLV_128;
                           enc_key1 : in SLV_32;
                          enc_key2 : in SLV_32;
                          dec_key1 : in SLV_32;
                          dec_key2 : in SLV_32;
                           ALG_ENC: in std_logic;
                            signal dataout : out SLV_128  ) is


begin
  dataout <= BACKWARD_KEYED_ROUND_FUNCT( ALG_DATA,              
                                         dec_key1, 
                                         dec_key2,                  
                                         ALG_ENC,
                                         enc_key1,
                                         enc_key2
                                         );
end BACKWARD_KEYED_ROUND;


-- ===================================================================
-- E function
-- ===================================================================
function  E_FUNCTION(enc_key1: in SLV_32;
                     enc_key2: in SLV_32;
                     in_value: in SLV_32)
                     return SLV_96 is
-- pragma map_to_operator E_FUNCTION_dw_op
-- pragma return_port_name E_FUNCTION_out

  variable L: SLV_32;
  variable M: SLV_32;
  variable R: SLV_32;
  variable temp : SLV_64;

begin
 M := std_logic_vector(unsigned(in_value) + unsigned(enc_key1));
 temp := std_logic_vector((unsigned(in_value) rol 13) * (unsigned(enc_key2)));
 R := temp(31 downto 0);
 L := SBOX_LOOKUP( M(8 downto 0));
 R := std_logic_vector(unsigned(R) rol 5);
 L := L xor R;
 M := std_logic_vector(unsigned(M) rol TO_INTEGER(unsigned(R(4 downto 0))));
 R := std_logic_vector(unsigned(R) rol 5);
 L := L xor R;
 
 L := std_logic_vector(unsigned(L) rol TO_INTEGER(unsigned(R(4 downto 0))));  

  return(L&M&R);

end E_FUNCTION;


-- ==========================================================================
--
--  function BACKWARD_UNKEYED_ROUND_FUNCT
--
--  Performs REVERSE MARS encrypt block cipher. Encryption or decryption
--  is performed based on the 'encrypt' signal.
--
-- ==========================================================================

function BACKWARD_UNKEYED_ROUND_FUNCT(datain    : in SLV_128;
                                      enc_dec_b : in std_logic;
                                      round     : in SLV_6 )
                                      return SLV_128 is
-- pragma map_to_operator BACKWARD_UNKEYED_ROUND_FUNCT_dw_op
-- pragma return_port_name BACKWARD_UNKEYED_ROUND_FUNCT_out

  variable D     : ARRAY_TYPE_128;
  variable b     : ARRAY_TYPE_32;

begin

  D := WORD_SWAP_FUNCT (datain, enc_dec_b, round,  std_logic_vector(TO_UNSIGNED(25,6)));

    -- Conditional subtraction
  if (  TO_INTEGER(unsigned(round)) = 27 or  TO_INTEGER(unsigned(round))  = 31 ) then
    D(0) := D(0) - D(3);
  elsif (  TO_INTEGER(unsigned(round))  = 28 or  TO_INTEGER(unsigned(round))  = 32 ) then
    D(0) := D(0) - D(1);
  else
    D(0) := D(0);
  end if;

    -- Assign S-box lookup values
  b(0) := std_logic_vector(D(0)(7 downto 0));
  b(1) := std_logic_vector(D(0)(15 downto 8));
  b(2) := std_logic_vector(D(0)(23 downto 16));
  b(3) := std_logic_vector(D(0)(31 downto 24));

    -- Mixing/S-box lookups
  D(1) := unsigned( std_logic_vector(D(1)) xor SBOX1_LOOKUP(b(0)) );
  D(2) := D(2) - unsigned( SBOX0_LOOKUP(b(3)) ) ;
  D(3) := D(3) - unsigned( SBOX1_LOOKUP(b(2)) );
  D(3) := unsigned( std_logic_vector(D(3)) xor SBOX0_LOOKUP(b(1)) );

    -- Rotate source word
  D(0) := D(0) rol 24;

    -- Rotate D by one 32-bit word to the right
  D := ROTATE_WORD_RIGHT( D );

  return std_logic_vector(BYTESWAPU32_FUNCT(D(3))) &
         std_logic_vector(BYTESWAPU32_FUNCT(D(2))) &
         std_logic_vector(BYTESWAPU32_FUNCT(D(1))) &
         std_logic_vector(BYTESWAPU32_FUNCT(D(0)));

end BACKWARD_UNKEYED_ROUND_FUNCT;


-- ==========================================================================
--
--  procedure BACKWARD_MARS_ROUND
--
--  Performs REVERSE MARS encrypt block cipher. Encryption or decryption
--  is performed based on the 'encrypt' signal.
--
-- ==========================================================================

procedure BACKWARD_UNKEYED_ROUND (datain         : in SLV_128;
                                  enc_decb       : in std_logic;
                                  round_step     : in SLV_6;
                                  signal DATAOUT : out SLV_128 ) is
begin
   DATAOUT <= BACKWARD_UNKEYED_ROUND_FUNCT( datain,
                                            enc_decb,
                                            round_step);

end BACKWARD_UNKEYED_ROUND;


-- ==========================================================================
--
-- function SBOX0_LOOKUP
--
-- ==========================================================================


function SBOX0_LOOKUP (S_BOX_INDEX : SLV_8)
                       return SLV_32 is
-- pragma map_to_operator SBOX0_LOOKUP_dw_op
-- pragma return_port_name SBOX0_LOOKUP_out

begin

return S(TO_INTEGER('0' & unsigned(S_BOX_INDEX)));

end SBOX0_LOOKUP;



-- ==========================================================================
--
-- function SBOX_LOOKUP
--
-- ==========================================================================


function SBOX1_LOOKUP (S_BOX_INDEX : SLV_8)
                      return SLV_32 is
-- pragma map_to_operator SBOX1_LOOKUP_dw_op
-- pragma return_port_name SBOX1_LOOKUP_out

begin

return S(TO_INTEGER('1' & unsigned(S_BOX_INDEX)));

end SBOX1_LOOKUP;



-- ==========================================================================
--
-- function ROTATE_WORD_RIGHT
--
-- ==========================================================================

function ROTATE_WORD_RIGHT ( datain  : ARRAY_TYPE_128 )
                             return ARRAY_TYPE_128 is
-- pragma map_to_operator ROTATE_WORD_RIGHT_dw_op
-- pragma return_port_name ROTATE_WORD_RIGHT_out

variable dataout : ARRAY_TYPE_128;
variable temp    : U_32;

begin

  temp := datain(0);

  for i in 2 downto 0 loop
    dataout(i) := datain(i+1);
  end loop;

  dataout(3) := temp;

  return dataout;

end ROTATE_WORD_RIGHT;


-- ==========================================================================
--
-- function ROTATE_WORD_LEFT
--
-- ==========================================================================

function ROTATE_WORD_LEFT  ( datain  : ARRAY_TYPE_128 )
                             return ARRAY_TYPE_128 is
-- pragma map_to_operator ROTATE_WORD_LEFT_dw_op
-- pragma return_port_name ROTATE_WORD_LEFT_out

variable dataout : ARRAY_TYPE_128;
variable temp    : U_32;

begin

  temp := datain(3);

  for i in 3 downto 1 loop
    dataout(i) := datain(i-1);
  end loop;

  dataout(0) := temp;

  return dataout;

end ROTATE_WORD_LEFT;



-- ==========================================================================
--
--  function WORD_SWAP_FUNCT
--
--  Performs REVERSE MARS encrypt block cipher. Encryption or decryption
--  is performed based on the 'encrypt' signal.
--
-- ==========================================================================

function WORD_SWAP_FUNCT (datain         : in SLV_128;
                     enc_decb       : in std_logic;
                     round_step     : in SLV_6;
                     swap_step      : in SLV_6 )
                     return ARRAY_TYPE_128 is
-- pragma map_to_operator WORD_SWAP_FUNCT_dw_op
-- pragma return_port_name WORD_SWAP_FUNCT_out

  variable D     : ARRAY_TYPE_128;

begin

  for i in 3 downto 0 loop
    if enc_decb = '0' and TO_INTEGER(unsigned(round_step)) = TO_INTEGER(unsigned(swap_step)) then
      D(3-i) := unsigned( BYTESWAP32_FUNCT(datain(32*i+31 downto 32*i)) );
    else
      D(i) := unsigned( BYTESWAP32_FUNCT(datain(32*i+31 downto 32*i)) );
   end if;
  end loop;

  return D;

end WORD_SWAP_FUNCT;


-- ==========================================================================
--
--  procedure WORD_SWAP
--
--  Performs REVERSE MARS encrypt block cipher. Encryption or decryption
--  is performed based on the 'encrypt' signal.
--
-- ==========================================================================

procedure WORD_SWAP (datain         : in SLV_128;
                     enc_decb       : in std_logic;
                     round_step     : in SLV_6;
                     swap_step      : in SLV_6;
                     signal DATAOUT : out ARRAY_TYPE_128 ) is

begin

  DATAOUT <= WORD_SWAP_FUNCT(datain, enc_decb, round_step, swap_step);

end WORD_SWAP;





-- ==========================================================================
-- ==========================================================================
--                 Definitions for the Key Schedule section                --
-- ==========================================================================
-- ==========================================================================

-- ==========================================================================
--
-- function GENERATE_MASK
--
-- ==========================================================================

function GENERATE_MASK (W : SLV_32)
                        return SLV_32 is
-- pragma map_to_operator GENERATE_MASK_dw_op
-- pragma return_port_name GENERATE_MASK_out

constant all_zeroes : SLV_10 := "0000000000";
constant all_ones   : SLV_10 := "1111111111";

variable mask      : SLV_32;

begin

   mask := (others => '0');
   for index in 31 downto 9 loop

      if (W(index downto index - 9) = all_zeroes) or
         (W(index downto index - 9) = all_ones)   then

         mask((index - 1) downto (index - 8)) := "11111111";

      else

         mask((index - 1) downto (index - 8)) := mask((index - 1) downto (index - 8));

      end if;

  mask(31) := '0';
  mask(1 downto 0) := "00";

   end loop;

   return mask;

end GENERATE_MASK;


-- ==========================================================================
--
-- function INIT_T
--
-- ==========================================================================


function INIT_T_FUNCT (CV      : CV_ARRAY_TYPE  )
                      return T_ARRAY_TYPE is

-- pragma map_to_operator INIT_T_FUNCT_dw_op
-- pragma return_port_name INIT_T_FUNCT_out

variable T : T_ARRAY_TYPE;
variable i : integer range 0 to 14;
variable n : integer range 4 to 8;

begin
n:=4;

  for i in 0 to 14 loop
    if ( i <= n-1 ) then
      T(i) := BYTESWAP32_FUNCT(CV(i mod 8));
    elsif i = n then
      T(i) := std_logic_vector( TO_UNSIGNED( n, 32) );
    else
      T(i) := ( others => '0' );
    end if;
  end loop;

  return T;
end INIT_T_FUNCT;


-- ==========================================================================
--
-- Procedure INIT_T 
--
-- ==========================================================================

procedure INIT_T (CV       : CV_ARRAY_TYPE;
                  signal T : out T_ARRAY_TYPE ) is

begin

  T <= INIT_T_FUNCT(CV);
       
end INIT_T;



-- ==========================================================================
--
-- function SBOX_LOOKUP
--
-- ==========================================================================


function SBOX_LOOKUP (S_BOX_INDEX : SLV_9)
                      return SLV_32 is
-- pragma map_to_operator SBOX_LOOKUP_dw_op
-- pragma return_port_name SBOX_LOOKUP_out

begin

return S(TO_INTEGER(unsigned(S_BOX_INDEX)));

end SBOX_LOOKUP;


-- ==========================================================================
--
-- function LINEAR_TRANSFORM
--
-- ==========================================================================


function LINEAR_TRANSFORM (OFFSET : SLV_2;
                           T      : T_ARRAY_TYPE )
                           return T_ARRAY_TYPE is
-- pragma map_to_operator LINEAR_TRANSFORM_dw_op
-- pragma return_port_name LINEAR_TRANSFORM_out

variable TEMP_T : T_ARRAY_TYPE;
variable TEMP_1 : SLV_32;
variable TEMP_2 : SLV_32;

begin
  for i in 0 to 14 loop
    TEMP_T(i) := T(i);
  end loop;

  for i in 0 to 14 loop

    TEMP_1 := std_logic_vector(
                unsigned( TEMP_T(LT_INDEX_TABLE_7(i)) xor
                          TEMP_T(LT_INDEX_TABLE_2(i))
                ) rol 3
              );

    TEMP_2 := std_logic_vector(
                TO_UNSIGNED(
                  (4*i+TO_INTEGER(unsigned(OFFSET))), 32
                )
              );

    TEMP_T(i) := TEMP_T(i) xor TEMP_1 xor TEMP_2;

  end loop;
return TEMP_T;

end LINEAR_TRANSFORM;



-- ==========================================================================
--
-- function STIR
--
-- ==========================================================================


function STIR (T : T_ARRAY_TYPE)
               return T_ARRAY_TYPE is
-- pragma map_to_operator STIR_dw_op
-- pragma return_port_name STIR_out

variable TEMP_T : T_ARRAY_TYPE;
variable TEMP_SBOX : SLV_32;

begin
  for i in 0 to 14 loop
    TEMP_T(i) := T(i);
  end loop;

  for i in 0 to 14 loop

    TEMP_SBOX := SBOX_LOOKUP( TEMP_T(STIR_INDEX_TABLE_1(i))(8 downto 0) );

    TEMP_T(i) := std_logic_vector(
                   (unsigned(TEMP_T(i)) + unsigned(TEMP_SBOX)) rol 9);

  end loop;

return TEMP_T;

end STIR;



-- ==========================================================================
--
-- function CREATE_ROUNDKEY_FUNCT
--
-- ==========================================================================


function CREATE_ROUNDKEY_FUNCT (CV      : CV_ARRAY_TYPE  )
                                return K_ARRAY_TYPE is
-- pragma map_to_operator CREATE_ROUNDKEY_FUNCT_dw_op
-- pragma return_port_name CREATE_ROUNDKEY_FUNCT_out

variable K : K_ARRAY_TYPE;
variable T : T_ARRAY_TYPE;
variable w : SLV_32;
variable i : KEY_INDEX_TYPE;
variable j : integer;
variable r : SLV_5;
variable M : SLV_32;
variable p : SLV_32;

begin

  T := INIT_T_FUNCT (CV);
     for j in 0 to 3 loop
    T := LINEAR_TRANSFORM(  std_logic_vector(TO_UNSIGNED(j,2)) ,T);
    for i in 0 to 3 loop
      T := STIR( T );
    end loop;
    for i in 0 to 9 loop
      K(10*j+i) := T(4*i mod 15);
    end loop;
  end loop;
  i := 5;
  while ( i /= 35 ) loop
    j := TO_INTEGER(unsigned(K(i)(1 downto 0)));
    w := K(i)(31 downto 2) & "11";
    M := GENERATE_MASK( w );
    r := K(i)(4 downto 0);
    p := std_logic_vector(
           unsigned(
             B( j) ) rol TO_INTEGER(unsigned(r)
           )
         );
    K(i) := w xor ( p and m );
    i := i + 2;
  end loop;

return K;

end CREATE_ROUNDKEY_FUNCT;



-- ==========================================================================
--
-- Procedure CREATE_ROUNDKEY 
--
-- ==========================================================================

procedure CREATE_ROUNDKEY (CV       : CV_ARRAY_TYPE;
                           signal K : out K_ARRAY_TYPE ) is

begin

  K <= CREATE_ROUNDKEY_FUNCT(CV); 

end CREATE_ROUNDKEY;



-- ==========================================================================
--
-- function FIX_WEAKKEYS_FUNCT
--
-- ==========================================================================


function FIX_WEAKKEYS_FUNCT (K : K_PARTIAL_TYPE;
                             j : SLV_2 )
                             return K_PARTIAL_TYPE is

-- pragma map_to_operator FIX_WEAKKEYS_FUNCT_dw_op
-- pragma return_port_name FIX_WEAKKEYS_FUNCT_out

variable TEMP_K : K_PARTIAL_TYPE;
variable w : SLV_32;
variable i : KEY_INDEX_TYPE;
variable j_temp : SLV_2;
variable r : SLV_5;
variable M : SLV_32;
variable p : SLV_32;
begin
  for i in 0 to 9 loop
    TEMP_K(i) := K(i);
  end loop;

  case j is
    when "00" =>  
     for i in 5 to 9 loop
       if  (i mod 2 = 0) then next;
       end if;
       j_temp := K(i)(1 downto 0);
       w := K(i)(31 downto 2) & "11";
       M := GENERATE_MASK( w );
       r := K(i-1)(4 downto 0);
       p := std_logic_vector(
              unsigned(
                B( TO_INTEGER(unsigned(j_temp))) ) rol TO_INTEGER(unsigned(r)
              )
            );
       TEMP_K(i) := w xor ( p and m );
     end loop;

    when "11" =>  
      for i in 1 to 5 loop
        if  (i mod 2 = 0) then next;
        end if;
       j_temp := K(i)(1 downto 0);
       w := K(i)(31 downto 2) & "11";
       M := GENERATE_MASK( w );
       r := K(i-1)(4 downto 0);
       p := std_logic_vector(
              unsigned(
                B( TO_INTEGER(unsigned(j_temp))) ) rol TO_INTEGER(unsigned(r)
              )
            );
       TEMP_K(i) := w xor ( p and m );
     end loop;

    when others =>  
     for i in 1 to 9 loop
        if  (i mod 2 = 0) then next;
        end if;
       j_temp := K(i)(1 downto 0);
       w := K(i)(31 downto 2) & "11";
       M := GENERATE_MASK( w );
       r := K(i-1)(4 downto 0);
       p := std_logic_vector(
              unsigned(
                B( TO_INTEGER(unsigned(j_temp))) ) rol TO_INTEGER(unsigned(r)
              )
            );
       TEMP_K(i) := w xor ( p and m );
        end loop;

  end case;

return TEMP_K;

end FIX_WEAKKEYS_FUNCT;


-- ==========================================================================
--
-- Procedure FIX_WEAKKEYS 
--
-- ==========================================================================

procedure FIX_WEAKKEYS (K_IN         : K_PARTIAL_TYPE;
                        j            : SLV_2;
                        signal K_OUT : out K_PARTIAL_TYPE ) is


begin

  K_OUT <= FIX_WEAKKEYS_FUNCT(K_IN, j);

end FIX_WEAKKEYS;



-- ==========================================================================
--
-- function CREATE_T_FUNCT
--
-- ==========================================================================


function CREATE_T_FUNCT (T      : T_ARRAY_TYPE;
                         j      : SLV_2 )
                         return T_ARRAY_TYPE is

-- pragma map_to_operator CREATE_T_FUNCT_dw_op
-- pragma return_port_name CREATE_T_FUNCT_out

variable K : K_ARRAY_TYPE;
variable T_TEMP : T_ARRAY_TYPE;
variable w : SLV_32;
variable i : KEY_INDEX_TYPE;
variable r : SLV_5;
variable M : SLV_32;
variable p : SLV_32;

begin

    T_TEMP := LINEAR_TRANSFORM( j, T );
    for i in 0 to 3 loop
      T_TEMP := STIR( T_TEMP );
    end loop;

return T_TEMP;

end CREATE_T_FUNCT;



-- ==========================================================================
--
-- Procedure CREATE_T 
--
-- ==========================================================================

procedure CREATE_T (T_IN         : T_ARRAY_TYPE;
                    j            : SLV_2;
                    signal T_OUT : out T_ARRAY_TYPE ) is

begin

  T_OUT <= CREATE_T_FUNCT(T_IN, j);

end CREATE_T;

-- ==========================================================================

-- ==========================================================================
--
-- function MAKE_SUBKEYS_FUNCT
--
-- ==========================================================================


function MAKE_SUBKEYS_FUNCT (T      : T_ARRAY_TYPE )
                         return K_PARTIAL_TYPE is

-- pragma map_to_operator MAKE_SUBKEYS_FUNCT_dw_op
-- pragma return_port_name MAKE_SUBKEYS_FUNCT_out

variable K : K_PARTIAL_TYPE;
variable i : KEY_INDEX_TYPE;

begin

    for i in 0 to 9 loop
      K(i) := T( (4*i) mod 15);
    end loop;

return K;

end MAKE_SUBKEYS_FUNCT;



-- ==========================================================================
--
-- Procedure MAKE_SUBKEYS 
--
-- ==========================================================================

procedure MAKE_SUBKEYS ( T_IN         : T_ARRAY_TYPE;
                         enc_decb     : std_logic;
                         signal K_OUT : out K_PARTIAL_TYPE ) is


begin

  K_OUT     <= MAKE_SUBKEYS_FUNCT(T_IN);

end MAKE_SUBKEYS;

-- ==========================================================================



end mars_pack;
