-- *************************************************************************
-- DISCLAIMER. THIS SOFTWARE WAS WRITTEN BY EMPLOYEES OF THE U.S.
-- GOVERNMENT AS A PART OF THEIR OFFICIAL DUTIES AND, THEREFORE, IS NOT
-- PROTECTED BY COPYRIGHT. HOWEVER, THIS SOFTWARE CODIFIES THE FINALIST
-- CANDIDATE ALGORITHMS (i.e., MARS, RC6tm, RIJNDAEL, SERPENT, AND
-- TWOFISH) IN THE ADVANCED ENCRYPTION STANDARD (AES) DEVELOPMENT EFFORT
-- SPONSORED BY THE NATIONAL INSTITUTE OF STANDARDS AND TECHNOLOGY (NIST)
-- AND MAY BE PROTECTED BY ONE OR MORE FORMS OF INTELLECTUAL PROPERTY. THE
-- U.S. GOVERNMENT MAKES NO WARRANTY, EITHER EXPRESSED OR IMPLIED,
-- INCLUDING BUT NO LIMITED TO ANY IMPLIED WARRANTIES OF MERCHANTABILITY
-- OR FITNESS FOR A PARTICULAR PURPOSE, REGARDING THIS SOFTWARE. THE U.S.
-- GOVERNMENT FURTHER MAKES NO WARRANTY THAT THIS SOFTWARE WILL NOT
-- INFRINGE ANY OTHER UNITED STATES OR FOREIGN PATENT OR OTHER
-- INTELLECTUAL PROPERTY RIGHT. IN NO EVENT SHALL THE U.S. GOVERNMENT BE
-- LIABLE TO ANYONE FOR COMPENSATORY, PUNITIVE, EXEMPLARY, SPECIAL,
-- COLLATERAL, INCIDENTAL, CONSEQUENTIAL, OR ANY OTHER TYPE OF DAMAGES IN
-- CONNECTION WITH OR ARISING OUT OF COPY OR USE OF THIS SOFTWARE.
-- *************************************************************************
--
-- File Name : rc6_pkg.vhdl
-- Author    : NSA
-- Date      : 07 October 99
-- Project   : AES Candidate Evaluation --RC6
-- Purpose   : This package defines common types, subtypes, constants,
--             and functions required to implement various VHDL models
--             for the creation of ASIC simulation of RC6, an Advanced
--             Encryption Standard (AES) candidate algorithm.
-- Notes     :
-- ===========================================================================

library ieee;
use ieee.std_logic_1164.all;
use ieee.numeric_std.all;


package rc6_pack is

-- ==========================================================================
-- ======= Type, sub-type and function declarations for general use =========
-- ==========================================================================
type STATE_TYPE         is (nop, wait4ks, ready, busy );

subtype ROUND_TYPE      is integer range 0 to 31;

subtype STEP_TYPE       is integer range 0 to 131;

subtype SLV_2           is std_logic_vector(1 downto 0);
subtype SLV_5           is std_logic_vector(4 downto 0);
subtype SLV_6           is std_logic_vector(5 downto 0);
subtype SLV_8           is std_logic_vector(7 downto 0);
subtype SLV_21          is std_logic_vector(21 downto 0);
subtype SLV_32          is std_logic_vector(31 downto 0);
subtype SLV_64          is std_logic_vector(63 downto 0);
subtype SLV_128         is std_logic_vector(127 downto 0);
subtype SLV_256         is std_logic_vector(255 downto 0);

constant FIRST_ROUND : ROUND_TYPE := 0;
constant LAST_ROUND  : ROUND_TYPE := 21;


procedure BYTESWAP128( datain  : in SLV_128;
                signal dataout : out SLV_128 );

procedure BYTESWAP32( datain  : in SLV_32;
               signal dataout : out SLV_32 );


-- ==========================================================================
-- ============ Declarations for the Encrypt/Decrypt section ================
-- ==========================================================================

type PIPE_DATA_TYPE is array(FIRST_ROUND to LAST_ROUND - 1) of SLV_128;

function PRE_WHITEN_FUNCT ( datain   : in SLV_128;
                            enc_key1 : in SLV_32;
                            enc_key2 : in SLV_32;
                            dec_key1 : in SLV_32;
                            dec_key2 : in SLV_32;
                            enc_decb : in std_logic )
                            return SLV_128;

procedure PRE_WHITEN ( datain   : in SLV_128;
                       enc_key1 : in SLV_32;
                       enc_key2 : in SLV_32;
                       dec_key1 : in SLV_32;
                       dec_key2 : in SLV_32;
                       enc_decb : in std_logic;
                signal dataout  : out SLV_128 );


function POST_WHITEN_FUNCT ( datain   : in SLV_128;
                             enc_key1 : in SLV_32;
                             enc_key2 : in SLV_32;
                             dec_key1 : in SLV_32;
                             dec_key2 : in SLV_32;
                             enc_decb : in std_logic )
                             return SLV_128;
    
procedure POST_WHITEN ( datain   : in SLV_128;
                        enc_key1 : in SLV_32;
                        enc_key2 : in SLV_32;
                        dec_key1 : in SLV_32;
                        dec_key2 : in SLV_32;
                        enc_decb : in std_logic;
                 signal dataout  : out SLV_128 );


function COMPUTE_TU ( vector_in    : in SLV_32 )
                      return SLV_32;

function COMPUTE_ALG_A ( A        : in SLV_32;
                         t        : in SLV_32;
                         u        : in SLV_32;
                         S        : in SLV_32;
                         enc_decb : in std_logic )
                         return SLV_32;

function COMPUTE_ALG_C ( C        : in SLV_32;
                         t        : in SLV_32;
                         u        : in SLV_32;
                         S        : in SLV_32;
                         enc_decb : in std_logic )
                         return SLV_32;

function ADD32 ( A : in SLV_32;                    
                 B : in SLV_32 )
                 return SLV_32;


function RC6_ROUND_FUNCT ( datain   : in SLV_128;
                           enc_key1 : in SLV_32;
                           enc_key2 : in SLV_32;
                           dec_key1 : in SLV_32;
                           dec_key2 : in SLV_32;
                           enc_decb : in std_logic )
                           return SLV_128;

procedure RC6_ROUND ( datain   : in SLV_128;
                      enc_key1 : in SLV_32;
                      enc_key2 : in SLV_32;
                      dec_key1 : in SLV_32;
                      dec_key2 : in SLV_32;
                      enc_decb : in std_logic;
               signal dataout  : out SLV_128  ); 


-- ==========================================================================
-- ============== Declarations for the Key Schedule section =================
-- ==========================================================================

constant Pw                : SLV_32  := X"B7E15163";
constant Qw                : SLV_32  := X"9E3779B9";
constant HOLD              : integer := 0;
constant CV256_WIDTH       : integer := 256;
constant RUNUP_CYCLES      : integer := 1;
constant PIPE_STAGES       : integer := (LAST_ROUND + 1) / RUNUP_CYCLES;

constant NUM_RUNUP_ROUNDS  : integer := 132;   -- Total number of steps
constant NUM_ROUND_PER_CLK : integer := 1;
constant LAST_CVRUNUP_STEP : integer := NUM_RUNUP_ROUNDS/NUM_ROUND_PER_CLK;   
                                        -- Number of clocks for cv runup

type S_ARRAY_TYPE is array (0 to 43) of SLV_32;
type NEW_S_ARRAY_TYPE  is array (0 to 131) of SLV_32;
type L_ARRAY_TYPE is array (0 to 7) of SLV_32;
type L_PIPE_TYPE  is array (0 to 131) of L_ARRAY_TYPE; 
type S_PIPE_TYPE  is array (0 to PIPE_STAGES) of S_ARRAY_TYPE;
type S_MESH_TYPE  is array (0 to ((((22/RUNUP_CYCLES)+1)*44)-1)) of SLV_32;
type MOD6_TABLE_TYPE is array (0 to 179) of integer range 0 to 5;
type MOD_DEC3_TABLE_TYPE is array (0 to 3) of integer range 0 to 3;
type MOD_DEC5_TABLE_TYPE is array (0 to 5) of integer range 0 to 5;
type MOD_DEC7_TABLE_TYPE is array (0 to 7) of integer range 0 to 7;

constant mod6_table : MOD6_TABLE_TYPE := ( 
0, 1, 2, 3, 4, 5,
0, 1, 2, 3, 4, 5,
0, 1, 2, 3, 4, 5,
0, 1, 2, 3, 4, 5,
0, 1, 2, 3, 4, 5,
0, 1, 2, 3, 4, 5,
0, 1, 2, 3, 4, 5,
0, 1, 2, 3, 4, 5,
0, 1, 2, 3, 4, 5,
0, 1, 2, 3, 4, 5,
0, 1, 2, 3, 4, 5,
0, 1, 2, 3, 4, 5,
0, 1, 2, 3, 4, 5,
0, 1, 2, 3, 4, 5,
0, 1, 2, 3, 4, 5,
0, 1, 2, 3, 4, 5,
0, 1, 2, 3, 4, 5,
0, 1, 2, 3, 4, 5,
0, 1, 2, 3, 4, 5,
0, 1, 2, 3, 4, 5, 
0, 1, 2, 3, 4, 5,
0, 1, 2, 3, 4, 5,
0, 1, 2, 3, 4, 5,
0, 1, 2, 3, 4, 5,
0, 1, 2, 3, 4, 5,
0, 1, 2, 3, 4, 5,
0, 1, 2, 3, 4, 5,
0, 1, 2, 3, 4, 5,
0, 1, 2, 3, 4, 5,
0, 1, 2, 3, 4, 5 );

constant mod_dec3_table : MOD_DEC3_TABLE_TYPE := ( 3, 0, 1, 2 );
constant mod_dec5_table : MOD_DEC5_TABLE_TYPE := ( 5, 0, 1, 2, 3, 4 );
constant mod_dec7_table : MOD_DEC7_TABLE_TYPE := ( 7, 0, 1, 2, 3, 4, 5, 6 );

subtype ARRAY_INDEX_TYPE is integer range 0 to 63;
subtype CV_TYPE          is std_logic_vector(255 downto 0);
subtype LATCH_TYPE       is std_logic_vector(0 to (22+RUNUP_CYCLES));

function COMPUTE_KS_A (current_S : SLV_32;
  		       current_A : SLV_32;
 		       current_B : L_ARRAY_TYPE;
                       v         : SLV_8;
                       cv_size   : SLV_2)
                       return SLV_32;

function COMPUTE_KS_B (current_L : L_ARRAY_TYPE;
                       updated_A : SLV_32;          -- from UPDATE_A function
                       current_B : L_ARRAY_TYPE;
                       v         : SLV_8;
                       cv_size   : SLV_2)
                       return L_ARRAY_TYPE;
            
end rc6_pack;


-- ==========================================================================
-- ==========================================================================
-- ==========================================================================


package body rc6_pack is


-- ==========================================================================
-- ===================== Definitions for general use ========================
-- ==========================================================================

-- ==========================================================================
--
--  procedure BYTESWAP128
--
--  This function swaps bytes in a 32-bit word where the input word is
--  four 32-bit words (128 bits total). For each 32 bit word the byte
--  swapping is performed as follows:
--
--     ABCD  -->  DCBA 
--
-- ==========================================================================

procedure BYTESWAP128( datain  : in SLV_128;
                signal dataout : out SLV_128 ) is

begin

  BYTESWAP32( datain(127 downto 96), dataout(127 downto 96) );
  BYTESWAP32( datain(95 downto 64),  dataout(95 downto 64) );
  BYTESWAP32( datain(63 downto 32),  dataout(63 downto 32) );
  BYTESWAP32( datain(31 downto 0),   dataout(31 downto 0) );


end BYTESWAP128;



-- ==========================================================================
--
--  procedure BYTESWAP32
--
--  This function swaps bytes in a 32-bit word where the input word is
--  a single 32-bit word. The byte swapping is performed as follows:
--
--     ABCD  -->  DCBA 
--
-- ==========================================================================

procedure BYTESWAP32( datain  : in SLV_32;
               signal dataout : out SLV_32 ) is

begin

  dataout <= datain(7 downto 0)   & datain(15 downto 8) &
             datain(23 downto 16) & datain(31 downto 24);

end BYTESWAP32;



-- ==========================================================================
-- ============= Definitions for the Encrypt/Decrypt section ================
-- ==========================================================================


-- ==========================================================================
--
--  function COMPUTE_TU 
--
-- ==========================================================================

function COMPUTE_TU (vector_in : in SLV_32 )
                     return SLV_32 is

-- pragma map_to_operator COMPUTE_TU_dw_op
-- pragma return_port_name COMPUTE_TU_out

variable temp    : unsigned(63 downto 0);
variable temp2   : unsigned(63 downto 0);
variable result  : SLV_32;
variable number  : integer;
variable number2 : unsigned(31 downto 0);

begin

   for index in 31 downto 0 loop
      number2(index) := vector_in(index);
   end loop;

   temp(63 downto 0) := (((number2) * 2) + 1 );
   temp2             := (temp(31 downto 0) * number2);
   result            := std_logic_vector(temp2(31 downto 0) rol 5);

return result;

end COMPUTE_TU;


-- ==========================================================================
--
--  function COMPUTE_ALG_C 
--
-- ==========================================================================

function COMPUTE_ALG_C (C        : in SLV_32;
                        t        : in SLV_32;
                        u        : in SLV_32;
                        S        : in SLV_32;
                        enc_decb : in std_logic)
                        return SLV_32 is

-- pragma map_to_operator COMPUTE_ALG_C_dw_op
-- pragma return_port_name COMPUTE_ALG_C_out

variable result          : SLV_32;
variable xor_result      : SLV_32;
variable subtract_result :SLV_32;

begin

   if enc_decb ='1' then

      xor_result := C xor u;
      result     := std_logic_vector((unsigned(xor_result) rol
                    TO_INTEGER(unsigned(t(4 downto 0)))) + unsigned(S) );

   else

      subtract_result := std_logic_vector((unsigned(C) - unsigned(S)) ror
                         TO_INTEGER(unsigned(t(4 downto 0))));
      result          := subtract_result xor u;

   end if;

return result;

end COMPUTE_ALG_C;


-- ==========================================================================
--
--  function ADD32 
--
-- ==========================================================================

function ADD32 (A : in SLV_32;                    
                B : in SLV_32)
                return SLV_32 is 

-- pragma map_to_operator ADD32_dw_op
-- pragma return_port_name ADD32_out

variable C : SLV_32;

begin

   C := std_logic_vector(unsigned(A) + unsigned(B));

return C;

end ADD32;


-- ==========================================================================
--
--  function PRE_WHITEN_FUNCT
--
--  Performs the preadd for the input data
--
-- ==========================================================================

function PRE_WHITEN_FUNCT (datain   : in SLV_128;
                           enc_key1 : in SLV_32;
                           enc_key2 : in SLV_32;
                           dec_key1 : in SLV_32;
                           dec_key2 : in SLV_32;
                           enc_decb : in std_logic )
                           return SLV_128 is
    
-- pragma map_to_operator PRE_WHITEN_FUNCT_dw_op
-- pragma return_port_name PRE_WHITEN_FUNCT_out

variable dataout : SLV_128;

begin

   if enc_decb = '1' then       -- encrypt case

      dataout(127 downto 96) := datain(127 downto 96);
      dataout(95 downto 64)  := std_logic_vector(unsigned(datain(95 downto 64))
                                + unsigned(enc_key1));    -- B + S(0)
      dataout(63 downto 32)  := datain(63 downto 32);
      dataout(31 downto 0)   := std_logic_vector(unsigned(datain(31 downto 0))
                                + unsigned(enc_key2));    -- D + S(1)

   else                         -- decrypt case

      dataout(127 downto 96) := std_logic_vector(unsigned(datain(127 downto 96))
                                - unsigned(dec_key2));    -- A - S(42)
      dataout(95 downto 64)  := datain(95 downto 64);
      dataout(63 downto 32)  := std_logic_vector(unsigned(datain(63 downto 32))
                                - unsigned(dec_key1));    -- C - S(43)
      dataout(31 downto 0)   := datain(31 downto 0);

   end if;

return dataout;

end PRE_WHITEN_FUNCT;


-- ==========================================================================
--
--  procedure PRE_WHITEN
--
--  Performs the pre-whitening on a 128-bit word
--
-- ==========================================================================

procedure PRE_WHITEN ( datain   : in SLV_128;
                       enc_key1 : in SLV_32;
                       enc_key2 : in SLV_32;
                       dec_key1 : in SLV_32;
                       dec_key2 : in SLV_32;
                       enc_decb : in std_logic;
                signal dataout  : out SLV_128 ) is


begin

   dataout <= PRE_WHITEN_FUNCT (datain, enc_key1, enc_key2,
                                dec_key1, dec_key2, enc_decb); 

end PRE_WHITEN;


-- ==========================================================================
--
--  function POST_WHITEN_FUNCT
--
--  Performs the postadd for the output data
--
-- ==========================================================================

function POST_WHITEN_FUNCT (datain   : in SLV_128;
                            enc_key1 : in SLV_32;
                            enc_key2 : in SLV_32;
                            dec_key1 : in SLV_32;
                            dec_key2 : in SLV_32;
                            enc_decb : in std_logic )
                            return SLV_128 is
    
-- pragma map_to_operator POST_WHITEN_FUNCT_dw_op
-- pragma return_port_name POST_WHITEN_FUNCT_out

variable dataout : SLV_128;

begin

   if enc_decb = '0' then             -- decrypt case

      dataout(127 downto 96) := datain(127 downto 96);
      dataout(95 downto 64)  := std_logic_vector(unsigned(datain(95 downto 64))
                                - unsigned(dec_key2));    -- B - S(0)
      dataout(63 downto 32)  := datain(63 downto 32);
      dataout(31 downto 0)   := std_logic_vector(unsigned(datain(31 downto 0))
                                - unsigned(dec_key1));    -- D - S(1)

   else                               -- encrypt case

      dataout(127 downto 96) := std_logic_vector(unsigned(datain(127 downto 96))
                                + unsigned(enc_key1));    -- A + S(42)
      dataout(95 downto 64)  := datain(95 downto 64);
      dataout(63 downto 32)  := std_logic_vector(unsigned(datain(63 downto 32))
                                + unsigned(enc_key2));    -- C + S(43)
      dataout(31 downto 0)   := datain(31 downto 0);

   end if;

return dataout;

end POST_WHITEN_FUNCT;


-- ==========================================================================
--
--  procedure POST_WHITEN
--
--  Performs the post-whitening on a 128-bit word
--
-- ==========================================================================

procedure POST_WHITEN ( datain   : in SLV_128;
                        enc_key1 : in SLV_32;
                        enc_key2 : in SLV_32;
                        dec_key1 : in SLV_32;
                        dec_key2 : in SLV_32;
                        enc_decb : in std_logic;
                 signal dataout  : out SLV_128 ) is


begin

   dataout <= POST_WHITEN_FUNCT (datain, enc_key1, enc_key2,
                                 dec_key1, dec_key2, enc_decb); 

end POST_WHITEN;


-- ==========================================================================
--
--  function RC6_ROUND
--
--  Performs one round of the RC6 block cipher. Encryption or decryption
--  is performed based on the 'encrypt' signal.
--
-- ==========================================================================

function RC6_ROUND_FUNCT (datain   : in SLV_128;
                          enc_key1 : in SLV_32;
                          enc_key2 : in SLV_32;
                          dec_key1 : in SLV_32;
                          dec_key2 : in SLV_32;
                          enc_decb : in std_logic )
                          return SLV_128 is

-- pragma map_to_operator RC6_ROUND_FUNCT_dw_op
-- pragma return_port_name RC6_ROUND_FUNCT_out

variable dataout : SLV_128;
variable temp    : SLV_128;
variable A       : SLV_32;
variable C       : SLV_32;
variable T       : SLV_32;
variable B       : SLV_32;
variable D       : SLV_32;
variable U       : SLV_32;
variable data_fixed :SLV_128;
variable key1       : SLV_32;
variable key2       : SLV_32;

begin

-- ===========================================================================
-- ============================== Encryption =================================
-- ===========================================================================

   if enc_decb = '1' then

      data_fixed := datain;
      key1       := enc_key1;
      key2       := enc_key2;

   else

-- ===========================================================================
-- ============================== Decryption =================================
-- ===========================================================================

      data_fixed := datain(31 downto 0) &
                    datain(127 downto 96) &
                    datain(95 downto 64) &
                    datain(63 downto 32);
      key1       := dec_key2;
      key2       := dec_key1;

   end if;

-- ===========================================================================
-- ==============================  Function  =================================
-- ===========================================================================

      T := COMPUTE_TU(data_fixed(95 downto 64));
      U := COMPUTE_TU(data_fixed(31 downto 0));
      A := COMPUTE_ALG_A(data_fixed(127 downto 96), T, U, key1, enc_decb);
      C := COMPUTE_ALG_C(data_fixed(63 downto 32), T, U, key2, enc_decb);
      B := data_fixed(95 downto 64);
      D := data_fixed(31 downto 0);

   if enc_decb = '1' then

      dataout := B & C & D & A;

   else

      dataout := A & B & C & D;
 
   end if;


return dataout;

end RC6_ROUND_FUNCT;


-- ==========================================================================
--
--  procedure RC6_ROUND
--
--  Performs one round of the RC6 block cipher. Encryption or decryption
--  is performed based on the 'encrypt' signal.
--  For encryption, the last round replaces the linear transform with
--  a key addition step using key2; all rounds use key1.
--  For decryption, the first round replaces the inverse linear transform
--  with a key addition step using key1; all rounds use key2.
--  Result of the round is returned on dataout.
--
-- ==========================================================================

procedure RC6_ROUND (datain   : in SLV_128;
                     enc_key1 : in SLV_32;
                     enc_key2 : in SLV_32;
                     dec_key1 : in SLV_32;
                     dec_key2 : in SLV_32;
                     enc_decb : in std_logic;
              signal dataout  : out SLV_128  ) is 

begin

   dataout <= RC6_ROUND_FUNCT (datain, enc_key1, enc_key2,
                               dec_key1, dec_key2, enc_decb);

end RC6_ROUND;


-- ==========================================================================
--                 Definitions for the Key Schedule section                --
-- ==========================================================================

-- ==========================================================================
--
--  function COMPUTE_KS_A 
--
-- ==========================================================================

function COMPUTE_KS_A (current_S : SLV_32;
  		       current_A : SLV_32;
 		       current_B : L_ARRAY_TYPE;
                       v         : SLV_8;
                       cv_size   : SLV_2)
                       return SLV_32 is

-- pragma map_to_operator COMPUTE_KS_A_dw_op
-- pragma return_port_name COMPUTE_KS_A_out

variable temp   : unsigned(31 downto 0);
variable result : unsigned(31 downto 0);
variable j             : INTEGER;

begin

 
case cv_size is

when "00" => j := mod_dec3_table(TO_INTEGER(UNSIGNED (v)) mod 4);
when "01" => j := mod_dec5_table(mod6_table(TO_INTEGER(UNSIGNED (v))));
when others => j := mod_dec7_table(TO_INTEGER(UNSIGNED (v)) mod 8);

end case;

   temp   := unsigned(current_S) + unsigned(current_A) + 
             unsigned(current_B(j));  

   result := (temp) rol 3;

   return std_logic_vector(result);

end COMPUTE_KS_A;


-- ==========================================================================
--
--  function COMPUTE_KS_B 
--
-- ==========================================================================

function COMPUTE_KS_B (current_L : L_ARRAY_TYPE;
                       updated_A : SLV_32;      -- from UPDATE_A function
                       current_B : L_ARRAY_TYPE;
                       v         : SLV_8;
                       cv_size   : SLV_2)
                       return L_ARRAY_TYPE is

-- pragma map_to_operator COMPUTE_KS_B_dw_op
-- pragma return_port_name COMPUTE_KS_B_out

variable first_part    : SLV_32;
variable rotate_amount : SLV_32; 
variable result        : SLV_32; 
variable j             : INTEGER;
variable L_MATRIX      : L_ARRAY_TYPE;
variable v_temp        : INTEGER;

begin

L_MATRIX := current_L;

case cv_size is

when "00" =>  v_temp := TO_INTEGER(UNSIGNED (v)) mod 4;
              j := mod_dec3_table(v_temp);

when "01" =>  v_temp := mod6_table(TO_INTEGER(UNSIGNED (v)));
              j := mod_dec5_table(v_temp);

when others => v_temp := TO_INTEGER(UNSIGNED (v)) mod 8;
               j := mod_dec7_table(v_temp);

end case;


   first_part    := std_logic_vector(unsigned(current_L(v_temp)) 
                     + unsigned(updated_A) 
                     + unsigned(current_B(j)));

   rotate_amount := std_logic_vector(unsigned(updated_A)
                     + unsigned(current_B(j)));

   result        := std_logic_vector(unsigned(first_part) rol
                    TO_INTEGER(unsigned(rotate_amount(4 downto 0))) );

L_MATRIX(v_temp) := result;

return L_MATRIX;

end COMPUTE_KS_B;


-- ==========================================================================
--
--  function COMPUTE_ALG_A 
--
-- ==========================================================================

function COMPUTE_ALG_A (A        : in SLV_32;
                        t        : in SLV_32;
                        u        : in SLV_32;
                        S        : in SLV_32;
                        enc_decb : in std_logic )
                        return SLV_32 is

-- pragma map_to_operator COMPUTE_ALG_A_dw_op
-- pragma return_port_name COMPUTE_ALG_A_out

variable result          : SLV_32;
variable xor_result      : SLV_32;
variable subtract_result : SLV_32;

begin

   if enc_decb = '1' then

      xor_result := A xor t;
      result     := std_logic_vector((unsigned(xor_result) rol
                    TO_INTEGER(unsigned(u(4 downto 0)))) + unsigned(S) );

   else

      subtract_result := std_logic_vector((unsigned(A)  - unsigned(S)) ror
                         TO_INTEGER(unsigned(u(4 downto 0))) );
      result          := subtract_result xor t;

   end if;


return result;

end COMPUTE_ALG_A;


-- ==========================================================================


end rc6_pack;
