-- *************************************************************************
-- DISCLAIMER. THIS SOFTWARE WAS WRITTEN BY EMPLOYEES OF THE U.S.
-- GOVERNMENT AS A PART OF THEIR OFFICIAL DUTIES AND, THEREFORE, IS NOT
-- PROTECTED BY COPYRIGHT. HOWEVER, THIS SOFTWARE CODIFIES THE FINALIST
-- CANDIDATE ALGORITHMS (i.e., MARS, RC6tm, RIJNDAEL, SERPENT, AND
-- TWOFISH) IN THE ADVANCED ENCRYPTION STANDARD (AES) DEVELOPMENT EFFORT
-- SPONSORED BY THE NATIONAL INSTITUTE OF STANDARDS AND TECHNOLOGY (NIST)
-- AND MAY BE PROTECTED BY ONE OR MORE FORMS OF INTELLECTUAL PROPERTY. THE
-- U.S. GOVERNMENT MAKES NO WARRANTY, EITHER EXPRESSED OR IMPLIED,
-- INCLUDING BUT NO LIMITED TO ANY IMPLIED WARRANTIES OF MERCHANTABILITY
-- OR FITNESS FOR A PARTICULAR PURPOSE, REGARDING THIS SOFTWARE. THE U.S.
-- GOVERNMENT FURTHER MAKES NO WARRANTY THAT THIS SOFTWARE WILL NOT
-- INFRINGE ANY OTHER UNITED STATES OR FOREIGN PATENT OR OTHER
-- INTELLECTUAL PROPERTY RIGHT. IN NO EVENT SHALL THE U.S. GOVERNMENT BE
-- LIABLE TO ANYONE FOR COMPENSATORY, PUNITIVE, EXEMPLARY, SPECIAL,
-- COLLATERAL, INCIDENTAL, CONSEQUENTIAL, OR ANY OTHER TYPE OF DAMAGES IN
-- CONNECTION WITH OR ARISING OUT OF COPY OR USE OF THIS SOFTWARE.
-- *************************************************************************
-- ===========================================================================
-- ===========================================================================
-- File Name : alg_pipe.vhdl
-- Author    : NSA
-- Date      : December 99
-- Project   : AES Candidate Evaluation - MARS
-- Purpose   : This block implements the MARS algorithm for a pipelined case
-- Notes     :
-- ===========================================================================

library ieee;
use ieee.std_logic_1164.all;
use ieee.numeric_std.all;
use WORK.mars_pack.all;

-- ===========================================================================
-- =========================== Interface Description =========================
-- ===========================================================================

entity ALG_PIPE is

  port (clock        :  in STD_LOGIC;      -- clock signal
        reset        :  in STD_LOGIC;      -- active high reset (asynchronous)

        ALG_DATAIN   :  in SLV_128;        -- input data
        ALG_KEY_ENC  :  in K_ARRAY_TYPE;   -- array of encrypt sub-keys (Si)
        ALG_KEY_DEC  :  in K_ARRAY_TYPE;   -- array of decrypt sub-keys (Si)
        ALG_ENC      :  in std_logic;      -- (1 = encrypt, 0 = decrypt)

        ALG_DATAOUT  :  out SLV_128        -- Processed output data
  );

end ALG_PIPE;

architecture ALG_PIPE_RTL of ALG_PIPE is


-- ===========================================================================
-- ========================== Component Definition ===========================
-- ===========================================================================

component REG128B 

  port (clock    :  in std_logic;   -- clock signal
        reset    :  in std_logic;   -- active high reset (asynchronous)

        DATA_IN  :  in SLV_128;     -- input data bus
        DATA_OUT :  out SLV_128     -- output data bus

  );

end component;

-- ===========================================================================
-- =========================== Signal Definition =============================
-- ===========================================================================

signal ALG_DATA        : PIPE_DATA_TYPE;  -- register output
signal ALG_DATA_INT    : PIPE_DATA_TYPE;  -- procedure call to reg interconnect
signal ROUND_STEP      : ROUND_TYPE;      -- indicates current algorithm step


begin

-- ===========================================================================
-- ======================== Generate Pipe Structure ==========================
-- ===========================================================================
--
-- PURPOSE:  
--
-- The following generate statements create a pipelined architecture for 
-- the encryption/decryption.


GEN: for ROUND_STEP in FIRST_ROUND to LAST_ROUND generate

  --ROUND 0
  R0: if ROUND_STEP = FIRST_ROUND generate

  PRE_ADD : KEY_ADD ( ALG_DATAIN,
                         ALG_KEY_ENC(0),
                         ALG_KEY_ENC(1),
                         ALG_KEY_ENC(2),
                         ALG_KEY_ENC(3),
                         ALG_KEY_DEC(3),
                         ALG_KEY_DEC(2),
                         ALG_KEY_DEC(1),
                         ALG_KEY_DEC(0),
                         ALG_ENC,
                         ALG_DATA_INT(0) ); --step 2-3 on pg 17


  DATAREG0 : REG128B port map ( clock,
                                reset,
                                ALG_DATA_INT(0),   -- interconnect
                                ALG_DATA(0) );     -- output data
  end generate;  -- R0


  R1_8: if ( ROUND_STEP >= 1 and ROUND_STEP <= 8 ) generate

    MIX1_5 : FORWARD_UNKEYED_ROUND ( ALG_DATA(ROUND_STEP - 1),
                                     ALG_ENC,
                                     std_logic_vector(TO_UNSIGNED(ROUND_STEP,6)),
                         ALG_DATA_INT(ROUND_STEP));

    DATAREG1_5 : REG128B port map ( clock,
                                reset,
                                ALG_DATA_INT(ROUND_STEP),   -- interconnect
                                ALG_DATA(ROUND_STEP) );     -- output data

  end generate;  -- R1_8



  R9_16: if ( ROUND_STEP >= 9 and ROUND_STEP <= 16 ) generate

    FWD_XFORM : FORWARD_KEYED_ROUND ( ALG_DATA(ROUND_STEP - 1),
                                ALG_KEY_ENC(2*ROUND_STEP - 14),
                                ALG_KEY_ENC(2*ROUND_STEP - 13), 
                                ALG_KEY_DEC(2*ROUND_STEP - 13),
                                ALG_KEY_DEC(2*ROUND_STEP - 14),
                                ALG_ENC,
                                std_logic_vector(TO_UNSIGNED(ROUND_STEP,6)),        
                                ALG_DATA_INT(ROUND_STEP) );
                                  
    FWD_DATAREG : REG128B port map ( clock,
                                reset,
                                ALG_DATA_INT(ROUND_STEP),   -- interconnect
                                ALG_DATA(ROUND_STEP) );     -- output data

  end generate;  -- R9_16


  R17_24: if ( ROUND_STEP >= 17 and ROUND_STEP <= 24 ) generate

    BCK_XFORM : BACKWARD_KEYED_ROUND ( ALG_DATA(ROUND_STEP - 1),
                                ALG_KEY_ENC(2*ROUND_STEP - 14),
                                ALG_KEY_ENC(2*ROUND_STEP - 13), 
                                ALG_KEY_DEC(2*ROUND_STEP - 13),
                                ALG_KEY_DEC(2*ROUND_STEP - 14),
                                ALG_ENC,
                                ALG_DATA_INT(ROUND_STEP) );
                                  
    BCK_DATAREG : REG128B port map ( clock,
                                reset,
                                ALG_DATA_INT(ROUND_STEP),   -- interconnect
                                ALG_DATA(ROUND_STEP) );     -- output data

  end generate;  -- R17_24


  R25_32: if ( ROUND_STEP >= 25 and ROUND_STEP <= 32 ) generate

    MIX25_32 : BACKWARD_UNKEYED_ROUND ( ALG_DATA(ROUND_STEP - 1),
                                     ALG_ENC,
                                    
                                     std_logic_vector(TO_UNSIGNED(ROUND_STEP,6)),
                                     ALG_DATA_INT(ROUND_STEP) );

    DATAREG25_32 : REG128B port map ( clock,
                                reset,
                                ALG_DATA_INT(ROUND_STEP),   -- interconnect
                                ALG_DATA(ROUND_STEP) );     -- output data
  end generate;  -- R25_32



  R33: if (ROUND_STEP = LAST_ROUND) generate

      POST_ADD : KEY_SUB( ALG_DATA(ROUND_STEP-1),
                          ALG_KEY_ENC(ROUND_STEP + 3),--using keys 39-36
                          ALG_KEY_ENC(ROUND_STEP + 4),
                          ALG_KEY_ENC(ROUND_STEP + 5),
                          ALG_KEY_ENC(ROUND_STEP + 6),
                          ALG_KEY_DEC(ROUND_STEP + 3),--using keys 39-36
                          ALG_KEY_DEC(ROUND_STEP + 4),
                          ALG_KEY_DEC(ROUND_STEP + 5),
                          ALG_KEY_DEC(ROUND_STEP + 6),
                          ALG_ENC,                         
                          ALG_DATA_INT(ROUND_STEP) );

      ALG_DATA(ROUND_STEP) <= ALG_DATA_INT(ROUND_STEP)(31 downto 0) &
                              ALG_DATA_INT(ROUND_STEP)(63 downto 32) &
                              ALG_DATA_INT(ROUND_STEP)(95 downto 64) &
                              ALG_DATA_INT(ROUND_STEP)(127 downto 96)
        when ALG_ENC = '1' else ALG_DATA_INT(ROUND_STEP);

      DATAREG : REG128B port map ( clock,
                                   reset,
                                   ALG_DATA(ROUND_STEP),
                                   ALG_DATAOUT );         -- output data

   end generate;  -- R33

end generate;  -- GEN


end ALG_PIPE_RTL;

-- ===========================================================================
-- ============================ Configuration ================================
-- ===========================================================================

configuration CFG_ALG_PIPE of ALG_PIPE is

   for ALG_PIPE_RTL

   end for;

end CFG_ALG_PIPE;
