/* Copyright 2000 Enhanced Software Technologies Inc.
   All Rights Reserved

   Released for public use under a BSD-style license. See file LICENSE
   for details. 

   decrypt a file in CFB-128 mode using AES candidate "twofish".

  RCS CHANGE LOG:
$Log: aesget.c,v $
Revision 1.10  2000/10/23 17:03:43  randy
Changed to support only 128 bit AES keys until we can get time to
revisit the AES module and refit it to specify key lengths we use
from the ticket file.

Revision 1.9  2000/10/12 21:38:18  randy
Support for end-to-end client authorization (DB, remote agents)

Revision 1.8  2000/10/10 23:40:59  randy
Integrate the Rijndael algorithm in place of the Twofish algorithm for
our AES package.  Changed crypt and get to no longer call the Twofish
specific (but much more efficient) reKey subroutine.  Ah, well, it's not
like we're running on Apple ][s anymore, eh?

Revision 1.7  2000/10/06 00:47:42  eric
Fixed key initialization error.

Revision 1.6  2000/08/03 22:05:34  eric
Added ability to accept '-' as arg to -k to indicate that the password
is the first 33 chars of stdin (32 hex chars + terminator). Removed
reference to Dr. Gladman's encryption routines, added reference to
Counterpane's encryption routines.

Revision 1.5  2000/05/05 20:56:49  eric
Modifications for getting 'kk=' string out of the server config files.

Revision 1.4  2000/04/24 19:46:57  eric
More autoconf tweaks. Changed to using Counterpane's TwoFish rather than
Dr. Gladman's TwoFish in an attempt to fix the big endian/little endian
problems once and for all.

Revision 1.1  2000/03/28 23:54:28  eric
Initial checkin -- aescrypt/aesget

$Date: 2000/10/23 17:03:43 $


*/
#include "config.h"

#include <stdio.h>
#include <errno.h>
#include <string.h>
#include <unistd.h>

#include "aes.h"
#include "bin2hex.h"
#include "dstring.h"

static char *progname;

/* static unsigned char *key; /* the key to the program... */
keyInstance key;
cipherInstance cipher;


static int keysize;        /* and how big it is. */
static int cfb128_idx=-1;

static char cfb_blk[16];
static char cfb_crypt[16];


static void usage(void) {
  fprintf(stderr,"%s:Usage: %s -k {keyfile}\n",progname,progname);
  exit(1);
}


/* write data, aborting if we have a write error. */

static void ewrite(int outfile,void *buf, size_t count) {
  int result;

  result=write(outfile,buf,count);
  if (result!=(int) count) {
    fprintf(stderr,"%s:error:could not write data.\n",progname);
    perror(progname);
    usage();
  }
}

static void decryptblock(char *src, int srclen) {
  int i,ch;

  /* Now to encrypt each individual character in cfb mode: */
  for (i=0; i<srclen; i++) {
    if (cfb128_idx < 0 || cfb128_idx > 15) {
      blockEncrypt(&cipher,&key,cfb_blk,128,cfb_crypt);

      /* encrypt(cfb_blk,cfb_crypt); */
      cfb128_idx=0;
    }
    ch=src[i];
    src[i]=ch ^ cfb_crypt[cfb128_idx]; 
    /* do output feedback: put crypted byte into next block to be crypted */
    cfb_blk[cfb128_idx++]=ch; 
  }

}


static void decryptfile(int infile,int outfile) {
  int result;

#define CRYPTBUF_SIZE 262144
  char buf[CRYPTBUF_SIZE];

  int bufsize;  /* how many chars are currently in that buffer. */


  result=read(infile,cfb_blk,16);  /* set feedback buffer to salt. */
  if (result<16) {
    fprintf(stderr,"%s:error:could not get salt from input\n",progname);
    perror(progname);
    usage();
  }

  /* Now initialize the cipher: */
  if (cipherInit(&cipher,MODE_ECB,NULL) != TRUE) {
    fprintf(stderr,"%s:Could not initialize cipher.\n",progname);
    usage();
  }

  while (bufsize=read(infile,buf,CRYPTBUF_SIZE), bufsize != 0) {
    if (bufsize<0) { /* some sort of I/O error... */
      fprintf(stderr,"%s:Error on read.\n",progname);
      perror(progname);
      exit(1);
    }

    decryptblock(buf,bufsize);
    ewrite(outfile,buf,bufsize); /* checks for error... */
  }

  /* okay, we've done everything... */
  
  return;
}
  
/* Routine to set the key once we've read it. */
static void aes_set_key(char *inkey) {
  int keylen,rtncode;
  char *hexkey;

  hexkey=d_trim(inkey); 

  keylen=strlen(hexkey);
  /* we're only going to support 128 bit AES keys for now. */
  /* if it's a long key, chop it! */
  if ( keylen > 32 ) {
    keylen = 32;  /* chop! */
    hexkey[32]=0; 
  }    
    
  if ( keylen==32) {
    keysize=keylen>>1;
        
    if ((rtncode = makeKey(&key,DIR_ENCRYPT,keylen*4,hexkey)) !=TRUE) {
      fprintf(stderr,"%s: reported from makeKey: Key format error:%i\n",progname,rtncode);
      usage();
    }

    /* Now copy the actual binary key, sigh. */
    /* memcpy(key.key32, binkey, keylen); */

    /* reKey(&key);    run the key schedule. */
   /* Previous lines commented out because Rijndael doesn't have this 
       simpler interface :( NIST = Numbskull Ignoramuses Suck Thouroghly. 
       after boning up on the relative merits of the finalist algorithms, 
       I have the sneaking suspicion that Twofish will be back. I think NIST
       picked Rijndael 'cause the NSA(=No Security in America) already has 
       an exploit for it.
       
       Randy will get off his soapbox now.  */  
    
    return;
  }
    
  fprintf(stderr,"%s:Key size error:%d[%s]\n",progname,keylen,hexkey);
  usage();
}



/* Routine to read a 128-bit key from stdin, as a 32-byte hex
 * null-terminated string:
 */
static void read_key_from_stdin(int infile) {
  unsigned char keybuf[33];
  int result;

  result=read(infile,keybuf,33);
  if (result<33) {
    fprintf(stderr,"%s: Error: Key not find on stdin.\n",progname);
    usage();
  }
  keybuf[32]=0; /* make sure it is null terminated! */
  aes_set_key(keybuf);
  return;
}

  
/* Routine to read a key from keyfile. */
static void read_key(char *filename) {
  char inbuf[1024];
  FILE *infile;
  char *result;

  infile=fopen(filename,"r");
  if (!infile) {
    fprintf(stderr,"%s:Error: Key file '%s' not found.\n",progname,filename);
    usage();
  }

  while (result=fgets(inbuf,1023,infile), result != NULL) {
    if ((strncmp(result,"kk=",3)==0) || strncmp(result,"kk:",3) == 0) {
      aes_set_key(result+3);
      return;
    }
  }

  fprintf(stderr,"%s: Error: Key not found in file '%s'.\n",progname,filename);
  usage();
}
  


int main(int argc, char **argv) {

  progname= argv[0];
  if (argc != 3) {
    usage();
  }

  if (strcmp(argv[1],"-k")) {
    usage();
    exit(1);
  }

  if (strcmp(argv[2],"-")==0) {
    read_key_from_stdin(fileno(stdin));
  } else {
    read_key(argv[2]); /* read the keyfile, hopefully! */
  }

  decryptfile(fileno(stdin),fileno(stdout));
  
  exit(0);

}
