


/* 
  Copyright 2001/2002 Markus Hahn <markus_hahn@gmx.net>
  All rights reserved.
*/



using System;
using System.Text;

using Blowfish_NET;


/// <summary>
///   some Blowfish test class
/// </summary>
class BlowfishTest
{

  static byte[] StringToBlocks
    (String sText)
  {
    int nI, nR = sText.Length % Blowfish.BLOCKSIZE;
    if (0 != nR) 
    {
      for (nI = 0; nI < (Blowfish.BLOCKSIZE - nR); nI++) sText += '.';
    }

    System.Console.WriteLine(sText);
    
    ASCIIEncoding ascEnc = new ASCIIEncoding();

    return ascEnc.GetBytes(sText);
  }

  static String BlocksToString
    (byte[] data)
  {
    ASCIIEncoding ascEnc = new ASCIIEncoding();

    return ascEnc.GetString(data);
  }



  const int BIGBUFDIM = 100000;
  const int TESTLOOPS = 2000;
  

  static void TestBlowfish()
  {
    if (!Blowfish.SelfTest())
    {
      System.Console.WriteLine("selftest failed.");
      return;
    }

    System.Console.WriteLine("selftest passed.");

    byte[] key = new byte[16];

    for (byte bI = 0; bI < key.Length; bI++) key[bI] = bI; 

    Blowfish bf = new Blowfish(key);

    String sTest = "this is something to encrypt";

    System.Console.WriteLine(sTest);
    
    byte[] plainText = StringToBlocks(sTest);

    byte[] cipherText = new byte[plainText.Length];

    bf.Encrypt(plainText, cipherText, 0, 0, plainText.Length);

    System.Console.WriteLine(BlocksToString(cipherText));

    bf.Decrypt(cipherText, cipherText, 0, 0, cipherText.Length);

    System.Console.WriteLine(BlocksToString(cipherText));     

    int nI, nSize = Blowfish.BLOCKSIZE * BIGBUFDIM;
    byte[] bigBuf = new byte[nSize];
    for (nI = 0; nI < nSize; nI++) bigBuf[nI] = (byte)nI;

    System.Console.WriteLine("benchmark running ...");     
  
    long lTm = DateTime.Now.Ticks;

    for (nI = 0; nI < TESTLOOPS; nI++)  
    {
      bf.Encrypt(bigBuf, bigBuf, 0, 0, nSize);

      if ((nI & 0x0f) == 0) System.Console.Write(".");
    }
      
    lTm = DateTime.Now.Ticks - lTm;

    lTm /= 10000;

    System.Console.WriteLine("\n{0} bytes in {1} millisecs", 
                             TESTLOOPS * nSize,
                             lTm);
         
    long lSize = (long)nSize * 1000 * TESTLOOPS;  
    lSize /= lTm;
    
    System.Console.WriteLine("(average of {0} bytes per second)", lSize);

    bf.Burn();
  }


  public static void TestBlowfishCBC()
  {
    String sDemo = "The Blowfish encryption algorithm was introduced in 1994.";

    System.Console.WriteLine(sDemo);

    byte[] ptext = StringToBlocks(sDemo);

    byte[] key = new byte[16];

    byte bI;
    for (bI = 0; bI < key.Length; bI++) key[bI] = bI; 

    byte[] iv = new byte[Blowfish.BLOCKSIZE];
    for (bI = 0; bI < iv.Length; bI++) iv[bI] = bI; 

    BlowfishCBC bfc = new BlowfishCBC(key, iv);

    byte[] ctext = new byte[ptext.Length];
 
    bfc.Encrypt(ptext, ctext, 0, 0, ptext.Length);

    System.Console.WriteLine(BlocksToString(ctext));

    bfc.Iv = iv;

    bfc.Decrypt(ctext, ctext, 0, 0, ctext.Length).Burn();
  
    System.Console.WriteLine(BlocksToString(ctext));
  }

  public static void TestBlowfishSimple()
  {
    String sKey = "psst, don't tell";

    BlowfishSimple bfs = new BlowfishSimple(sKey);

    String sKeyChecksum = bfs.KeyChecksum; 

    System.Console.WriteLine("the key checksum is \"{0}\"", sKeyChecksum);

    String sStr = "Hello, please make that a secret.";

    String sEnc = bfs.Encrypt(sStr);

    System.Console.WriteLine("<<<{0}>>>", sStr);
    System.Console.WriteLine(sEnc);

    bfs.Burn();
    bfs = null;

    System.Console.WriteLine(
      (BlowfishSimple.VerifyKey("hubba!", sKeyChecksum)) ? "?!?" : "as expected");

    System.Console.WriteLine(
      (BlowfishSimple.VerifyKey(sKey, sKeyChecksum)) ? "as expected" : "?!?");

    bfs = new BlowfishSimple(sKey);

    String sDec = bfs.Decrypt(sEnc);
    
    System.Console.WriteLine("<<<{0}>>>", sDec);

    bfs.Burn();
  }

  /// <summary>
  ///   application entry point
  /// </summary>
  public static void Main()
  {
    TestBlowfish();
    TestBlowfishCBC();
    TestBlowfishSimple();
  }
}




