/*
 * Interface to BSD-style Pseudo-ttys.
 *
 * You may want to lookup the pty ioctl's:
 *
 * HP-UX Only!:
 *     TIOCTRAP    - (disabled default) => no exceptional selects will happen.
 *                 = enabled => select detects open and close 
 *		     slave hangs on open or ioctl until master acks.
 *		     non-termio ioctl's detected.
 *		     termio ioctl's detected if TIOCTTY disabled.
 *     TIOCTTY     - enabled => termio ioctl's valid on slave
 *                              termio ioctl's detected only if TIOCMONITOR
 *                 - disabled => TIOCTRAP detects termio ioctl's.
 *     TIOCMONITOR - enabled  => termio ioctls detected if TIOCTRAP & TIOCTTY
 *
 *     if TIOCTRAP is enabled, then traps must be ack'd by a TIOCREQCHECK,
 *     TIOCREQSET pair in sequence.
 *
 *     TIOCSIGMODE - TIOCSIGBLOCK - Caught Slave signals are posponed on ioctl
 *                   TIOCSIGABORT - All slave signals return EINTR on ioctl
 *                   TIOSIGNORMAL - default, restarted requests => a new ioctl
 *
 * None of the above exists on any BSD implemenation I can find.  Some mention
 * is made of setting B0 baudrate causing the master to see a close.
 */
#define _POSIX_SOURCE
#include <unistd.h>	/* open close */
#include <stdlib.h>	/* free malloc */
#include <stdio.h>
#include <string.h>
#include <fcntl.h>
#include "pty.h"
#include "log.h"

extern int debug;

static char ptynum[] = "0123456789abcdef";
static char ptylet[] = "pqrs";
static char mname[]  = "/dev/ptyxx";
static char sname[]  = "/dev/ttyxx";

#define LOC	((sizeof mname) - 2)	/* index of last character */

/*
 * Find an available unused pty.  Open the master side and slave side.
 * Return the fd and names for the master and slave sides.
 *
 * The reasons you must open the slave are:
 *   - The open must be atomic to opening the master to prevent another
 *     process from grabbing the slave and stealing control.
 *
 *   - The slave open may be impossible because of permissions.  The slave
 *     pty changes ownership when login processes grab it.
 *
 * This can cause problems because the slave could become the new controlling
 * terminal for this process under certain conditions.
 * HP-UX9.0:pty(7)manPage:
 *
 *    The slave side of the pty interprets opening or closing the master
 *    side as a modem connection or disconnection on a real terminal.  Only
 *    one open to the master side of a pty is permitted.  An attempt to open
 *    an already open master side returns -1 and sets the external variable
 *    errno to EBUSY.  An attempt to open the master side of a pty that has
 *    a slave with an open file descriptor returns -1 and sets errno to
 *    EBUSY...An ioctl() request made on the slave side of a pty after
 *    the master side is closed returns -1 and sets the external variable
 *    errno to EIO.
 *
 * Ultrix:pty(4): The slave device can be opened multiple times, while the 
 *    master half can be opened only once.
 *
 * Thus, at the instant the open succeeds, we are guarenteed that the slave
 * side was not open by any other process, and we are the exclusive holder
 * of the open fd for the master side.  
 *
 * Two questions remain:  What happens if a read occurs on the master
 * before any process has opened the slave?  and, how can the master prevent
 * multiple opens on the slave?  On my machine, the only thing that prevents
 * a sufficiently determined process from opening the slave (even if it's 
 * already open by another process as the controlling terminal) is the 
 * permissions on the slave pty filename.
 *
 * The DEC alpha provides an openpty(3) and forkpty(3) call, but they are not 
 * available on HP-UX or Ultrix.
 */
int openPty(mode, slaveName, masterName, size, slaveFd)
   int mode;
   char *slaveName, *masterName;
   unsigned size;
   int *slaveFd;
{
   register char *letp = ptylet, *nump;
   int fd;

   do {
      mname[LOC-1] = sname[LOC-1] = *letp;
      nump = ptynum;
      do {
	 mname[LOC] = sname[LOC] = *nump;
	 fd = open(mname, mode, 0);
	 if (fd >= 0) {
	    if (slaveFd == (int *) 0) goto gotPty;
	    *slaveFd = open(sname, mode, 0);
	    if (*slaveFd >= 0) goto gotPty;
	    close(fd);
	 }
      } while (*++nump != '\0');
   } while (*++letp != '\0');

gotPty:
   if (fd >= 0) {
      if (masterName != (char *) 0) {
	 strncpy(masterName, mname, size);
      }
      if (slaveName != (char *) 0) {
	 strncpy(slaveName, sname, size);
      }
   }
   return fd;
}
