;************************************************************************************************/
;* Noekeon_ARM7.s
;*
;* Last Modified: 00/08/30             Created: 00/08/30
;*
;* Project    : Nessie Proposal: NOEKEON
;*
;* Authors    : Joan Daemen, Michael Peeters, Vincent Rijmen, Gilles Van Assche
;*
;* Written by : Michael Peeters
;*
;* References : [NESSIE] see http://cryptonessie.org
;*
;* Description: Optimised implementation on an ARM7 processor of NOEKEON in DIRECT KEY MODE
;*              timing-attack resistant    
;*              optimised for size
;*
;* Comments:
;*   Memory model is BIG ENDIAN: r0=0x12345678 
;*                               --> Write in memory at 0x1000:  0x1000-78 56 34 12
;*   PC=R15, LR=R14, SP=R13
;*   Stack when interfacing with c: Decreasing before PUSH, Increasing after POP
;*
;************************************************************************************************/

  EXPORT NESSIEencrypt
  EXPORT NESSIEdecrypt
  EXPORT NESSIEkeysetup

  AREA ASM_Noekeon_Code, CODE, READONLY

NROUND			EQU 16				; Number of Computation rounds in the block cipher

enccstad		DCD	enccst               
enccst			DCB	0x80, 0x1b, 0x36, 0x6c	; Value of constant rounds
				DCB	0xd8, 0xab, 0x4d, 0x9a
				DCB	0x2f, 0x5e, 0xbc, 0x63
				DCB	0xc6, 0x97, 0x35, 0x6a
				DCB 0xd4, 0x00, 0x00, 0x00  ; last three constants added for alignment
deccstad		DCD	deccst               
deccst			DCB	0xd4, 0x6a, 0x35, 0x97	; Value of constant rounds
				DCB	0xc6, 0x63, 0xbc, 0x5e
				DCB	0x2f, 0x9a, 0x4d, 0xab
				DCB	0xd8, 0x6c, 0x36, 0x1b
				DCB 0x80, 0x00, 0x00, 0x00  ; last three constants added for alignment
nulcstad		DCD	nulcst               
nulcst			DCB	0x00, 0x00, 0x00, 0x00	; Value of constant rounds
				DCB	0x00, 0x00, 0x00, 0x00
				DCB	0x00, 0x00, 0x00, 0x00
				DCB	0x00, 0x00, 0x00, 0x00
				DCB	0x00, 0x00, 0x00, 0x00

;================================================================================================
; halftheta - First and Third stage of the LINEAR - MACRO
; -----------------------------------------------
;
; $rout0 <- $rout0 + F($rin0,$rin1)
; $rout1 <- $rout1 + F($rin0,$rin1)
;
; Modified reg. : $rout0,$rout1,$r5,$r6
;================================================================================================
		MACRO
$label halftheta $rout0,$rout1,$rin0,$rin1,$r5,$r6   
$label                           
		eor $r6,$rin0,$rin1
		eor $r5,$r6,$r6,ROR #8
		eor $r5,$r5,$r6,ROR #24
		eor $rout0,$rout0,$r5
		eor $rout1,$rout1,$r5
		MEND
;================================================================================================

;================================================================================================
; (pi2)theta - (pi2)LINEAR Step - MACRO
; -----------------------------
; DESCRIPTION
;   perform theta on state, as well as round constant addition
; INPUT:
;   r4-r7       = Half-theta key      (key is applied outside)
;   r0,r1,r2,r3 = a0,a1 ROL #1,a2,a3 ROL #2            !!!!!!!!!!!!!
;   r10         = first round constant
;   r11         = second round constant
; OUTPUT:
;   r0,r1,r2,r9 = new state  !!!!!!!!!!!!!
;
; Reserved reg. : R0,R12-R15
; Modified reg. : $r0-$r3,$r8-$r9
; Subproc Call  : Allowed
;------------------------------------------------------------------------------------------------
  MACRO
$label theta $r0,$r1,$r2,$r3,$r4,$r5,$r6,$r7,$r8,$r9,$r10,$r11,$r12
$label                           

  ldrb $r8,[$r10,$r12]        	; r10 = first round constant for this round
  eor $r0,$r0,$r8			   ; add first round constant

  eor $r0,$r4,$r0              ; add key ...
  eor $r1,$r5,$r1,ROR #1
  eor $r2,$r6,$r2
  eor $r9,$r7,$r3,ROR #2                           

  halftheta $r1,$r9,$r0,$r2,$r8,$r3
  
  halftheta $r0,$r2,$r1,$r9,$r8,$r3

  ldrb $r8,[$r11,$r12]        	; r10 = first round constant for this round
  eor $r0,$r0,$r8				; add second round constant
  
  MEND
;================================================================================================


;================================================================================================
; Pi1gammaPi2 - Pi1, NONLINEAR Step, Pi2 - MACRO
; ----------------------------------------------
; DESCRIPTION
;   perform Pi2 o gamma o Pi1 on state
; INPUT:
;   $r0,$r1,$r2,$r3 = state
; OUTPUT:
;   $r0,$r1 ROR #31,$r2,$r5 ROR #30 = new state
;
; Reserved reg. : R0,R12-R15
; Modified reg. : $r0-$r5
; Subproc Call  : Allowed
;
; a reduced definition of gamma is  
;       r0 <- r0 ^ (r2 . r1)
;       r1 <- r1 ^ (r3 v r2)
;       r2 <- ~(r3 ^ r2 ^ r1 ^ r0)
;       r1 <- r1 ^ (r0 v r2)           
;       tmp<- r3 ^ (r2 . r1)
;       r3 <- r0                   SKIPPED
;       r0 <- tmp                  SKIPPED
;
; rotations are merged in the computation of gamma & theta
;------------------------------------------------------------------------------------------------
  MACRO
$label Pi1gammaPi2 $r0,$r1,$r2,$r3,$r4,$r5
$label
;  mov $r1,$r1,ROR #31				;$r1 = a1 ROL #1     INCLUDED IN S-BOX
;  mov $r2,$r2,ROR #27				;$r2 = a2 ROL #5     INCLUDED IN S-BOX
;  mov $r3,$r3,ROR #30				;$r3 = a3 ROL #2     INCLUDED IN S-BOX

  and $r4,$r2,$r1,ROR #4
  eor $r5,$r0,$r4,ROR #27			;tmp <- r0 ^ (r2 . r1)
  orr $r4,$r2,$r3,ROR #3         
  eor $r1,$r4,$r1,ROR #4   			;r1 <- r1 ^ (r3 v r2)
  eor $r2,$r2,$r3,ROR #3
  eor $r2,$r2,$r1
  eor $r2,$r5,$r2,ROR #27
  mvn $r2,$r2,ROR #5				;r2 <- ~(r3 ^ r2 ^ r1 ^ tmp)
  orr $r4,$r5,$r2,ROR #27
  eor $r1,$r4,$r1,ROR #27           ;r1 <- r1 ^ (tmp v r2)
  and $r4,$r1,$r2,ROR #27  
  eor $r0,$r4,$r3,ROR #30           ;r0<- r3 ^ (r2 . r1)

;  mov $r1,$r1,ROR #1				;$r1 = a1 ROR #1    INCLUDED IN THETA - NEED PRE-LOOP ROR #31
;  mov $r2,$r2,ROR #5				;$r2 = a2 ROR #5    INCLUDED IN S-BOX
;  mov $r3,$r5,ROR #2				;$r3 = tmp ROR #2   INCLUDED IN THETA - NEED PRE-LOOP ROR #30

; ==> current state is $r0,$r1 ROR #31,$r2,$r5 ROR #30

  MEND
;================================================================================================

;================================================================================================
; NoekeonCore
;
; DESCRIPTION:
;    perform the Noekeon Core
; INPUT:
;   r4-r7 = half-theta key
;   r1 = from text buffer
;   r2 = to text buffer
;   r10 = Forward const ad
;   r11 = Backward const ad
;
; OUTPUT:
;   cipher text written at r2
;
; Reserved reg. : R12-R15
; Modified reg. : none
; Unmodif. reg. : all
; Subproc Call  : Allowed
;------------------------------------------------------------------------------------------------
		^ 0,sp
totext		# 4		;totext   = [sp]
fwdconst	# 4		;fwdconst = [sp,#4]
bwdconst	# 4		;bwdconst = [sp,#8]

NoekeonCore
;-------------------------------------------------------------------------------------------------
; Entry code 
;-------------------------------------------------------------------------------------------------
  eor r12,r12,r12
  stmdb sp!,{r2}         ;[sp]   =to text buffer pointer
								;[sp,#4]=fwd const,[sp,#8]=bwd const

  ldmia r1,{r0-r3}				;Read state from memory
  mov r1,r1,ROR #31				; because theta take r1 = r1 ROR #31 as input  
  mov r9,r3,ROR #30             ;                and r3 = r9 ROR #30  
;-------------------------------------------------------------------------------------------------
commonLoop     					; Main Loop
								; Pre: r12=round cntr, r0-r3=state, r4-r7=key
								;      r10 = fwdconst ad, r11 = bwdconst ad
;--- start of loop -------------------------------------------------------------------------------

  theta r0,r1,r2,r9,r4,r5,r6,r7,r8,r3,r10,r11,r12

  add r12,r12,#1				; increment round counter
  cmp r12,#NROUND
  bhi endCommonLoop

  Pi1gammaPi2 r0,r1,r2,r3,r8,r9			; do gamma
  b commonLoop

;--- end of loop ---------------------------------------------------------------------------------
endCommonLoop  

  ldr r12,totext			    ; r12=to text buffer addr
  stmia r12,{r0-r3}             ; Write back state in memory

  ldmia sp!,{r2}			; pop round counter
  mov pc,lr						; return from subroutine
;================================================================================================


;================================================================================================
; void NESSIEencrypt (NESSIEstruct * const structpointer,
;                     const unsigned char * const plaintext,
;                     const unsigned char * const ciphertext)
; DESCRIPTION:
;    encrypt the plaintext
; INPUT:
;   r0 = NESSIEstruct * const structpointer
;   r1 = const unsigned char * const plaintext
;   r2 = const unsigned char * const ciphertext
; OUTPUT:
;   cipher text written at r2
;
; Reserved reg. : R12-R15
; Modified reg. : r0-r12        !!!!!!!!!!
;------------------------------------------------------------------------------------------------

NESSIEencrypt
;-------------------------------------------------------------------------------------------------
; Entry code 
;-------------------------------------------------------------------------------------------------
  ldmia r0,{r4-r7}      		;load first key

  halftheta r5,r7,r4,r6,r8,r9

  ldr r10,enccstad
  ldr r11,nulcstad
  
  b NoekeonCore
;================================================================================================


;================================================================================================
; void NESSIEdecrypt (NESSIEstruct * const structpointer,
;                     const unsigned char * const ciphertext,
;                     const unsigned char * const plaintext)
; DESCRIPTION:
;    decrypt the ciphertext
; INPUT:
;   r0 = NESSIEstruct * const structpointer
;   r1 = const unsigned char * const ciphertext
;   r2 = const unsigned char * const plaintext
; OUTPUT:
;   plaintext text written at r2
;
; Reserved reg. : R12-R15
; Modified reg. : r0-r12        !!!!!!!!!!
;------------------------------------------------------------------------------------------------

NESSIEdecrypt
;-------------------------------------------------------------------------------------------------
; Entry code 
;-------------------------------------------------------------------------------------------------
  ldmia r0,{r4-r7}      		;load first key
  halftheta r4,r6,r5,r7,r8,r9

  ldr r10,nulcstad
  ldr r11,deccstad
  
  b NoekeonCore
;=================================================================================================



;================================================================================================
; void NESSIEkeysetup (const unsigned char * const key,
;                      NESSIEstruct * const structpointer)
; DESCRIPTION:
;   Set the key in the right format
; INPUT:
;   r0 = const unsigned char * const key
;   r1 = NESSIEstruct * const structpointer
; OUTPUT:
;   key written at r1
;
; Reserved reg. : R12-R15
; Modified reg. : none
; Unmodif. reg. : all
; Subproc Call  : Allowed
;------------------------------------------------------------------------------------------------
NESSIEkeysetup
  stmdb sp!,{r2-r6,lr} 		;Work registers allowed are R2-R6

  ldmia r0,{r2-r5}		    ;Read key from memory BIG ENDIAN MODEL
							;      R0              R1              R2              R3
							;k00-k01-k02-k03|k04-k05-k06-k07|k08-k09-k10-k11|k12-k13-k14-k15 |

  stmia r1,{r2-r5}

  ldmia sp!,{r2-r6,pc}  	; Retrieve registers and return from subroutine
;================================================================================================

  END
