;************************************************************************************/
;* Noekeon_ARM7.s
;*
;* Last Modified: 00/08/30             Created: 00/08/30
;*
;* Project    : Nessie Proposal: NOEKEON
;*
;* Authors    : Joan Daemen, Michael Peeters, Vincent Rijmen, Gilles Van Assche
;*
;* Written by : Michael Peeters
;*
;* References : [NESSIE] see http://cryptonessie.org
;*
;* Description: Implementation on an ARM7 processor of NOEKEON in DIRECT KEY MODE
;*              timing-attack resistant    
;*              straightforward implementation
;*
;* Comments:
;*   Memory model is BIG ENDIAN: r0=0x12345678 
;*                               --> Write in memory at 0x1000:  0x1000-78 56 34 12
;*   PC=R15, LR=R14, SP=R13
;*   Stack when interfacing with c: Decreasing before PUSH, Increasing after POP
;*
;************************************************************************************/

  EXPORT NESSIEencrypt
  EXPORT NESSIEdecrypt
  EXPORT NESSIEkeysetup

  AREA ASM_Noekeon_Data, DATA, READWRITE

thetakey		DCD	0x00000000, 0x00000000, 0x00000000, 0x00000000

  AREA ASM_Noekeon_Code, CODE, READONLY

NROUND			EQU 16				; Number of Computation rounds in the block cipher

enccstad		DCD	enccst               
enccst			DCB	0x80, 0x1b, 0x36, 0x6c	; Value of constant rounds
				DCB	0xd8, 0xab, 0x4d, 0x9a
				DCB	0x2f, 0x5e, 0xbc, 0x63
				DCB	0xc6, 0x97, 0x35, 0x6a
				DCB 0xd4, 0x00, 0x00, 0x00  ; last three constants added for alignment
deccstad		DCD	deccst               
deccst			DCB	0xd4, 0x6a, 0x35, 0x97	; Value of constant rounds
				DCB	0xc6, 0x63, 0xbc, 0x5e
				DCB	0x2f, 0x9a, 0x4d, 0xab
				DCB	0xd8, 0x6c, 0x36, 0x1b
				DCB 0x80, 0x00, 0x00, 0x00  ; last three constants added for alignment
nulcstad		DCD	nulcst               
nulcst			DCB	0x00, 0x00, 0x00, 0x00	; Value of constant rounds
				DCB	0x00, 0x00, 0x00, 0x00
				DCB	0x00, 0x00, 0x00, 0x00
				DCB	0x00, 0x00, 0x00, 0x00
				DCB	0x00, 0x00, 0x00, 0x00

thetakeyad		DCD thetakey

;================================================================================================
; halftheta - First and Third stage of the LINEAR - MACRO
; -----------------------------------------------
;
; $rout0 <- $rout0 + F($rin0,$rin1)
; $rout1 <- $rout1 + F($rin0,$rin1)
;
; Modified reg. : $rout0,$rout1,$r5,$r6
;================================================================================================
		MACRO
$label halftheta $rout0,$rout1,$rin0,$rin1,$r5,$r6   
$label                           
		eor $r6,$rin0,$rin1
		eor $r5,$r6,$r6,ROR #8
		eor $r5,$r5,$r6,ROR #24
		eor $rout0,$rout0,$r5
		eor $rout1,$rout1,$r5
		MEND
;================================================================================================

;================================================================================================
; theta_no_add - LINEAR Step without key addition - MACRO
; -----------------------------------------------
; DESCRIPTION
;   perform theta on input vector - used for key transformation for decryption
; INPUT:
;   $r1-$r4 = vector
; OUTPUT:
;   $r1-$r4 = new vector
;
; Reserved reg. : R0,R12-R15
; Modified reg. : $R1-$R4
;------------------------------------------------------------------------------------------------
		MACRO
$label theta_no_add $r1,$r2,$r3,$r4,$r5,$r6
$label                           
	   halftheta $r2,$r4,$r1,$r3,$r5,$r6
	   halftheta $r1,$r3,$r2,$r4,$r5,$r6
 	   MEND  
;================================================================================================

;================================================================================================
; theta - LINEAR Step
; -------------------
; DESCRIPTION
;   perform theta on state, as well as round constant addition
; INPUT:
;   r0    = key pointer
;   r1-r4 = state
;   r10   = first round constant
;   r11   = second round constant
; OUTPUT:
;   r1-r4 = new state
;
; Reserved reg. : R0,R12-R15
; Modified reg. : R1-R4,R5-R8,R11
; Subproc Call  : Allowed
;------------------------------------------------------------------------------------------------
theta

  eor r1,r1,r10				; add first round constant

  halftheta r2,r4,r1,r3,r5,r6
  
  ldmia r0,{r5-r8}			; load key vector
  eor r1,r1,r5              ; add key ...
  eor r2,r2,r6
  eor r3,r3,r7
  eor r4,r4,r8                           
  
  halftheta r1,r3,r2,r4,r5,r6

  eor r1,r1,r11				; add second round constant
  
  mov pc,lr
;================================================================================================


;================================================================================================
; gamma - NONLINEAR Step
; ---------------------
; DESCRIPTION
;   perform gamma on state, as well as key addition
; INPUT:
;   r0    = key pointer
;   r1-r4 = state
; OUTPUT:
;   r1-r4 = new state
;
; Reserved reg. : R0,R12-R15
; Modified reg. : R1-R4,R5
; Subproc Call  : Allowed
;
; a reduced definition of gamma is  
;       r1 <- r1 ^ (r3 . r2)
;       r2 <- r2 ^ (r4 v r3)
;       r3 <- ~(r4 ^ r3 ^ r2 ^ r1)
;       r2 <- r2 ^ (r1 v r3)           
;       tmp<- r4 ^ (r3 . r2)
;       r4 <- r1
;       r1 <- tmp
;------------------------------------------------------------------------------------------------
gamma
  and r5,r3,r2
  eor r1,r1,r5				;r1 <- r1 ^ (r3 . r2)
  orr r5,r4,r3
  eor r2,r2,r5              ;r2 <- r2 ^ (r4 v r3)
  eor r3,r3,r4
  eor r3,r3,r2
  eor r3,r3,r1
  mvn r3,r3					;r3 <- ~(r4 ^ r3 ^ r2 ^ r1)
  orr r5,r1,r3
  eor r2,r2,r5              ;r2 <- r2 ^ (r1 v r3)
  and r5,r3,r2  
  eor r5,r5,r4              ;tmp<- r4 ^ (r3 . r2)
  mov r4,r1                 ;r4 <- r1
  mov r1,r5					;r1 <- tmp		

  mov pc,lr
;================================================================================================

;================================================================================================
; pi1 - DISPERSION Step - MACRO
; ---------------------
; DESCRIPTION
;   perform pi1 on state
; INPUT:
;   r0    = key pointer
;   $r1-$r4 = state
; OUTPUT:
;   $r1-$r4 = new state
;
; Reserved reg. : R0,R12-R15
; Modified reg. : $r2-$r4
;------------------------------------------------------------------------------------------------
  MACRO
$label pi1 $r1,$r2,$r3,$r4
$label                           
  mov $r2,$r2,ROR #31				;$r2 = a1 ROL #1
  mov $r3,$r3,ROR #27				;$r3 = a2 ROL #5
  mov $r4,$r4,ROR #30				;$r4 = a3 ROL #2

  MEND
;================================================================================================


;================================================================================================
; pi2 - DISPERSION Step - MACRO
; ---------------------
; DESCRIPTION
;   perform pi2 on state
; INPUT:
;   r0    = key pointer
;   $r1-$r4 = state
; OUTPUT:
;   $r1-$r4 = new state
;
; Reserved reg. : R0,R12-R15
; Modified reg. : $r2-$r4
;------------------------------------------------------------------------------------------------
  MACRO
$label pi2 $r1,$r2,$r3,$r4
$label                           
  mov $r2,$r2,ROR #1				;$r2 = a1 ROR #1
  mov $r3,$r3,ROR #5				;$r3 = a2 ROR #5
  mov $r4,$r4,ROR #2				;$r4 = a3 ROR #2

  MEND
;================================================================================================


;================================================================================================
; NoekeonCore
;
; DESCRIPTION:
;    perform the Noekeon Core
; INPUT:
;   r0 = NESSIEstruct * const structpointer
;   r1 = from text buffer
;   r2 = to text buffer
;   r9 = Forward const
;   r10= Backward const
;
; OUTPUT:
;   cipher text written at r2
;
; Reserved reg. : R12-R15
; Modified reg. : none
; Unmodif. reg. : all
; Subproc Call  : Allowed
;------------------------------------------------------------------------------------------------
		^ 0,sp
totext		# 4		;totext   = [sp]
fwdconst	# 4		;fwdconst = [sp,#4]
bwdconst	# 4		;bwdconst = [sp,#8]

NoekeonCore
;-------------------------------------------------------------------------------------------------
; Entry code 
;-------------------------------------------------------------------------------------------------
  stmdb sp!,{lr}				;Work registers allowed are R0-R12
  
  eor r12,r12,r12
  stmdb sp!,{r2,r9,r10}         ;[sp]  =to text buffer pointer
								;[sp,#4]=fwd const,[sp,#8]=bwd const

  ldmia r1,{r1-r4}				;Read state from memory
  								;      R1              R2              R3              R4
  								;w00-w01-w02-w03|w04-w05-w06-w07|w08-w09-w10-w11|w12-w13-w14-w15 |

;-------------------------------------------------------------------------------------------------
commonloop     					; Main Loop
								; Pre: r12=round cntr, r0=key1 pointer, r1-r4=state1
;--- start of loop -------------------------------------------------------------------------------

  ldr r10,fwdconst		    	; r10 = round constants address in memory
  ldrb r10,[r10,r12]        	; r10 = first round constant for this round
  ldr r11,bwdconst		    	; r10 = round constants address in memory
  ldrb r11,[r11,r12]        	; r10 = first round constant for this round
  bl theta						; do theta
  pi1 r1,r2,r3,r4				; do pi1 on state1
  bl gamma						; do gamma
  pi2 r1,r2,r3,r4				; do pi2 on state1

  add r12,r12,#1				; increment round counter
  teq r12,#NROUND
  bne commonloop

;--- end of loop ---------------------------------------------------------------------------------
  ldr r10,fwdconst		    	; r10 = round constants address in memory
  ldrb r10,[r10,r12]        	; r10 = first round constant for this round
  ldr r11,bwdconst		    	; r10 = round constants address in memory
  ldrb r11,[r11,r12]        	; r10 = first round constant for this round
  bl theta						; do theta

  ldr r12,totext			    ; r12=to text buffer addr
  stmia r12,{r1-r4}             ; Write back state in memory

  ldmia sp!,{r2,r9,r10}			; pop round counter
  ldmia sp!,{pc}				; Retrieve registers and return from subroutine
;================================================================================================




;================================================================================================
; void NESSIEencrypt (NESSIEstruct * const structpointer,
;                     const unsigned char * const plaintext,
;                     const unsigned char * const ciphertext)
; DESCRIPTION:
;    encrypt the plaintext
; INPUT:
;   r0 = NESSIEstruct * const structpointer
;   r1 = const unsigned char * const plaintext
;   r2 = const unsigned char * const ciphertext
; OUTPUT:
;   cipher text written at r2
;
; Reserved reg. : R12-R15
; Modified reg. : none
; Unmodif. reg. : all
; Subproc Call  : Allowed
;------------------------------------------------------------------------------------------------

NESSIEencrypt
;-------------------------------------------------------------------------------------------------
; Entry code 
;-------------------------------------------------------------------------------------------------
  stmdb sp!,{r0-r12,lr}			;Work registers allowed are R0-R12

  ldr r9,enccstad
  ldr r10,nulcstad
  
  bl NoekeonCore

  ldmia sp!,{r0-r12,pc}			; Retrieve registers and return from subroutine
;================================================================================================


;================================================================================================
; void NESSIEdecrypt (NESSIEstruct * const structpointer,
;                     const unsigned char * const ciphertext,
;                     const unsigned char * const plaintext)
; DESCRIPTION:
;    decrypt the ciphertext
; INPUT:
;   r0 = NESSIEstruct * const structpointer
;   r1 = const unsigned char * const ciphertext
;   r2 = const unsigned char * const plaintext
; OUTPUT:
;   plaintext text written at r2
;
; Reserved reg. : R12-R15
; Modified reg. : none
; Unmodif. reg. : all
; Subproc Call  : Allowed
;------------------------------------------------------------------------------------------------

NESSIEdecrypt
;-------------------------------------------------------------------------------------------------
; Entry code 
;-------------------------------------------------------------------------------------------------
  stmdb sp!,{r0-r12,lr}			;Work registers allowed are R0-R12

  ldmia r0,{r3-r6}      		;load first key
  theta_no_add r3,r4,r5,r6,r7,r8
  ldr r0,thetakeyad				;r0 = transformed key address
  stmia r0,{r3-r6}				;store theta (first key)

  ldr r9,nulcstad
  ldr r10,deccstad
  
  bl NoekeonCore

  ldmia sp!,{r0-r12,pc}			; Retrieve registers and return from subroutine
;================================================================================================



;================================================================================================
; void NESSIEkeysetup (const unsigned char * const key,
;                      NESSIEstruct * const structpointer)
; DESCRIPTION:
;   Set the key in the right format
; INPUT:
;   r0 = const unsigned char * const key
;   r1 = NESSIEstruct * const structpointer
; OUTPUT:
;   key written at r1
;
; Reserved reg. : R12-R15
; Modified reg. : none
; Unmodif. reg. : all
; Subproc Call  : Allowed
;------------------------------------------------------------------------------------------------
NESSIEkeysetup
  stmdb sp!,{r2-r6,lr} 		;Work registers allowed are R2-R6

  ldmia r0,{r2-r5}		    ;Read key from memory BIG ENDIAN MODEL
							;      R0              R1              R2              R3
							;k00-k01-k02-k03|k04-k05-k06-k07|k08-k09-k10-k11|k12-k13-k14-k15 |

  stmia r1,{r2-r5}

  ldmia sp!,{r2-r6,pc}  	; Retrieve registers and return from subroutine
;================================================================================================

  END
