;************************************************************************************************/
;* Noekeon_ARM7.s
;*
;* Last Modified: 00/08/30             Created: 00/08/30
;*
;* Project    : Nessie Proposal: NOEKEON
;*
;* Authors    : Joan Daemen, Michael Peeters, Vincent Rijmen, Gilles Van Assche
;*
;* Written by : Michael Peeters
;*
;* References : [NESSIE] see http://cryptonessie.org
;*
;* Description: Optimised implementation on an ARM7 processor of NOEKEON in DIRECT KEY MODE
;*              timing-attack resistant   
;*              optimised for speed
;*
;* Comments:
;*   Memory model is BIG ENDIAN: r0=0x12345678 
;*                               --> Write in memory at 0x1000:  0x1000-78 56 34 12
;*   PC=R15, LR=R14, SP=R13
;*   Stack when interfacing with c: Decreasing before PUSH, Increasing after POP
;*
;************************************************************************************************/

  EXPORT NESSIEencrypt
  EXPORT NESSIEdecrypt
  EXPORT NESSIEkeysetup

  AREA ASM_Noekeon_Code, CODE, READONLY

;================================================================================================
; halftheta - First and Third stage of the LINEAR - MACRO
; -----------------------------------------------
;
; $rout0 <- $rout0 + F($rin0,$rin1)
; $rout1 <- $rout1 + F($rin0,$rin1)
;
; Modified reg. : $rout0,$rout1,$r5,$r6
;================================================================================================
		MACRO
$label halftheta $rout0,$rout1,$rin0,$rin1,$r5,$r6   
$label                           
		eor $r6,$rin0,$rin1
		eor $r5,$r6,$r6,ROR #8
		eor $r5,$r5,$r6,ROR #24
		eor $rout0,$rout0,$r5
		eor $rout1,$rout1,$r5
		MEND
;================================================================================================

;================================================================================================
; theta No Round Const - LINEAR Step - MACRO
; ------------------------------------------
; DESCRIPTION
;   perform theta on state
; INPUT:
;   r4-r7       = Half-theta key      (key is applied outside)
;   r0,r1,r2,r3 = a0,a1 ROL #1,a2,a3 ROL #2            !!!!!!!!!!!!!
; OUTPUT:
;   r0,r1,r2,r9 = new state  !!!!!!!!!!!!!
;
; Reserved reg. : R0,R12-R15
; Modified reg. : $r0-$r3,$r8-$r9
; Subproc Call  : Allowed
;------------------------------------------------------------------------------------------------
  MACRO
$label thetaNC $r0,$r1,$r2,$r3,$r4,$r5,$r6,$r7,$r8,$r9
$label                           

  eor $r0,$r4,$r0              ; add key ...
  eor $r1,$r5,$r1,ROR #1
  eor $r2,$r6,$r2
  eor $r9,$r7,$r3,ROR #2                           

  halftheta $r1,$r9,$r0,$r2,$r8,$r3
  
  halftheta $r0,$r2,$r1,$r9,$r8,$r3

  MEND
;================================================================================================


;================================================================================================
; Pi1gammaPi2 - Pi1, NONLINEAR Step, Pi2 - MACRO
; ----------------------------------------------
; DESCRIPTION
;   perform Pi2 o gamma o Pi1 on state
; INPUT:
;   $r0,$r1,$r2,$r3 = state
; OUTPUT:
;   $r0,$r1 ROR #31,$r2,$r5 ROR #30 = new state
;
; Reserved reg. : R0,R12-R15
; Modified reg. : $r0-$r5
; Subproc Call  : Allowed
;
; a reduced definition of gamma is  
;       r0 <- r0 ^ (r2 . r1)
;       r1 <- r1 ^ (r3 v r2)
;       r2 <- ~(r3 ^ r2 ^ r1 ^ r0)
;       r1 <- r1 ^ (r0 v r2)           
;       tmp<- r3 ^ (r2 . r1)
;       r3 <- r0                   SKIPPED
;       r0 <- tmp                  SKIPPED
;
; rotations are merged in the computation of gamma & theta
;------------------------------------------------------------------------------------------------
  MACRO
$label Pi1gammaPi2 $r0,$r1,$r2,$r3,$r4,$r5
$label
;  mov $r1,$r1,ROR #31				;$r1 = a1 ROL #1     INCLUDED IN S-BOX
;  mov $r2,$r2,ROR #27				;$r2 = a2 ROL #5     INCLUDED IN S-BOX
;  mov $r3,$r3,ROR #30				;$r3 = a3 ROL #2     INCLUDED IN S-BOX

  and $r4,$r2,$r1,ROR #4
  eor $r5,$r0,$r4,ROR #27			;tmp <- r0 ^ (r2 . r1)
  orr $r4,$r2,$r3,ROR #3         
  eor $r1,$r4,$r1,ROR #4   			;r1 <- r1 ^ (r3 v r2)
  eor $r2,$r2,$r3,ROR #3
  eor $r2,$r2,$r1
  eor $r2,$r5,$r2,ROR #27
  mvn $r2,$r2,ROR #5				;r2 <- ~(r3 ^ r2 ^ r1 ^ tmp)
  orr $r4,$r5,$r2,ROR #27
  eor $r1,$r4,$r1,ROR #27           ;r1 <- r1 ^ (tmp v r2)
  and $r4,$r1,$r2,ROR #27  
  eor $r0,$r4,$r3,ROR #30           ;r0<- r3 ^ (r2 . r1)

;  mov $r1,$r1,ROR #1				;$r1 = a1 ROR #1    INCLUDED IN THETA - NEED PRE-LOOP ROR #31
;  mov $r2,$r2,ROR #5				;$r2 = a2 ROR #5    INCLUDED IN S-BOX
;  mov $r3,$r5,ROR #2				;$r3 = tmp ROR #2   INCLUDED IN THETA - NEED PRE-LOOP ROR #30

; ==> current state is $r0,$r1 ROR #31,$r2,$r5 ROR #30

  MEND
;================================================================================================

  MACRO
$label fround $RC,$r0,$r1,$r2,$r3,$r4,$r5,$r6,$r7,$r8,$r9
  eor $r0,$r0,#$RC
  thetaNC $r0,$r1,$r2,$r3,$r4,$r5,$r6,$r7,$r8,$r9
  Pi1gammaPi2 $r0,$r1,$r2,$r9,$r8,$r3
  MEND

;================================================================================================
; void NESSIEencrypt (NESSIEstruct * const structpointer,
;                     const unsigned char * const plaintext,
;                     const unsigned char * const ciphertext)
; DESCRIPTION:
;    encrypt the plaintext
; INPUT:
;   r0 = NESSIEstruct * const structpointer
;   r1 = const unsigned char * const plaintext
;   r2 = const unsigned char * const ciphertext
; OUTPUT:
;   cipher text written at r2
;
; Reserved reg. : R12-R15
; Modified reg. : r0-r12        !!!!!!!!!!
;------------------------------------------------------------------------------------------------

NESSIEencrypt
;-------------------------------------------------------------------------------------------------
; Entry code 
;-------------------------------------------------------------------------------------------------
  ldmia r0,{r4-r7}      		;load first key

  halftheta r5,r7,r4,r6,r8,r9

;-------------------------------------------------------------------------------------------------
; Entry code 
;-------------------------------------------------------------------------------------------------
  mov r12,r2	                ;[sp]   =to text buffer pointer

  ldmia r1,{r0-r3}				;Read state from memory
  mov r1,r1,ROR #31				; because theta take r1 = r1 ROR #31 as input  
  mov r9,r3,ROR #30             ;                and r3 = r9 ROR #30  
;-------------------------------------------------------------------------------------------------


  fround 0x80,r0,r1,r2,r9,r4,r5,r6,r7,r8,r3
  fround 0x1B,r0,r1,r2,r9,r4,r5,r6,r7,r8,r3
  fround 0x36,r0,r1,r2,r9,r4,r5,r6,r7,r8,r3
  fround 0x6C,r0,r1,r2,r9,r4,r5,r6,r7,r8,r3

  fround 0xD8,r0,r1,r2,r9,r4,r5,r6,r7,r8,r3
  fround 0xAB,r0,r1,r2,r9,r4,r5,r6,r7,r8,r3
  fround 0x4D,r0,r1,r2,r9,r4,r5,r6,r7,r8,r3
  fround 0x9A,r0,r1,r2,r9,r4,r5,r6,r7,r8,r3

  fround 0x2F,r0,r1,r2,r9,r4,r5,r6,r7,r8,r3
  fround 0x5E,r0,r1,r2,r9,r4,r5,r6,r7,r8,r3
  fround 0xBC,r0,r1,r2,r9,r4,r5,r6,r7,r8,r3
  fround 0x63,r0,r1,r2,r9,r4,r5,r6,r7,r8,r3

  fround 0xC6,r0,r1,r2,r9,r4,r5,r6,r7,r8,r3
  fround 0x97,r0,r1,r2,r9,r4,r5,r6,r7,r8,r3
  fround 0x35,r0,r1,r2,r9,r4,r5,r6,r7,r8,r3
  fround 0x6A,r0,r1,r2,r9,r4,r5,r6,r7,r8,r3

  eor r0,r0,#0xD4
  thetaNC r0,r1,r2,r9,r4,r5,r6,r7,r8,r3

  stmia r12,{r0-r3}             ; Write back state in memory

  mov pc,lr						; return from subroutine
;================================================================================================


  MACRO
$label iround $RC,$r0,$r1,$r2,$r3,$r4,$r5,$r6,$r7,$r8,$r9
  thetaNC $r0,$r1,$r2,$r3,$r4,$r5,$r6,$r7,$r8,$r9
  eor $r0,$r0,#$RC
  Pi1gammaPi2 $r0,$r1,$r2,$r9,$r8,$r3
  MEND

;================================================================================================
; void NESSIEdecrypt (NESSIEstruct * const structpointer,
;                     const unsigned char * const ciphertext,
;                     const unsigned char * const plaintext)
; DESCRIPTION:
;    decrypt the ciphertext
; INPUT:
;   r0 = NESSIEstruct * const structpointer
;   r1 = const unsigned char * const ciphertext
;   r2 = const unsigned char * const plaintext
; OUTPUT:
;   plaintext text written at r2
;
; Reserved reg. : R12-R15
; Modified reg. : r0-r12        !!!!!!!!!!
;------------------------------------------------------------------------------------------------

NESSIEdecrypt
;-------------------------------------------------------------------------------------------------
; Entry code 
;-------------------------------------------------------------------------------------------------
  ldmia r0,{r4-r7}      		;load first key
  halftheta r4,r6,r5,r7,r8,r9

;-------------------------------------------------------------------------------------------------
; Entry code 
;-------------------------------------------------------------------------------------------------
  mov r12,r2         			;r12   =to text buffer pointer

  ldmia r1,{r0-r3}				;Read state from memory
  mov r1,r1,ROR #31				; because theta take r1 = r1 ROR #31 as input  
  mov r9,r3,ROR #30             ;                and r3 = r9 ROR #30  
;-------------------------------------------------------------------------------------------------

  iround 0xD4,r0,r1,r2,r9,r4,r5,r6,r7,r8,r3
  iround 0x6A,r0,r1,r2,r9,r4,r5,r6,r7,r8,r3
  iround 0x35,r0,r1,r2,r9,r4,r5,r6,r7,r8,r3
  iround 0x97,r0,r1,r2,r9,r4,r5,r6,r7,r8,r3

  iround 0xC6,r0,r1,r2,r9,r4,r5,r6,r7,r8,r3
  iround 0x63,r0,r1,r2,r9,r4,r5,r6,r7,r8,r3
  iround 0xBC,r0,r1,r2,r9,r4,r5,r6,r7,r8,r3
  iround 0x5E,r0,r1,r2,r9,r4,r5,r6,r7,r8,r3

  iround 0x2F,r0,r1,r2,r9,r4,r5,r6,r7,r8,r3
  iround 0x9A,r0,r1,r2,r9,r4,r5,r6,r7,r8,r3
  iround 0x4D,r0,r1,r2,r9,r4,r5,r6,r7,r8,r3
  iround 0xAB,r0,r1,r2,r9,r4,r5,r6,r7,r8,r3

  iround 0xD8,r0,r1,r2,r9,r4,r5,r6,r7,r8,r3
  iround 0x6C,r0,r1,r2,r9,r4,r5,r6,r7,r8,r3
  iround 0x36,r0,r1,r2,r9,r4,r5,r6,r7,r8,r3
  iround 0x1B,r0,r1,r2,r9,r4,r5,r6,r7,r8,r3

  thetaNC r0,r1,r2,r9,r4,r5,r6,r7,r8,r3
  eor r0,r0,#0x80

  stmia r12,{r0-r3}             ; Write back state in memory

  mov pc,lr						; return from subroutine
;================================================================================================



;================================================================================================
; void NESSIEkeysetup (const unsigned char * const key,
;                      NESSIEstruct * const structpointer)
; DESCRIPTION:
;   Set the key in the right format
; INPUT:
;   r0 = const unsigned char * const key
;   r1 = NESSIEstruct * const structpointer
; OUTPUT:
;   key written at r1
;
; Reserved reg. : R12-R15
; Modified reg. : none
; Unmodif. reg. : all
; Subproc Call  : Allowed
;------------------------------------------------------------------------------------------------
NESSIEkeysetup
  stmdb sp!,{r2-r6,lr} 		;Work registers allowed are R2-R6

  ldmia r0,{r2-r5}		    ;Read key from memory BIG ENDIAN MODEL
							;      R0              R1              R2              R3
							;k00-k01-k02-k03|k04-k05-k06-k07|k08-k09-k10-k11|k12-k13-k14-k15 |

  stmia r1,{r2-r5}

  ldmia sp!,{r2-r6,pc}  	; Retrieve registers and return from subroutine
;================================================================================================

  END
