#!/bin/sh

# prandom.sh

# This shell script is used to generate "unpredictable" text that is run
# through a cryptographic hash to produce "random" numbers for the
# TIS/MOSS system.  Under normal cicumstances, it is only used when 
# generating RSA key pairs.  It is a lot of work to generate enough
# unpredictability, but it is well worth it to have adequate security.
# There is a limit to the amount of unpredictability this method can
# produce, especially when an adversary has access to the same system at
# approximately the same time, but it is portable and there is going to
# be a limit of some sort or another until systems include a source of
# true random numbers.
# 
# The unpredictability of the text is important to the security of
# TIS/MOSS.  Read the comments carefully and test changes before
# incorporating them.  Programs included in this script should produce
# as widely varying results as possible and at the same time they should
# not take too long a time to execute.  The longer it takes to execute
# this script, the longer it will take to generate key pairs.
# 
# All programs should use fully qualified paths to prevent trojan
# horses.  Different systems put programs in different places, so you
# will probably have to change the paths or file names somewhere along
# the line. To make it easier to work on different systems, we have put
# in tests for program location for programs which have a well known set
# of hiding places.  Programs should be called with options that produce
# the most statistics without unduly slowing things down.  Programs can
# be called more than once for different types of output.

# Set path variable to prevent trojan horses.

PATH="/bin:/usr/bin"

# Put error messages in variables

cantfind="$0: Warning: Can not find "
badreturn="$0: Warning: Bad return value from "

# Start by getting the date (date & time).  It is usually in /bin.

if [ -f /bin/date ]; then
   /bin/date 2>&1
   if [ $? != 0 ]; then
      /bin/echo $badreturn date 1>&2
   fi
else
   /bin/echo $cantfind date 1>&2
fi


# Run the "ps" program with options to print as many statistics as
# possible about all processes.  The options vary between systems.  "ps
# -laxw" works on BSD systems, while "ps -el" works on SYSV.  We will
# try the BSD one first, and if that doesn work we will do SYSV.  It is
# best to not choose options that print user and group names (as opposed
# to uids and gids) because they only serve to slow things down since
# they require access to password and group entries.  "ps" can be slow,
# but it is worth it.  "ps" is usually in /bin.

if [ -f /bin/ps ]; then
#  Try BSD options first.  Redirect stderr to stdout in case of error.
   /bin/ps -axlw 2>&1
   if [ $? != 0 ]; then
#     The BSD options did not work, so try SYSV.
      /bin/ps -el 2>&1
      if [ $? != 0 ]; then
         /bin/echo $badreturn ps 1>&2
      fi
   fi
else
   /bin/echo $cantfind ps 1>&2
fi


# Run the "df" program.  df is usually in /bin.  

if [ -f /bin/df ]; then
   /bin/df 2>&1
   if [ $? != 0 ]; then
      /bin/echo $badreturn df 1>&2
   fi
else
   /bin/echo $cantfind df 1>&2
fi


# Run the "iostat" and "vmstat" programs.  They report disk i/o and
# memory statistics.  We have seen iostat in /bin and /usr/bin and
# vmstat as /usr/ucb/vmstat, /usr/bin/vmstat, and /usr/bin/vm_stat!

if [ -f /bin/iostat ]; then
   /bin/iostat 2>&1
   if [ $? != 0 ]; then
      /bin/echo $badreturn iostat 1>&2
   fi
elif [ -f /usr/bin/iostat ]; then 
   /usr/bin/iostat 2>&1
   if [ $? != 0 ]; then
      /bin/echo $badreturn iostat 1>&2
   fi
else
   /bin/echo $cantfind iostat 1>&2
fi

if [ -f /usr/ucb/vmstat ]; then
   /usr/ucb/vmstat 2>&1
   if [ $? != 0 ]; then
      /bin/echo $badreturn vmstat 1>&2
   fi
elif [ -f /usr/bin/vmstat ]; then
   /usr/bin/vmstat 2>&1
   if [ $? != 0 ]; then
      /bin/echo $badreturn vmstat 1>&2
   fi
elif [ -f /usr/bin/vm_stat ]; then
   /usr/bin/vm_stat 2>&1
   if [ $? != 0 ]; then
      /bin/echo $badreturn vm_stat 1>&2
   fi
else
   /bin/echo $cantfind vmstat or vm_stat 1>&2
fi


# Run the "netstat" program.  Like "ps", use the options that generate
# the most statistics.  Make sure that your options do not cause IP
# addresses to be converted to domain names because the conversion can
# slow things down and adds little unpredictability.  This program is
# called twice to generate two distinct reports.  We have seen netstat
# in /usr/ucb and /usr/bin.

if [ -f /usr/ucb/netstat ]; then
#  Connections
   /usr/ucb/netstat -an 2>&1
   if [ $? != 0 ]; then
      /bin/echo $badreturn netstat 1>&2
   fi
#  Statistics
   /usr/ucb/netstat -s 2>&1
   if [ $? != 0 ]; then
      /bin/echo $badreturn netstat 1>&2
   fi
elif [ -f /usr/bin/netstat ]; then
#  Connections
   /usr/bin/netstat -an 2>&1
   if [ $? != 0 ]; then
      /bin/echo $badreturn netstat 1>&2
   fi
#  Statistics
   /usr/bin/netstat -s 2>&1
   if [ $? != 0 ]; then
      /bin/echo $badreturn netstat 1>&2
   fi
else
   /bin/echo $cantfind netstat 1>&2
fi


# While we're doing networking stuff, use the arp command to dump the 
# arp cache.  We've seen arp in /usr/etc, /etc, /usr/bin, and /usr/sbin!

if [ -f /usr/etc/arp ]; then
   /usr/etc/arp -a 2>&1
   if [ $? != 0 ]; then
      /bin/echo $badreturn arp 1>&2
   fi
elif [ -f /etc/arp ]; then
   /etc/arp -a 2>&1
   if [ $? != 0 ]; then
      /bin/echo $badreturn arp 1>&2
   fi
elif [ -f /usr/bin/arp ]; then
   /usr/bin/arp -a 2>&1
   if [ $? != 0 ]; then
      /bin/echo $badreturn arp 1>&2
   fi
elif [ -f /usr/sbin/arp ]; then
   /usr/sbin/arp -a 2>&1
   if [ $? != 0 ]; then
      /bin/echo $badreturn arp 1>&2
   fi
else
   /bin/echo $cantfind arp 1>&2
fi


# Do an extended listing of the files in /tmp and any other directory
# where things change often.  If any of these directories contains 
# a very large number of entries, performance may be a problem.
# "ls" is in /bin.

if [ -f /bin/ls ]; then
   if [ -d /tmp/ ]; then
      /bin/ls -laiu /tmp/ 2>&1
      if [ $? != 0 ]; then
         /bin/echo $badreturn ls of /tmp/ 1>&2
      fi
   else
      /bin/echo $cantfind /tmp/ for ls 1>&2
   fi
   if [ -d /usr/tmp/ ]; then
      /bin/ls -laiu /usr/tmp/ 2>&1
      if [ $? != 0 ]; then
         /bin/echo $badreturn ls of /usr/tmp/ 1>&2
      fi
   else
      /bin/echo $cantfind /usr/tmp/ for ls 1>&2
   fi
   if [ -d /usr/adm/ ]; then
      /bin/ls -laiu /usr/adm/ 2>&1
      if [ $? != 0 ]; then
         /bin/echo $badreturn ls of /usr/adm/ 1>&2
      fi
   else
      /bin/echo $cantfind /usr/adm for ls 1>&2
   fi
   if [ -d /usr/spool/ ]; then
      /bin/ls -laiu /usr/spool/ 2>&1
      if [ $? != 0 ]; then
         /bin/echo $badreturn ls of /usr/spool 1>&2
      fi
   else
      /bin/echo $cantfind /usr/spool for ls 1>&2
   fi
   if [ -d /usr/spool/mail/ ]; then
      /bin/ls -laiu /usr/spool/mail/ 2>&1
      if [ $? != 0 ]; then
         /bin/echo $badreturn ls of /usr/spool/mail 1>&2
      fi
   elif [ -d /usr/mail/ ]; then
      /bin/ls -laiu /usr/mail/ 2>&1
      if [ $? != 0 ]; then
         /bin/echo $badreturn ls of /usr/spool/mail 1>&2
      fi
   else
      /bin/echo $cantfind /usr/spool/mail or /usr/mail for ls 1>&2
   fi
   if [ -d /dev/ ]; then
      /bin/ls -laiu /dev/ 2>&1
      if [ $? != 0 ]; then
         /bin/echo $badreturn ls of /dev 1>&2
      fi
   else
      /bin/echo $cantfind /dev for ls 1>&2
   fi
else
   /bin/echo $cantfind ls 1>&2
fi


# Do an "uptime".  We have seen uptime in /usr/ucb and /usr/bin.

if [ -f /usr/ucb/uptime ]; then
   /usr/ucb/uptime 2>&1
   if [ $? != 0 ]; then
      /bin/echo $badreturn uptime 1>&2
   fi
elif [ -f /usr/bin/uptime ]; then
   /usr/bin/uptime 2>&1
   if [ $? != 0 ]; then
      /bin/echo $badreturn uptime 1>&2
   fi
else
   /bin/echo $cantfind uptime 1>&2
fi


# Include user environment variables using printenv.  We've seen printenv 
# in /usr/ucb and /usr/bin.

if [ -f /usr/ucb/printenv ]; then
   /usr/ucb/printenv 2>&1
   if [ $? != 0 ]; then
      /bin/echo $badreturn printenv 1>&2
   fi
elif [ -f /usr/bin/printenv ]; then
   /usr/bin/printenv 2>&1
   if [ $? != 0 ]; then
      /bin/echo $badreturn printenv 1>&2
   fi
else
   /bin/echo $cantfind printenv 1>&2
fi


# Throw in the hostname command.  It will not add much unpredictability,
# but it will make differences on different systems and it does not take
# long to run, either.  Usually in /bin, but sometimes in /usr/bin or 
# /usr/ucb.

if [ -f /bin/hostname ]; then
   /bin/hostname 2>&1
   if [ $? != 0 ]; then
      /bin/echo $badreturn hostname 1>&2
   fi
elif [ -f /usr/bin/hostname ]; then
   /usr/bin/hostname 2>&1
   if [ $? != 0 ]; then
      /bin/echo $badreturn hostname 1>&2
   fi
elif [ -f /usr/ucb/hostname ]; then
   /usr/ucb/hostname 2>&1
   if [ $? != 0 ]; then
      /bin/echo $badreturn hostname 1>&2
   fi
else
   /bin/echo $cantfind hostname 1>&2
fi


# Get the date again.  

if [ -f /bin/date ]; then
   /bin/date 2>&1
   if [ $? != 0 ]; then
      /bin/echo $badreturn date 1>&2
   fi
else
   /bin/echo $cantfind date 1>&2
fi
