/*
Copyright (C) 1992,1993,1994 Trusted Information Systems, Inc.

Export of this software from the United States of America or
Canada requires a specific license from the United States
Government.  This version of this software is not suitable for
export.

WITHIN THAT CONSTRAINT, the full text of the license agreement
that specifies the conditions under which this software may be
used is published in the file license.txt in the same directory
as that containing the TIS/PEM source.

Trusted Information Systems makes no representation about the
suitability of this software for any purpose.  It is provided
"as is" without express or implied warranty.
*/

/****************************************************************************
 *  gen_rsakey.c
 ****************************************************************************/

#include "config.h"
#include "general.h"

#include "algid.h"
#include "bbuf.h"
#include "crypto.h"
#include "key.h"
#include "util.h"

#include "./rsa.h"

static int gen_rsakey_aux();

int gen_rsakey(alg, len, pubkey, privkey)
int alg;			/* algorithm */
int len;			/* modulus length (in bits) */
struct key **pubkey;		/* public key */
struct key **privkey;		/* private key */
{
    int r;
    R_RSA_PUBLIC_KEY Pu;
    R_RSA_PRIVATE_KEY Pr;
    struct key *kd1 = NULL_KEY;
    struct key *kd2 = NULL_KEY;

    /* DOUBLE-CHECK PARAMETERS */
    if (pubkey == (struct key **)0) 
	return(NOTOK);

    FREE_KEY(*pubkey);
    if (privkey == (struct key **)0) 
	return(NOTOK);

    FREE_KEY(*privkey);

    /* CHECK THAT MODULUS LENGTH IS WITHIN IMPLEMENTATION RANGE */
    if (len < MIN_RSA_MODULUS_BITS || len > MAX_RSA_MODULUS_BITS) 
	return(NOTOK);

    /* GENERATE THE RSAREF KEYS */
    if ((r = gen_rsakey_aux(alg, len, &Pu, &Pr)) != OK)
	goto cleanup;

    /* CONSTRUCT PUBLIC KEY STRUCTURE */
    if ((r = alloc_key(pubkey)) != OK)
	goto cleanup;
    kd1 = (*pubkey);
    kd1->algid = alloc_algid();
    kd1->algid->alg = alg;
    if ((alg & CRYPTO_MASK) == RSA) {
	if ((r = encode_rsakeyparms(len, &(kd1->algid->parms))) != OK)
	    goto cleanup;
    }
    else kd1->algid->parms = NULLBB;
    if ((r = encode_rsapubkeydata(&Pu, &(kd1->data))) != OK)
	goto cleanup;

    /* CONSTRUCT PRIVATE KEY STRUCTURE */
    if ((r = alloc_key(privkey)) != OK)
	goto cleanup;
    kd2 = (*privkey);
    kd2->algid = alloc_algid();
    if ((alg & CRYPTO_MASK) == RSA) {
	kd2->algid->alg = RSA_PRIV;
	if ((r = encode_rsakeyparms(len, &(kd2->algid->parms))) != OK)
	    goto cleanup;
    } else {
	kd2->algid->alg = RSAENC_PRIV;
	kd2->algid->parms = NULLBB;
    }
    if ((r = encode_rsaprikeydata(&Pr, &(kd2->data))) != OK)
	goto cleanup;

    r = OK;

 cleanup:

    if (r != OK) {
	FREE_KEY(*pubkey);
	FREE_KEY(*privkey);
    }

    return(r);
}

/****************************************************************************
 * gen_rsakey_aux()
 ****************************************************************************/

static int gen_rsakey_aux(alg, len, Pu, Pr)
int alg;
int len;
R_RSA_PUBLIC_KEY *Pu;
R_RSA_PRIVATE_KEY *Pr;
{
    int r;
    R_RSA_PROTO_KEY Pk;
    struct bbuf *seed = NULLBB;
    R_RANDOM_STRUCT Rs;
    unsigned int bytesNeeded;
  
    /* INITIALIZE PROTOTYPE KEY */
    Pk.bits = (unsigned int)len;
    switch (alg) {
    case RSA_PUBF4:
    case RSAENC_PUBF4:
	Pk.useFermat4 = 1;
	break;
    case RSA:
    case RSA_PUB:
    case RSA_PUB3:
    case RSA_PRIV:
    case RSAENC:
    case RSAENC_PUB:
    case RSAENC_PUB3:
    case RSAENC_PRIV:
	Pk.useFermat4 = 0;
	break;
    default:
	return(NOTOK);
    }
	    
    /* PREPARE RSAREF RANDOM STRUCTURE */
    R_RandomInit(&Rs);
    while (1) {
	R_GetRandomBytesNeeded(&bytesNeeded, &Rs);
	if (bytesNeeded == 0) break;
	if ((seed = pseudorandom(RSASEEDSIZE)) == NULLBB) {
	    r = NOTOK;
	    goto cleanup;
	}
	R_RandomUpdate(&Rs, seed->data, (unsigned int)seed->length);
	FREE_BBUF(seed);
    }

    /* GENERATE THE PUBLIC AND PRIVATE KEYS */
    if ((r = R_GeneratePEMKeys(Pu, Pr, &Pk, &Rs)) != 0) {
	r = NOTOK;
	goto cleanup;
    }

    r = OK;

 cleanup:

    /* ZEROIZE THE RSAREF RANDOM STRUCTURE */
    R_RandomFinal(&Rs);

    return(r);
}

/***************************************************************************/
