#include <openssl/rand.h>
#include <openssl/sha.h>

#include "config.h"
#include "prngd.h"

static unsigned char obfuscator = 0;

 /*
  * rand_add(): wrapper for the OpenSSL RAND_add() function. The wrapper
  * is used to keep the calling functions simpler. Before adding entropy
  * to the pool, we want to prevent "chosen seed" attacks. Therefore, the
  * bytes obtained are not directly added, but they are "obfuscated" with
  * a more-or-less random value and then run through the SHA1-algorithm.
  * Hence, if an attacker would want to choose the seed-bytes, he would have
  * to construct the necessary SHA1-input(!) (which should be computationally
  * hard) _and_ predict the obfuscator-value.
  */
void rand_add(void *buf, int num, double entropy)
{
  int i, numbytes;
  unsigned char *uchar_buf;
  unsigned char sha_digest[SHA_DIGEST_LENGTH];

  uchar_buf = (unsigned char *)buf;

  for (i = 0; i < num; i++)
    uchar_buf[i] ^= obfuscator++;

  for (i = 0; i < num; i += SHA_DIGEST_LENGTH) {
    numbytes = num - i;
    if (numbytes > SHA_DIGEST_LENGTH)
      numbytes = SHA_DIGEST_LENGTH;
    SHA1(uchar_buf + i, numbytes, sha_digest);
    RAND_add(sha_digest, SHA_DIGEST_LENGTH, entropy * SHA_DIGEST_LENGTH / num);
  }
}

 /*
  * rand_bytes(): wrapper around RAND_bytes(). Actually, all it does by now is
  * to stir around the "obfuscator" bits.
  */
int rand_bytes(unsigned char *buf, int num)
{
  int i, retval;

  retval = RAND_bytes(buf, num);
  for (i = 0; i < num; i++)
    obfuscator ^= buf[i];
  return retval;
}
